package io.airbridge.statistics;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.location.Location;

import org.json.JSONException;
import org.json.JSONObject;

import io.airbridge.AirBridge;
import io.airbridge.Constants;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.networking.ABRequest;
import io.airbridge.internal.networking.ABResponse;
import io.airbridge.internal.networking.Param;
import io.airbridge.internal.networking.RequestQueue;
import io.airbridge.internal.tasks.AirBridgeExecutor;
import io.airbridge.statistics.events.DeepLinkLaunchEvent;
import io.airbridge.statistics.events.Event;
import io.airbridge.statistics.page.PageTracker;

import static io.airbridge.statistics.StateContainer.State.DEVICE_INFO_FETCHED;

/**
 * Sends statistical events to the AirBridge server.
 * @author Hyojun Kim
 */
public class Tracker {

    private RequestQueue queue;
    public PageTracker pageTracker;

    // optional information - can used to track in detail;
    private Param locationInfo;
    private Param userInfo;

    // if this flag is true, the Tracker won't track any information of an app.
    // (12/05/2017) disabled by default - will be enabled default in future.
    private boolean pageTrackingEnabled = false;

    public Tracker(Context context) {
        queue = new RequestQueue(context);
        pageTracker = new PageTracker(context, this);
    }

    /**
     * If it's disabled, Airbridge SDK will not track any page view data of your app.
     */
    public void setPageTrackingEnabled(boolean pageTrackingEnabled) {
        this.pageTrackingEnabled = pageTrackingEnabled;
    }

    /**
     * @return {@code true} if page view tracking is enabled.
     * default is {@code false}.
     */
    public boolean isPageTrackingEnabled() {
        return pageTrackingEnabled;
    }

    /**
     * Sets current user (of your app)'s email info. optional.
     * If it's given, it will be used to track user more accurately.
     *
     * @param email {String} A email of the current user in your app
     */
    public void setUserEmail(String email) {
        if (userInfo == null) userInfo = new Param();
        userInfo.put("externalUserEmail", email);
    }

    /**
     * Sets current user (of your app)'s ID. optional.
     * If it's given, it will be used to track user more accurately.
     *
     * @param userId {String} An unique ID of the current user in your app
     */
    public void setUserId(String userId) {
        if (userInfo == null) userInfo = new Param();
        userInfo.put("externalUserID", userId);
    }

    /**
     * Sets location data. optional.
     * If it's given, it will be used to track user more accurately.
     *
     * @param location Location data from GPS.
     */
    public void setLocationData(Location location) {
        locationInfo = new Param()
                .put("latitude", location.getLatitude())
                .put("longitude", location.getLongitude())
                .put("altitude", location.getAltitude())
                .put("speed", location.getSpeed());
    }

    /**
     * 통계 이벤트를 보낸다.
     * @param event {@link Event}
     */
    public void send(final Event event) {
        if (!event.canBeSent()) return;

        AirBridgeExecutor.run(new Runnable() {
            @Override
            public void run() {
                // event SHOULDN'T BE sent before the device info is fetched
                StateContainer.waitFor(DEVICE_INFO_FETCHED, Constants.UUID_WAIT_TIMEOUT);

                event.onBeforeSendingEvent();

                ABRequest request = makeEventRequest(event);
                if (event.isImportant()) queue.sendNow(request);
                else queue.enqueue(request);
            }
        });
    }


    /**
     * Sends {@link DeepLinkLaunchEvent} if given intent is a deep link.
     *   This method must be called at {@link Activity#onCreate} or {@link Activity#onNewIntent}.
     *
     * @param intent Incoming Intent to activity.
     */
    public void onNewIntent(Intent intent) {
        if (intent != null && Intent.ACTION_VIEW.equals(intent.getAction()) && intent.getData() != null) {
            send(new DeepLinkLaunchEvent(intent.getDataString()));
        }
    }

    /**
     * 통계 이벤트를 큐 없이 즉시 보내고, 서버로부터 응답을 받는다.
     * @param event {@link Event}
     * @param callback {@link EventCallback} 이벤트 호출 결과에 대한 콜백
     */
    public void call(final Event event, final EventCallback callback) {
        if (!event.canBeSent()) return;

        AirBridgeExecutor.run(new Runnable() {
            @Override
            public void run() {
                // event SHOULDN'T BE sent before the device info is fetched
                StateContainer.waitFor(DEVICE_INFO_FETCHED, Constants.UUID_WAIT_TIMEOUT);

                event.onBeforeSendingEvent();

                makeEventRequest(event).callAsync(new ABRequest.Callback() {
                    @Override
                    public void done(ABRequest request, ABResponse response) {
                        if (response.isFailed()) {
                            Logger.e("Failed to send event.");
                            return;
                        }

                        try {
                            JSONObject json = new JSONObject(response.body);
                            callback.done(json.getJSONObject("resource"));

                        } catch (JSONException e) {
                            Logger.e("Failed to parse JSON.", e);

                        } catch (Throwable e) {
                            Logger.wtf("An error occurred while processing event result.", e);
                        }
                    }
                });
            }
        });
    }

    private ABRequest makeEventRequest(Event event) {
        Logger.v("Event occurred : %s (%d)", event.getClass().getSimpleName(), event.getCategory());
        DeviceInfo device = DeviceInfo.getInstance();

        // fill event request body
        Param payload = new Param()
                .put("eventTimestamp", System.currentTimeMillis())
                .put("sdkVersion", "M_A_v" + Constants.VERSION)
                .put("app", getApplicationData(device));

        Param deviceData = device.serialize();
        deviceData.put("limitAppTracking", !pageTrackingEnabled);
        if (locationInfo != null) {
            deviceData.put("location", locationInfo);
        }
        payload.put("device", deviceData);

        if (userInfo != null) {
            payload.put("user", userInfo);
        }

        // every event data must contain a current page data.
        Param eventData = event.getEventData(Tracker.this);
        eventData.put("exActiveStatus", pageTracker.getPreviousState());
        eventData.put("page", pageTracker.getCurrentPageData());
        payload.put("eventData", eventData);

        String url = Constants.STATS_ENDPOINT + AirBridge.appId + "/events/mobile-app/" + event.getCategory();
        return new ABRequest("POST", url).setParameter(payload);
    }

    private Param getApplicationData(DeviceInfo device) {
        return new Param()
                .put("version", device.appVersion)
                .put("versionCode", device.appVersionCode)
                .put("packageName", device.packageName);
    }

    public interface EventCallback {
        void done(JSONObject results) throws Exception;
    }
}
