package io.airbridge.statistics;

import android.Manifest;
import android.annotation.SuppressLint;
import android.bluetooth.BluetoothAdapter;
import android.content.ContentResolver;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.Point;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.WindowManager;

import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.UUID;

import io.airbridge.internal.log.Logger;
import io.airbridge.internal.networking.Param;

/**
 * 기기 정보를 가져온다.
 */
public class DeviceInfo {

    /**
     * Custom UUID를 저장할 키값. Custom UUID는
     * <a href="http://d.android.com/guide/topics/data/autobackup.html">Auto Backup</a>에 의해 백업되는
     * Shared Preferences에 저장되기 때문에, 앱 재설치 이후에도 고유해진다.
     */
    private static final String PREF_UNIQUE_ID = "PREF_UNIQUE_ID";

    /**
     * To get mobile cookie (Attribution data) from content provider
     */
    public static final String ATTRIBUTION_ID_COLUMN_NAME = "aid";
    private static final Uri ATTRIBUTION_ID_CONTENT_URI =
            Uri.parse("content://com.facebook.katana.provider.AttributionIdProvider");

    private static DeviceInfo instance;

    private Context context;

    /**
     * Fetched attribution data.
     */
    String facebookAttributionId;

    /**
     * LAT (Limit Ad Tracking) 플래그. 사용자가 광고 ID 사용을 Google 콘솔에서 거부했는지 여부이다.
     */
    public Boolean isLAT = false;

    /**
     * Google Advertising ID 값.
     */
    public String gaid;

    /**
     * Device UUID. Usually GAID, but it can be customly generated UUID.
     */
    public String uuid;


    /**
     * Package informations.
     */
    public String packageName, appVersion, appVersionCode;

    /**
     * Screen (Display) informations.
     */
    public int screenWidth, screenHeight, screenDensity;

    /**
     * Country informations.
     */
    public String locale, timezone;

    public DeviceInfo(Context context) {
        this.context = context;
    }

    public static DeviceInfo getInstance() {
        if (instance == null) {
            throw new IllegalStateException("You must call DeviceInfo.fetch before you use it.");
        }
        return instance;
    }

    /**
     * Fetch device informations.
     * <b>NOTE that</b> this method must not be called in UI thread.
     *
     * @param context Application context
     */
    public static void fetch(Context context) {
        instance = new DeviceInfo(context);
        instance.fetchGaid(context);
        instance.fetchAttributionId(context);
        instance.fetchScreenSize(context);

        // fetch country informations
        instance.locale = Locale.getDefault().getLanguage() + "-" + Locale.getDefault().getCountry();
        instance.timezone = TimeZone.getDefault().getID();

        try {
            // fetch package info
            PackageInfo info = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            instance.appVersion = info.versionName;
            instance.appVersionCode = String.valueOf(info.versionCode);
            instance.packageName = info.packageName;

        } catch (Exception e) {
            instance.appVersion = "(N/A)";
            instance.appVersionCode = "(N/A)";
            instance.packageName = context.getPackageName();
        }

        // use GAID as UUID.
        // if GAID is not available, generate a custom UUID.
        if (instance.gaid != null) instance.uuid = instance.gaid;
        else {
            try {
                Logger.d("Using UUID instead...");
                instance.uuid = generateTelephonyUuid(context);

            } catch (Exception ex) {
                Logger.d("Using random UUID instead...");
                instance.uuid = generateRandomUuid(context);
            }
        }
    }

    /**
     * GAID 및 LAT 값을 가져온다.
     * NOTE: 동기식 함수이므로 Main Thread에서 실행되서는 안된다.
     */
    private void fetchGaid(Context context) {
        try {
            // reflection을 통해 받아옴.
            Class<?> adidClass = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient");
            Method getAdvertisingIdInfoMethod = adidClass.getMethod("getAdvertisingIdInfo", Context.class);
            Object adidInfoObject = getAdvertisingIdInfoMethod.invoke(null, context);
            Method isLimitAdTrackingEnabledMethod = adidInfoObject.getClass().getMethod("isLimitAdTrackingEnabled");
            Method getAdIdMethod = adidInfoObject.getClass().getMethod("getId");

            gaid = getAdIdMethod.invoke(adidInfoObject).toString();
            isLAT = !(boolean) isLimitAdTrackingEnabledMethod.invoke(adidInfoObject);

        } catch (NoClassDefFoundError | NoSuchMethodError | ClassNotFoundException ignored) {
            Logger.e("Google Play Services SDK not found. We recommend to add Google Play Service " +
                    "to your project.");

        } catch (NullPointerException ignored) {
            // FIXME: 간헐적으로 GAID Fetching이 실패한다. getAdvertisingIdInfo가 null을 반환
            Log.w("AirBridge", "GAID fetching failed. Trying to use custom UUID...");

        } catch (Exception e) {
            Logger.e("Error occurred while fetching GAID", e);
        }
        Logger.v("GAID Set.");
    }

    /**
     * Facebook Mobile Cookie로부터 Facebook 광고 Attribution ID를 가져온다.
     * @param context Application context
     */
    private void fetchAttributionId(Context context) {
        try {
            String[] projection = {ATTRIBUTION_ID_COLUMN_NAME};
            ContentResolver contentResolver = context.getContentResolver();
            Cursor c = contentResolver.query(ATTRIBUTION_ID_CONTENT_URI, projection, null, null, null);
            if (c == null || !c.moveToFirst()) {
                return;
            }
            facebookAttributionId = c.getString(c.getColumnIndex(ATTRIBUTION_ID_COLUMN_NAME));
            if (facebookAttributionId == null) facebookAttributionId = "";
            c.close();

        } catch (Throwable e) {
            Logger.d("Unable to retrieve Facebook Attribution ID.");
        }
    }

    private void fetchScreenSize(Context context) {
        WindowManager w = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display d = w.getDefaultDisplay();
        DisplayMetrics metrics = new DisplayMetrics();
        d.getMetrics(metrics);

        screenDensity = metrics.densityDpi;

        if (Build.VERSION.SDK_INT >= 17) {
            // includes window decorations (statusbar bar/menu bar) - using official API
            Point realSize = new Point();
            d.getRealSize(realSize);
            screenWidth = realSize.x;
            screenHeight = realSize.y;
            return;

        } else if (Build.VERSION.SDK_INT >= 14) {
            // includes window decorations (statusbar bar/menu bar) - using UNDOCUMENTED API
            try {
                screenWidth = (Integer) Display.class.getMethod("getRawWidth").invoke(d);
                screenHeight = (Integer) Display.class.getMethod("getRawHeight").invoke(d);
                return;

            } catch (Exception ignored) {}
        }

        // since SDK_INT = 1; not including window decorations
        screenWidth = metrics.widthPixels;
        screenHeight = metrics.heightPixels;
    }

    /**
     * Serialize device info according to the Stats API v2 format.
     * @return A serialized device info as {@link Param}
     */
    public Param serialize() {
        Param deviceData = new Param()
                .put("deviceModel", Build.MODEL)
                .put("manufacturer", Build.MANUFACTURER)
                .put("osName", "Android")
                .put("osVersion", Build.VERSION.RELEASE)
                .put("locale", locale)
                .put("timezone", timezone)
                .put("deviceIP", getIPAddress())
                .put("deviceUUID", uuid)
                .put("gaid", gaid)
                .put("limitAdTracking", isLAT)
                .put("facebookAttributionID", facebookAttributionId);

        Param screenData = new Param()
                .put("width", screenWidth)
                .put("height", screenHeight)
                .put("density", screenDensity);
        deviceData.put("screen", screenData);

        deviceData.put("network", getNetworkInfo());
        return deviceData;
    }

    /**
     * Get IP address from first non-localhost interface
     * @return address or empty string
     */
    String getIPAddress() {
        try {
            List<NetworkInterface> interfaces = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                List<InetAddress> addrs = Collections.list(intf.getInetAddresses());
                for (InetAddress addr : addrs) {
                    if (!addr.isLoopbackAddress()) {
                        String sAddr = addr.getHostAddress();
                        boolean isIPv4 = sAddr.indexOf(':') < 0;
                        if (isIPv4) return sAddr;
                    }
                }
            }
        } catch (Exception ignored) { }
        return "";
    }

    @SuppressWarnings("MissingPermission")
    Param getNetworkInfo() {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

        boolean wifiOn = false, bluetoothOn = false, cellularOn = false;
        String carrier = null;
        try {
            // wifi and mobile data.
            NetworkInfo activeInfo = connectivityManager.getActiveNetworkInfo();
            if (activeInfo != null) {
                if (activeInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                    wifiOn = true;
                } else if (activeInfo.getType() == ConnectivityManager.TYPE_BLUETOOTH) {
                    // in case of using bluetooth tethering.
                    bluetoothOn = true;
                } else {
                    cellularOn = true;
                }
            }

            // check bluetooth if permission is available
            if (!bluetoothOn && hasPermission(Manifest.permission.BLUETOOTH)) {
                BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
                bluetoothOn = bluetoothAdapter.isEnabled();
            }

            // check carrier name.
            if (hasPermission(Manifest.permission.READ_PHONE_STATE)) {
                TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
                carrier = telephonyManager.getNetworkOperatorName();
            }

        } catch (Exception e) {
            Logger.e("Failed to fetch network info", e);
        }

        ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE)).getNetworkOperatorName();

        return new Param()
                .put("wifi", wifiOn)
                .put("bluetooth", bluetoothOn)
                .put("cellular", cellularOn)
                .put("carrier", carrier);
    }

    private boolean hasPermission(String permission) {
        return (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && context.checkSelfPermission(permission) == PackageManager.PERMISSION_GRANTED) ||
                context.getPackageManager().checkPermission(permission, context.getPackageName()) == PackageManager.PERMISSION_GRANTED;
    }

    private static String generateRandomUuid(Context context) {
        SharedPreferences prefs = context.getSharedPreferences(PREF_UNIQUE_ID, Context.MODE_PRIVATE);
        String uniqueID = prefs.getString(PREF_UNIQUE_ID, null);
        if (uniqueID == null) {
            uniqueID = UUID.randomUUID().toString();
            SharedPreferences.Editor editor = prefs.edit();
            editor.putString(PREF_UNIQUE_ID, uniqueID);
            editor.apply();
        }

        return uniqueID;
    }

    @SuppressLint("HardwareIds")
    private static String generateTelephonyUuid(Context context) throws Exception {
        TelephonyManager tm =
                (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);

        String tmDevice, tmSerial, androidId;
        tmDevice = "" + tm.getDeviceId();
        tmSerial = "" + tm.getSimSerialNumber();
        androidId = "" + Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);

        UUID deviceUuid = new UUID(androidId.hashCode(),
                ((long)tmDevice.hashCode() << 32) | tmSerial.hashCode());
        return deviceUuid.toString();
    }
}
