package io.airbridge.internal.log;

import android.util.Log;

import java.util.MissingFormatArgumentException;

import io.airbridge.AirBridge;
import io.airbridge.Constants;

/**
 * AirBridge Logger
 * Copyright (C) 2015 AB180. All rights are reserved.
 */
public class Logger {

    private static CrashReporter crashReporter = new CrashReporter();

    public static void v(String format, Object ...args) {
        String message;
        try {
            message = String.format(format, args);

        } catch (MissingFormatArgumentException e) {
            message = format;
        }

        crashReporter.record(new LogEntry("verbose", message));

        if (!AirBridge.isDebugMode) return;
        Log.v(Constants.LOG_TAG, message);
    }

    public static void d(String format, Object ...args) {
        String message;
        try {
            message = String.format(format, args);

        } catch (MissingFormatArgumentException e) {
            message = format;
        }

        crashReporter.record(new LogEntry("debug", message));

        if (!AirBridge.isAppDebuggable) return;
        Log.d(Constants.LOG_TAG, message);
    }

    public static void w(String message, Throwable error) {
        crashReporter.record(new LogEntry("warning", message)
                .setErrorInfo(error));

        Log.w(Constants.LOG_TAG, "Warning: " + message, error);
    }

    public static void e(String message, Throwable error) {
        crashReporter.record(new LogEntry("error", message)
                .setErrorInfo(error));

        Log.e(Constants.LOG_TAG, message, error);
    }

    public static void e(String message) {
        crashReporter.record(new LogEntry("error", message));
        Log.e(Constants.LOG_TAG, message);
    }

    /**
     * 이슈가 될 수 있는 중대한 에러를 로깅하고, 서버에 리포트한다.
     * 중대한 에러가 아니라면 {@link Logger#e} 혹은 {@link Logger#w}의 사용을 권고한다.
     * @param message 에러 메세지
     * @param error 에러
     */
    public static void wtf(String message, Throwable error) {
        crashReporter.record(new LogEntry("critical", message).setErrorInfo(error));
        Log.e(Constants.LOG_TAG, message, error);

        crashReporter.reportToServer();
    }
}
