package io.airbridge.installation;

import android.content.Context;
import android.content.Intent;

import org.json.JSONObject;

import io.airbridge.AirBridge;
import io.airbridge.Constants;
import io.airbridge.Session;
import io.airbridge.deeplink.DeepLink;
import io.airbridge.integration.IntegrationManager;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.tasks.AirBridgeExecutor;
import io.airbridge.internal.tasks.CompletableFuture;
import io.airbridge.statistics.Attribution;
import io.airbridge.statistics.StateContainer;
import io.airbridge.statistics.Tracker;
import io.airbridge.statistics.events.Event;
import io.airbridge.statistics.events.InstallEvent;

import static io.airbridge.statistics.StateContainer.State.INIT_FINISHED;
import static io.airbridge.statistics.StateContainer.State.INSTALL_WITH_DEEP_LINK;

/**
 * 설치 인증 작업을 수행한다.
 * 최초 설치시 단 한번만 실행되며, Deferred Deep Link나 심플링크를 이 과정중에 가져온다.
 *
 * @author Hyojun Kim
 */
public class Installer {

    /**
     * DeferredLinkReceiver로부터 Referrer를 받기 위함.
     */
    public static CompletableFuture<String> referrerFuture = new CompletableFuture<>();

    /**
     * 5초 지나서 Deferred Link가 실행되는 건 UX적으로 부자연스러워 보이므로,
     * 그 경우엔 세션까진 남기되 핸들러 실행만 취소한다.
     */
    private static boolean tooLongToDeferredAction;

    /**
     * Google Play로부터 referrer 인텐트가 늦게 들어올 수 있으므로, 타임아웃만큼 대기한다.
     */
    private static int referrerTimeout = Constants.DEFAULT_REFERRER_TIMEOUT;

    /**
     * Google Play로부터 레퍼러를 얼마나 기다릴지 설정한다. (기본 1500ms)
     * @param millis Milliseconds
     */
    public static void setPlayStoreReferrerTimeout(int millis) {
        referrerTimeout = millis;
    }

    public static void install(final Context context, boolean doReferrerTimeout) {
        // 5초(실제로 4.7초)가 지난 후 deferred deeplink가 들어오면,
        // 설치인증만 한 후 deferred deeplink 는 작동 안하게 한다.
        // 왜냐면 딥링크에 대한 반응이 너무 늦을 경우 부자연스러운 동작이 됨.
        if (doReferrerTimeout) {
            AirBridgeExecutor.runAfterTime(5, new Runnable() {
                @Override public void run() {
                    tooLongToDeferredAction = true;
                }
            });
        }
        Session.getCurrent().setInstalled();

        if (StateContainer.get() == INSTALL_WITH_DEEP_LINK) {
            Logger.d("Skipping InstallEvent because the app is installed with deep link.");
            return;
        }

        // 설치 이벤트를 Integrator에 보내서 서드파티 플러그인이 별도의 설치 작업을 할 수 있게 함.
        IntegrationManager.notifyInstalled(context);

        // Google Play로부터 referrer 인텐트가 늦게 들어올 수 있으므로, 기다린다.
        String referrer = referrerFuture.await(referrerTimeout);

        Event installEvent = new InstallEvent(referrer);
        AirBridge.getTracker().call(installEvent, new Tracker.EventCallback() {
            @Override
            public void done(JSONObject results) throws Exception {
                finishInstall(context, results);
            }
        });
    }

    /**
     * 설치 과정을 마무리하고, 서버에서 온 결과를 바탕으로 심플링크 / 딥링크 핸들러를 호출한다.
     * @param context {@link Context}
     * @param serverResult Install Event를 보냈을때 서버에서 준 응답.
     * @throws Exception
     */
    private static void finishInstall(Context context, JSONObject serverResult) throws Exception {
        if (serverResult.has("lastTouchpoint")) {
            JSONObject deferredData = serverResult.getJSONObject("lastTouchpoint");
            if (tooLongToDeferredAction) {
                Logger.d("Ignoring calling handler due to the late response from server.");

            } else if (deferredData != null) {
                String simpleLink = deferredData.getString("simplelink");
                SimpleLink.callHandler(new SimpleLink(simpleLink), context);

                if (deferredData.has("deeplink")) {
                    String deeplink = deferredData.getString("deeplink");
                    Intent intent = DeepLink.createIntent(deeplink);
                    context.startActivity(intent);
                }
            }
            Attribution.getInstance().update(context);
        }
        StateContainer.set(INIT_FINISHED);
    }
}
