package io.airbridge.deeplink;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;

import java.util.ArrayList;
import java.util.List;

import io.airbridge.internal.log.Logger;

/**
 * Router
 * Copyright (C) 2015 AB180. All rights are reserved.
 */
public class Router {

    List<Layer> layerStack = new ArrayList<>();
    List<DeepLink.Filter> filters = new ArrayList<>();
    List<DeepLink.Handler> handlers = new ArrayList<>();
    Object defaultHandler;

    Router() {}

    /**
     * Registers an route.
     * You can do this by calling this method directly
     * or putting the {@link DeepLinkRoute} annotation to the activity.
     *
     * @param url Route URL that activity will handle.
     *            You can contain route parameter. For example, if you set the  (ex: "/user/:id") -
     * @param handler {@link Activity} or {@link DeepLink.Handler} launched when the route is matched.
     */
    public void addRoute(String url, Object handler) {
        layerStack.add(new Layer(url, handler));
    }

    /**
     * Registers filter.
     * If given {@link DeepLink.Filter} returns {@code true}, the deep link will not be launched.
     * @param filter Deep link filter
     */
    public void filter(DeepLink.Filter filter) {
        filters.add(filter);
    }

    /**
     * Registers handler that will be must called when deep link comes.
     * NOTE THAT handler will be called first before the route is called.
     * @param handler Deep link handler.
     */
    public void addHandler(DeepLink.Handler handler) {
        handlers.add(handler);
    }

    /**
     * @param defaultHandler
     */
    public void setDefaultHandler(Object defaultHandler) {
        this.defaultHandler = defaultHandler;
    }

    /**
     * Find an activity class match with given URI.
     * @return handler (null if nothing matches)
     */
    Object findMatches(DeepLink link) {
        for (Layer layer : layerStack) {
            Object handler = layer.match(link);
            if (handler != null) return handler;
        }
        return null;
    }

    /**
     * Handles a deep link when it's come.
     * @param link {@link DeepLink} incoming deep link.
     * @param context {@link Context}
     * @return true on matched something
     */
    public boolean callHandlers(DeepLink link, Context context) {
        boolean matched = true;

        // filter first.
        for (DeepLink.Filter filter : filters) {
            if (filter.filter(context, link)) return false;
        }

        // call handler
        for (DeepLink.Handler handler : handlers) {
            handler.onLink(context, link);
        }

        // match route.
        Object handler = findMatches(link);
        if (handler == null) {
            Logger.d("No match route found with URL : " + link.getUri());
            if (defaultHandler != null) {
                handler = defaultHandler;
            }
            matched = false;
        }

        // Activity or BroadcastReceiver
        if (handler instanceof Class) {
            Intent intent = new Intent(context, (Class) handler);
            intent.setAction(Intent.ACTION_VIEW);
            intent.addCategory(Intent.CATEGORY_BROWSABLE);
            intent.putExtras(link.toBundle());
            intent.putExtra("airbridge", true);
            intent.setData(link.getUri());

            if (!(context instanceof Activity)) intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(intent);

        } else if (handler instanceof DeepLink.Handler) {
            // run handler
            ((DeepLink.Handler) handler).onLink(context, link);
        }
        return matched;
    }

    /**
     * An route with path and an handler that will handle Deep Link.
     */
    static class Layer {
        public List<String> pathSegments;
        public Object handler;

        public Layer(String path, Object handler) {
            // non-strict path matching
            if (path.endsWith("/")) path = path.substring(0, path.length()-1);
            if (!path.startsWith("/")) path = "/" + path;

            this.pathSegments = Uri.parse(path.toLowerCase()).getPathSegments();
            this.handler = handler;
        }

        /**
         * Check that given URL is matched with this path.
         * Also, fills path parameter from given URL if the route matches.
         *
         * @param link {@link DeepLink} incoming deep link.
         * @return {@link Activity} or {@link DeepLink.Handler} if url matches with path, else return null
         */
        public Object match(DeepLink link) {
            List<String> splitUrl = link.getUri().getPathSegments();
            if (splitUrl.size() != pathSegments.size()) return null;

            Bundle pathParams = new Bundle();

            int i = 0;
            for (String pathSegment : pathSegments) {
                String givenPath = splitUrl.get(i++).toLowerCase();

                // check and fill parameter path
                if (pathSegment.startsWith("{{") && pathSegment.endsWith("}}")) {
                    String paramKey = pathSegment.substring(2, pathSegment.length()-2);
                    pathParams.putString(paramKey, givenPath);
                    continue;
                }

                // *(asterisk) means match all path after the asterisk
                if (pathSegment.equals("*")) break;

                // not matched
                if (!pathSegment.equals(givenPath)) return null;
            }

            // fill path parameter only when it matches.
            link.toBundle().putAll(pathParams);

            return handler;
        }
    }
}
