package io.airbridge.deeplink;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.net.Uri;

import io.airbridge.DeepLinkActivity;
import io.airbridge.Link;
import io.airbridge.internal.log.Logger;

/**
 * Airbridge에서의 딥 링크 사용을 도와주는 클래스이다.
 * @author Hyojun Kim
 */
public class DeepLink extends Link {

    static Router router;

    public static void init(Context context) {
        if (hasRouterActivityRegistered(context)) {
            Logger.d("Using Airbridge DeepLink Router.");

            router = new Router();
            AnnotationScanner.scanAndRegister(context, router);
        }
    }

    /**
     * @return true if {@link DeepLinkActivity} is registered in AndroidManifest.xml
     */
    private static boolean hasRouterActivityRegistered(Context context) {
        String routerActivityName = DeepLinkActivity.class.getName();

        try {
            ActivityInfo activities[] = context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), PackageManager.GET_ACTIVITIES)
                    .activities;

            for (ActivityInfo activity : activities) {
                if (activity.name.equals(routerActivityName)) return true;
            }
            return false;

        } catch (Throwable e) {
            return false;
        }
    }

    /**
     * 딥 링크 라우터를 반환한다.
     * @return {@link Router}
     */
    public static Router getRouter() {
        return router;
    }

    /**
     * URI를 통해 딥 링크 Intent를 생성한다.
     * @param deepLinkUri Deep Link URI
     * @return {@link Intent}
     */
    public static Intent createIntent(String deepLinkUri) {
        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.addCategory(Intent.CATEGORY_BROWSABLE);
        intent.setData(Uri.parse(deepLinkUri));
        return intent;
    }

    /**
     * 주어진 {@link Activity}가 Airbridge 딥 링크로 실행되었는지 여부를 확인한다.
     * @param activity 액티비티
     * @return 해당 액티비티가 딥 링크로 실행되었는지 여부
     */
    public static boolean hadOpened(Activity activity) {
        return activity.getIntent() != null && Intent.ACTION_VIEW.equals(activity.getIntent().getAction());
    }

    /**
     * 주어진 {@link Activity}를 실행시킨 딥 링크의 파라미터를 가져온다.
     * @param activity 액티비티
     * @return {@link DeepLink} 객체. 딥 링크로 열린게 아닐 시엔 {@code null}을 반환.
     */
    public static DeepLink fromActivity(Activity activity) {
        if (!DeepLink.hadOpened(activity)) return null;
        return new DeepLink(activity.getIntent().getData().toString());
    }

    public DeepLink(String uriString) {
        super(uriString);
    }

    /**
     * An handler interface that handles deep link.
     */
    public interface Handler {
        void onLink(Context context, DeepLink deepLink);
    }

    /**
     * An filter interface that filters deep link.
     */
    public interface Filter {
        boolean filter(Context context, DeepLink deepLink);
    }

}
