package io.airbridge;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.Map;
import java.util.Properties;

import io.airbridge.internal.log.Logger;

/**
 * AirBridge의 트랜젝션, 세션 정보를 관리한다.
 * @author Hyojun Kim
 */
public class Session {

    private static final String KEY_INSTALLED = "installed";

    /**
     * 세션 객체는 Singleton으로 단 하나만 생성한다.
     */
    static Session instance;

    static boolean disableOffline = false;

    private Properties props;
    private File preferenceFile;

    /**
     * 세션을 로딩하고 초기화한다.
     * @param context Application Context
     */
    static void init(Context context) {
        instance = new Session(context);
    }

    /**
     * 현재 세션을 가져온다.
     * @return 현재 세션
     */
    public static Session getCurrent() {
        if (instance == null) {
            throw new IllegalStateException("You must initialize AirBridge SDK first.");
        }
        return instance;
    }

    /**
     * @param context Application Context
     */
    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private Session(Context context) {
        // SharedPreference를 쓰면 세션 정보가 자동 백업되는 문제가 있어서 별도의 방법을 사용한다.
        File preferenceDir = Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
                ? context.getNoBackupFilesDir()
                : context.getFilesDir();

        preferenceFile = new File(preferenceDir, Constants.PREFS);
        props = new Properties();
        load(context);
    }

    /**
     * 로컬에서 세션 정보를 로딩한다.
     */
    public void load(Context context) {
        if (preferenceFile.exists() && !disableOffline) {
            try {
                props.load(new FileInputStream(preferenceFile));

            } catch (Exception e) {
                Logger.e("Failed to load existing session.", e);
            }
        } else {
            SharedPreferences prefs = context.getSharedPreferences(Constants.PREFS, Context.MODE_APPEND);
            if (prefs.contains("initialTransactionId")) {
                Logger.d("Old session file found. Migrating...");
                Map<String, ?> prefData = prefs.getAll();
                for (String key : prefData.keySet()) {
                    put(key, prefData.get(key));
                }

                // 기존껄 삭제해야 Auto Backup에 동기화되지 않음.
                prefs.edit().clear().apply();
            }
        }

        // Migration : installed 플래그는 v0.10.11에서 추가되었음. 기존에는 InitialTransactionID 사용.
        String initialTransactionId = getString("initialTransactionId", null);
        boolean hasInitialTransaction = initialTransactionId != null && !"".equals(initialTransactionId);
        if (isFirstTime() && hasInitialTransaction) {
            Logger.d("Initial transaction migration.");
            put(KEY_INSTALLED, true);
        }
    }

    /**
     * 세션 정보를 변경하고, 오프라인에 저장한다.
     * NOTE: 값이 null이거나 똑같으면 저장하지 않는다.
     *
     * @param key Key {String}
     * @param value Value
     */
    public void put(String key, Object value) {
        if (value == null || value.equals(props.get(key))) return;
        props.put(key, value.toString());
        save();
    }

    /**
     * 세션 정보를 저장한다.
     */
    public synchronized void save() {
        if (disableOffline) return;

        try {
            Logger.d("Saving session data...");
            props.store(new FileOutputStream(preferenceFile), null);

        } catch (Throwable e) {
            Logger.e("Failed to save session data.", e);
        }
    }

    /**
     * 인스톨 플래그를 설정한다. 최초 설치 작업 완료 후 호출된다.
     */
    public void setInstalled() {
        put(KEY_INSTALLED, true);
    }

    /**
     * {@link #isFirstTime}과 같지만 좀더 Fluent한 코드를 만들기 위해 메서드 또만듬.
     * @return 심플링크 / 딥링크 세션 (or Transaction) 여부
     */
    public boolean hasActiveSession() {
        return !isFirstTime();
    }

    /**
     * @return 앱이 설치되고 나서 처음 실행되었는지 여부
     */
    public boolean isFirstTime() {
        return !getBoolean(KEY_INSTALLED, false);
    }

    /**
     * 세션 정보를 삭제한다.
     * NOTE: 디버그 목적 이외엔 호출을 권장하지 않음.
     */
    public void delete() {
        props.clear();

        try {
            // delete file immediately
            preferenceFile.delete();

        } catch (Exception e) {
            Logger.e("Failed to remove session data", e);
        }

        put(KEY_INSTALLED, false);
    }

    /**
     * String 정보를 읽어온다.
     * @param key 키값.
     * @param defaults 기본값.
     * @return 읽어온 값 (없을 시 defaults)
     */
    public String getString(String key, String defaults) {
        return props.getProperty(key, defaults);
    }

    /**
     * int형 정보를 읽어온다.
     * @param key 키값.
     * @param defaults 기본값.
     * @return 읽어온 값 (없을 시 defaults)
     */
    public int getInt(String key, int defaults) {
        return Integer.valueOf(props.getProperty(key, String.valueOf(defaults)));
    }

    /**
     * long형 정보를 읽어온다.
     * @param key 키값.
     * @param defaults 기본값.
     * @return 읽어온 값 (없을 시 defaults)
     */
    public long getLong(String key, long defaults) {
        return Long.valueOf(props.getProperty(key, String.valueOf(defaults)));
    }

    /**
     * Boolean형의 정보를 읽어온다.
     * @param key 키값.
     * @param defaults 기본값.
     * @return 읽어온 값 (없을 시 defaults)
     */
    public boolean getBoolean(String key, boolean defaults) {
        return Boolean.valueOf(props.getProperty(key, String.valueOf(defaults)));
    }
}
