package io.airbridge.internal.networking;

import android.os.Handler;
import android.os.Looper;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.ConnectException;
import java.net.UnknownHostException;

import io.airbridge.Constants;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.tasks.AirBridgeExecutor;

/**
 * HTTP 비동기적 호출을 하기 위해서 사용된다.
 * Copyright (C) 2015 AB180. All rights are reserved.
 */
public class ABRequest {
    private static ABHttpClient httpClient = new ABHttpClient();

    protected String url, method;
    protected JSONObject param;

    /**
     * Retry를 사용할 지 여부. 당장 응답을 받을 필요가 없는 통계 이벤트에 한정됨.
     */
    boolean shouldBackoff = false;

    /**
     * 다음번 재시도 딜레이.
     */
    int retryDelay = 0;

    /**
     * 메인 서버에 에러가 생길 시 서브 서버로 재요청할지 결정.
     */
    private boolean retryingToSecondaryServer = false;

    /**
     * Callback을 실행할 핸들러. UI 스레드에서 호출됨.
     */
    private static Handler handler;

    static void setHttpClient(ABHttpClient client) {
        httpClient = client;
    }

    public static ABRequest fromJson(String str) throws JSONException {
        JSONObject data = new JSONObject(str);
        ABRequest request = new ABRequest(
                data.getString("requestMethod"),
                data.getString("requestEndpoint"));
        request.param = data.getJSONObject("requestBody");
        request.retryDelay = data.getInt("retryDelay");
        request.shouldBackoff = data.getBoolean("shouldBackoff");
        return request;
    }

    /*
     * Author : Donutt
     * init Date : 16.01.06
     * Last Update : 16.01.06
     * Desc : ABRequest Overloading initializer
     */
    public ABRequest(String method, String url) {
        this.method = method;
        this.url = url;
        this.param = new JSONObject();
    }

    /**
     * API 콜을 호출한다.
     */
    public void callAsync(final Callback callback) {
        final ABRequest request = this;

        AirBridgeExecutor.runSerialTask(new Runnable() {
            @Override
            public void run() {
                try {
                    request.param.put("requestTimestamp", System.currentTimeMillis());
                    Logger.v("Request    => %s %s", request.getUrl(), request.param);

                    final ABResponse response = httpClient.call(request);
                    Logger.v("└ Response => HTTP %d, %s", response.status, response.body);
                    if (response.isFailed()) {

                        if (response.status == 401)
                            throw new Exception("Unauthorized Error. Please reissue the app token in Dashboard.");

                        else if (response.status == 404 || response.status >= 500) {
                            if (!retryingToSecondaryServer) {
                                if (shouldBackoff) {
                                    // 그냥 예외처리. Backoff 정책이 알아서 처리하게 냅두자.
                                    throw new Exception("Internal server error. Retrying in minutes...");
                                }
                                // Backoff가 불가능하므로, 세컨더리 서버로 바로 한번 더 요청을 시도한다.
                                Logger.d("Internal server error occured. Trying again to SQS...");
                                retryingToSecondaryServer = true;
                                callAsync(callback);
                                return;

                            } else {
                                // 세컨 서버로의 요청도 실패했으면 그냥 예외처리.
                                retryingToSecondaryServer = false;
                                throw new Exception("Internal server error.");
                            }
                        }
                    } else if (retryingToSecondaryServer) {
                        // 세컨더리 서버 요청이 성공했을 경우.
                        retryingToSecondaryServer = false;
                    }

                    // ignore empty response
                    if (response.status == 204) return;

                    runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callback.done(request, response);
                        }
                    });

                } catch (UnknownHostException | ConnectException e) {
                    Logger.e("Request failed - Internet not connected.");
                    if (retryingToSecondaryServer) retryingToSecondaryServer = false;
                    runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            // ERROR CODE = 1000.
                            callback.done(request, new ABResponse(1000));
                        }
                    });

                } catch (Exception e) {
                    Logger.e("Request failed - " + request.param, e);
                    runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            // 세컨더리 서버 재요청땐 콜백을 부르지 않는다.
                            if (retryingToSecondaryServer) return;

                            // ERROR CODE = 1000.
                            callback.done(request, new ABResponse(1000));
                        }
                    });
                }
            }
        });
    }

    /**
     * UI Thread에서 작업을 실행한다. {@link ABRequest.Callback} 호출시에 사용된다.
     * @param action 실행할 작업
     */
    private void runOnUiThread(Runnable action) {
        if (handler == null) handler = new Handler(Looper.getMainLooper());
        handler.post(action);
    }

    /**
     * Request Body에 실어보낼 파라미터를 지정한다.
     * @param param Param
     */
    public ABRequest setParameter(Param param) {
        this.param = param.toJson();
        return this;
    }

    /**
     * @return HTTP method
     */
    public String getMethod() {
        return method;
    }

    /**
     * @return URL
     */
    public final String getUrl() {
        return isRetryingToSecondaryServer()
                ? Constants.SECONDARY_HOST
                : Constants.getHost() + url;
    }

    public boolean isRetryingToSecondaryServer() {
        return retryingToSecondaryServer;
    }

    /**
     * @return parameters (JSON)
     */
    public String getParam() {
        if (isRetryingToSecondaryServer()) {
            // Secondary Server인 Amazon SQS에는 특수한 형식으로 보내줘야 한다.
            try {
                JSONObject sqsRequest = this.toJson();
                sqsRequest.put("requestEndpoint",
                        Constants.getHost() + sqsRequest.getString("requestEndpoint"));
                return sqsRequest.toString();

            } catch (JSONException ignored) {}
        }
        return param.toString();
    }

    public JSONObject toJson() {
        try {
            JSONObject self = new JSONObject();
            self.put("requestMethod", method);
            self.put("requestEndpoint", url);
            self.put("requestBody", param);
            self.put("retryDelay", retryDelay);
            self.put("shouldBackoff", shouldBackoff);
            return self;

        } catch (Exception ignored) {}
        return new JSONObject();
    }

    @Override
    public String toString() {
        return param.toString();
    }

    public interface Callback {
        void done(ABRequest request, ABResponse response);
    }
}