package io.airbridge.internal.tasks;

import android.annotation.SuppressLint;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import io.airbridge.internal.log.Logger;

/**
 * Airbridge SDK 내에서 사용되는 스레드 관리를 위함.
 * @author Hyojun Kim
 */
@SuppressLint("NewApi")
public class AirBridgeExecutor {

    /**
     * 몇개까지 백그라운드 태스크를 병렬적으로 실행할 것인가?
     */
    static final int MAX_BACKGROUND_THREADS = 5;

    /**
     * 타이머 스레드풀의 개수. 네트워킹시 Exponential Backoff 처리를 감안해 사이즈를 10으로 잡음.
     */
    static final int TIMER_POOL_SIZE = 10;

    static volatile ExecutorService PARALLEL_EXECUTOR;
    static volatile ScheduledExecutorService TIMER_EXECUTOR;
    static volatile ExecutorService SERIAL_EXECUTOR;

    /**
     * 백그라운드에서 태스크를 실행한다. 한번에 복수의 태스크 실행이 가능하다.
     * @param runnable 실행할 Runnable
     */
    public static void run(final Runnable runnable) {
        if (PARALLEL_EXECUTOR == null) {
            PARALLEL_EXECUTOR = Executors.newFixedThreadPool(MAX_BACKGROUND_THREADS);
        }
        PARALLEL_EXECUTOR.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    runnable.run();

                } catch (Throwable e) {
                    Logger.wtf("Error occurred while running background task", e);
                }
            }
        });
    }

    /**
     * 특정 시간 후에 작업을 실행한다. 별도의 TIMER_EXECUTOR 스레드풀에 스케줄링된다.
     * @param seconds 딜레이할 시간
     * @param runnable 실행할 작업
     */
    public static void runAfterTime(int seconds, Runnable runnable) {
        if (TIMER_EXECUTOR == null) {
            TIMER_EXECUTOR = Executors.newScheduledThreadPool(TIMER_POOL_SIZE);
        }
        TIMER_EXECUTOR.schedule(runnable, seconds, TimeUnit.SECONDS);
    }

    /**
     * 순차적으로 태스크를 실행한다. 한번에 한 태스크밖에 실행할 수 없다.
     * @param runnable 실행할 Runnable
     */
    public static void runSerialTask(final Runnable runnable) {
        if (SERIAL_EXECUTOR == null) {
            SERIAL_EXECUTOR = Executors.newSingleThreadExecutor();
        }
        SERIAL_EXECUTOR.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    runnable.run();

                } catch (Throwable e) {
                    Logger.wtf("Error occurred while running network task", e);
                }
            }
        });
    }
}
