package io.airbridge;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.os.SystemClock;

import java.io.ByteArrayInputStream;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

import io.airbridge.deeplink.DeepLink;
import io.airbridge.installation.Installer;
import io.airbridge.integration.IntegrationManager;
import io.airbridge.integration.Integrator;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.networking.RequestQueue;
import io.airbridge.internal.tasks.AirBridgeExecutor;
import io.airbridge.statistics.Attribution;
import io.airbridge.statistics.DeviceInfo;
import io.airbridge.statistics.StateContainer;
import io.airbridge.statistics.Tracker;
import io.airbridge.statistics.events.DeepLinkLaunchEvent;
import io.airbridge.statistics.events.Event;
import io.airbridge.statistics.events.GoalEvent;
import io.airbridge.statistics.events.LaunchEvent;
import io.airbridge.statistics.goals.SignInGoalBuilder;
import io.airbridge.statistics.goals.SignUpGoalBuilder;
import io.airbridge.statistics.page.PageTracker;

import static io.airbridge.statistics.StateContainer.State.DEVICE_INFO_FETCHED;
import static io.airbridge.statistics.StateContainer.State.INIT_FINISHED;
import static io.airbridge.statistics.StateContainer.State.INSTALL_WITH_DEEP_LINK;

/**
 * AirBridge SDK 클래스.
 * Copyright (C) 2015 AB180. All rights are reserved.
 */
public class AirBridge {
    private static final String NUMBER_FILTER = "^-?\\d+$";

    public static String appId, appToken;

    public static boolean isDebugMode, isAppDebuggable;
    private static boolean doInstallCheck = true, doReferrerTimeout = true;

    private static Session session;
    private static Tracker tracker;

    /**
     * AirBridge SDK를 초기화하고, Activity 정보를 등록한다.
     * Application#onCreate 메서드에서 반드시 한 번 호출되어야 한다.
     *
     * @param context Application Context
     * @param appName Application Name (Airbridge 대시보드에서 "앱 영문명" 확인)
     * @param userToken User Token (Airbridge 대시보드에서 발급)
     */
    public static void init(Context context, String appName, String userToken) {
        if (appName == null) throw new IllegalArgumentException("App Name was not given!");
        if (userToken == null) throw new IllegalArgumentException("User Token was not given!");

        if (appName.matches(NUMBER_FILTER)) {
            throw new IllegalArgumentException(
                    "App name must be a sub-domain name of your Airbridge App; Seems like you haven't been migrated from old version.\n" +
                            "Please migrate according to http://docs.airbridge.io/ko/start/1-3-1.html");
        }

        context = context.getApplicationContext();
        AirBridge.appId = appName;
        AirBridge.appToken = userToken;
        fetchAppDebuggable(context);

        try {
            Session.init(context);
            session = Session.getCurrent();
            tracker = new Tracker(context);
            DeepLink.init(context);

            // GAID를 가져오고, Deferred Link 체크는 별도 스레드에서.
            AirBridgeExecutor.run(new InitializerTask(context));

        } catch (Throwable e) {
            Logger.wtf("Error occurred while initializing", e);
        }
    }

    /**
     * <a href="https://cordova.apache.org/">Cordova</a>에서 사용한다.
     * AirBridge SDK를 초기화하고, Activity 정보를 등록한다.
     * Application#onCreate 메서드에서 반드시 한 번 호출되어야 한다.
     *
     * @param context Application Context
     * @param appName Application Name (Airbridge 대시보드에서 "앱 영문명" 확인)
     * @param userToken User Token (Airbridge 대시보드에서 발급)
     */
    public static void initCordovaSDK(Context context, String appName, String userToken) {
        doReferrerTimeout = false;
        init(context, appName, userToken);
    }

    static class InitializerTask implements Runnable {

        private Context context;

        public InitializerTask(Context context) {
            this.context = context;
        }

        @Override
        public void run() {
            // 모든 이벤트는 디바이스 정보를 받아온 이후에 보내진다.
            DeviceInfo.fetch(context);
            StateContainer.set(DEVICE_INFO_FETCHED);

            if (!doInstallCheck) return;

            if (session.isFirstTime() && StateContainer.get() != INSTALL_WITH_DEEP_LINK) {
                Installer.install(context, doReferrerTimeout);

            } else {
                // AirBridge.deepLinkClicked 함수가 언제 호출될지 모르니 일정 시간 대기하고
                // 만약 이후에 호출되었다면 Launch 중복 발송을 방지하기 위해 Init Thread에선 Launch 이벤트를 보내지 않음.
                SystemClock.sleep(Constants.LAUNCH_DELAY);

                tracker.send(new LaunchEvent());
                Attribution.getInstance().update(context);

                StateContainer.set(INIT_FINISHED);
            }
        }
     }

    public static Tracker getTracker() {
        return tracker;
    }

    public static PageTracker getPageTracker() {
        return tracker.pageTracker;
    }

    /**
     * 3rd-party 연동을 추가한다.
     *
     * <code>
     *     AirBridge.turnOnIntegration(new AppboyIntegrator());
     * </code>
     *
     * @param integrator {@link Integrator}
     */
    public static void turnOnIntegration(Integrator integrator) {
        IntegrationManager.registerIntegration(integrator);
    }

    /**
     * 딥링크가 클릭되었을 때, 클릭 이벤트를 전송한다.
     *
     * @deprecated {@link DeepLinkLaunchEvent}를 {@link Tracker#send(Event)}를 통해 보내주세요.
     *
     * @param uri 들어온 딥 링크 URI
     */
    public static void deepLinkClicked(String uri) {
        tracker.send(new DeepLinkLaunchEvent(uri));
    }

    /**
     * 회원가입 목표 달성 이벤트를 전송한다.
     * @deprecated {@link SignUpGoalBuilder}를 사용해 주세요.
     */
    public static void userSignup(String value) {
        GoalEvent event = new SignUpGoalBuilder()
                .setUserId(value)
                .build();

        tracker.send(event);
    }

    /**
     * 회원가입 목표 달성 이벤트를 전송한다.
     * @deprecated {@link SignInGoalBuilder}를 사용해 주세요.
     */
    public static void userSignin(String value) {
        GoalEvent event = new SignUpGoalBuilder()
                .setUserId(value)
                .build();

        tracker.send(event);
    }


    /**
     * 특정 목표 (ex: 구매버튼 클릭) 달성 시, 목표 달성 이벤트를 전송한다.
     * @deprecated {@link GoalEvent}를 {@link Tracker#send(Event)} 메서드를 통해 보내주세요.
     * @param label 목표 이벤트 이름 (ex: 구매버튼 클릭)
     */
    public static void goal(String label) {
        goal(label, label, null);
    }

    /**
     * 특정 목표 (ex: 구매버튼 클릭) 달성 시, 목표 달성 이벤트를 전송한다.
     * @deprecated {@link GoalEvent}를 {@link Tracker#send(Event)} 메서드를 통해 보내주세요.
     * @param label 목표 이벤트 이름 (ex: 구매버튼 클릭)
     * @param category 목표 이벤트 카테고리. 동일하게 지정한 이벤트끼리 묶임.
     */
    public static void goal(String label, String category) {
        goal(label, category, null);
    }

    /**
     * 특정 목표 (ex: 구매버튼 클릭) 달성 시, 목표 달성 이벤트를 전송한다.
     * @deprecated {@link GoalEvent}를 {@link Tracker#send(Event)} 메서드를 통해 보내주세요.
     *
     * @param label 목표 이벤트 이름 (ex: 구매버튼 클릭)
     * @param category 목표 이벤트 카테고리. 동일하게 지정한 이벤트끼리 묶임.
     * @param key 외부 연동을 위한 키. (Optional)
     */
    public static void goal(String label, String category, String key) {
        tracker.send(new GoalEvent(category).setLabel(label).setCustomAttribute("key", key));
    }

    /**
     * 디버그 모드를 설정한다. 설정 시, 로그가 표시된다.
     * @param enabled 디버그 모드 여부
     */
    public static void setDebugMode(boolean enabled) {
        isDebugMode = enabled;
    }

    /**
     * 설정 시, 네트워크 에러나 서버 에러시에도 재시도하여 리퀘스트를 무조건 전송한다. 기본값은 true이다.
     * @param enabled 리퀘스트 보장 모드 여부
     */
    public static void setEnsureRequestMode(boolean enabled) {
        RequestQueue.ensureRequestMode = enabled;
    }

    /**
     * Install 체킹을 끈다. 테스트시 사용
     */
    static void disableInstallCheck() {
        doInstallCheck = false;
    }

    /**
     * 앱이 디버그 빌드인지 여부를 가져온다.
     */
    private static void fetchAppDebuggable(Context context) {
        try {
            PackageInfo packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(),
                    PackageManager.GET_SIGNATURES);
            Signature signatures[] = packageInfo.signatures;
            CertificateFactory cf = CertificateFactory.getInstance("X.509");

            for (Signature signature : signatures) {
                ByteArrayInputStream stream = new ByteArrayInputStream(signature.toByteArray());
                X509Certificate cert = (X509Certificate) cf.generateCertificate(stream);
                String certString = cert.getSubjectX500Principal().toString();
                isAppDebuggable = certString.contains("CN=Android Debug")
                        && certString.contains("O=Android");
            }

        } catch (Exception ignored) {
            isAppDebuggable = false;
        }
    }
}
