package io.airbridge.statistics.page;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Build;

import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

import io.airbridge.internal.log.Logger;
import io.airbridge.statistics.Tracker;
import io.airbridge.statistics.events.PageViewEvent;

/**
 * Tracks user interaction - especially about page views.
 * @author Hyojun Kim
 */
public class PageTracker {

    private Tracker tracker;

    // current page information.
    PageInfo currentPageInfo = new PageInfo();
    private Map<Activity, PageInfo> pageInfoCache = new HashMap<>();

    // save a previous state because server needs it.
    volatile AppActiveState previousState, currentState;

    public PageTracker(Context context, Tracker tracker) {
        this.tracker = tracker;
        previousState = currentState = AppActiveState.OFF;
        registerActivityLifecycleHandler(context);
    }

    /**
     * Registers handler of the foreground / background event.
     */
    private void registerActivityLifecycleHandler(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
            LifecycleTracker lifecycleCallback = new LifecycleTracker(this, tracker);

            Application app = (Application) context.getApplicationContext();
            app.registerComponentCallbacks(lifecycleCallback);
            app.registerActivityLifecycleCallbacks(lifecycleCallback);
        }
    }

    void setCurrentState(AppActiveState state) {
        if (previousState == state && currentState == state) return;
        Logger.v("State change : " + currentState.name() + " -> " + state.name());
        previousState = currentState;
        currentState = state;
    }

    public int getPreviousState() {
        return previousState.state;
    }

    /**
     * 사용자가 현재 어느 화면 (Activity)에 있는지에 대한 정보를 설정한다.
     * @param activity 현재 사용자가 머무르는 {@link Activity}
     */
    public void setCurrentPage(Activity activity) {
        PageInfo pageInfo = pageInfoCache.get(activity);
        if (pageInfo == null) {
            pageInfo = new PageInfo(activity);
            pageInfoCache.put(activity, pageInfo);
        }

        // update activity reference
        pageInfo.activity = new WeakReference<>(activity);

        currentPageInfo = pageInfo;
        if (tracker.isPageTrackingEnabled()) {
            // TODO: need to refactor to use Event#canBeSent
            tracker.send(new PageViewEvent());
        }
    }

    /**
     * 현재 화면 이름을 설정한다. Airbridge 대시보드에서 식별용으로 사용된다.
     * @param name 화면 이름
     */
    public void setCurrentPageName(String name) {
        if (currentPageInfo == null) {
            throw new IllegalStateException("You need to set current page first.");
        }
        currentPageInfo.name = name;
    }

    /**
     * 현재 화면에 대한 부가적인 정보를 추가한다.
     * @param key 정보의 키 값
     * @param value 정보 값
     */
    public void setCurrentPageData(String key, Object value) {
        if (currentPageInfo == null) {
            throw new IllegalStateException("You need to set current page first.");
        }
        currentPageInfo.customInfo.put(key, value);
    }

    public Activity getCurrentActivity() {
        if (currentPageInfo == null) {
            throw new IllegalStateException("You need to set current page first.");
        }
        return currentPageInfo.activity.get();
    }

    public JSONObject getCurrentPageData() {
        return currentPageInfo != null ? currentPageInfo.toJson() : new JSONObject();
    }
}
