package io.airbridge.internal.networking;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import org.json.JSONArray;

import java.util.Queue;
import java.util.concurrent.LinkedBlockingQueue;

import io.airbridge.Constants;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.tasks.AirBridgeExecutor;

/**
 * {@link ABRequest}를 오프라인에 큐잉하고 네트워크가 연결되면 다시 보낸다.
 */
public class RequestQueue {

     // 5초간 기다렸다가 한번에 이벤트를 보낸다.
    private static final int SEND_DELAY = 5;

    private static final IntentFilter RECONNECT_INTENT =
            new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION);

    /**
     * {@code true}일 시, 통계 전송 실패시에도 재시도 및 백오프를 통해 무조건 데이터를 서버로 보내려고 시도한다.
     */
    public static boolean ensureRequestMode = true;

    /**
     * 리퀘스트를 쌓는 중인가?
     */
    private volatile boolean collectingRequests = false;

    private Queue<ABRequest> requestQueue = new LinkedBlockingQueue<>();
    private Queue<ABRequest> failQueue = new LinkedBlockingQueue<>();

    private SharedPreferences prefs;
    private ConnectivityManager cm;
    private Context context;

    public RequestQueue(Context context) {
        this.context = context;
        prefs = context.getSharedPreferences(Constants.PREFS, Context.MODE_PRIVATE);
        cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

        loadOfflineQueue();
    }

    /**
     * Load failed / pending requests from offline.
     */
    private void loadOfflineQueue() {
        if (!ensureRequestMode) return;

        try {
            JSONArray jsonQueue = new JSONArray(prefs.getString("queue", "[]"));

            int length = jsonQueue.length();
            for (int i=0; i<length; i++) {
                failQueue.add(ABRequest.fromJson(jsonQueue.get(i).toString()));
            }
        } catch (Exception e) {
            Logger.e("Failed to load offline request queue", e);
        }
    }

    /**
     * Send all requests in fail queue.
     */
    void retrySending() {
        if (!ensureRequestMode) return;

        ABRequest request;
        while ((request = failQueue.poll()) != null) {
            if (request.retryDelay > Constants.MAX_RETRY_DELAY) {
                removeFromFailQueue(request);
                continue;
            }
            request.callAsync(responseCallback);
        }
        saveQueueToOffline();
        Logger.d("Retry sending all request");
    }

    /**
     * Send the request immediately. Retry if the network state appears to be offline.
     * @param request {@link ABRequest}
     */
    public void sendNow(ABRequest request) {
        // Exponential Backoff (https://en.wikipedia.org/wiki/Exponential_backoff) 를 수행하기 위함.
        request.shouldBackoff = true;

        if (!isNetworkConnected()) {
            addToFailQueue(request);
            context.registerReceiver(reconnectReceiver, RECONNECT_INTENT);
            Logger.d("Internet is not connected - pending request. [waitingQueueLength=%d]",
                    failQueue.size());
        } else {
            request.callAsync(responseCallback);

            // retry failed requests
            if (!failQueue.isEmpty()) retrySending();
        }
    }

    /**
     * Enqueue the request.
     * The queue thread will collect request for a while and send them at a go.
     * Retry if the network state appears to be offline.
     * @param request {ABRequest} that you want to sent
     */
    public void enqueue(ABRequest request) {
        requestQueue.add(request);

        if (collectingRequests) return;

        Logger.v("Collecting requests...");
        collectingRequests = true;

        // launch the queue thread.
        AirBridgeExecutor.runAfterTime(SEND_DELAY, new Runnable() {
            @Override
            public void run() {
                Logger.v("Finished collecting. Sending %d requests.", requestQueue.size());
                collectingRequests = false;
                ABRequest req;
                while ((req = requestQueue.poll()) != null) {
                    sendNow(req);
                }
            }
        });
    }

    private void addToFailQueue(ABRequest request) {
        failQueue.add(request);
        saveQueueToOffline();
    }

    private void removeFromFailQueue(ABRequest request) {
        failQueue.remove(request);
        saveQueueToOffline();
    }

    private void saveQueueToOffline() {
        JSONArray jsonQueue = new JSONArray();
        for (ABRequest req : failQueue) jsonQueue.put(req.toJson());
        prefs.edit().putString("queue", jsonQueue.toString()).apply();
    }

    private boolean isNetworkConnected() {
        NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
        return activeNetwork != null && activeNetwork.isConnectedOrConnecting();
    }

    /**
     * Called after the response comes.
     */
    private final ABRequest.Callback responseCallback = new ABRequest.Callback() {
        @Override
        public void done(final ABRequest request, ABResponse response) {
            if (response.isFailed() && response.status != 400) {
                // do binary backoff - multiply retry time by 2.
                if (request.retryDelay == 0) {
                    request.retryDelay = 1;
                }
                else request.retryDelay *= 2;

                if (request.retryDelay > Constants.MAX_RETRY_DELAY) return;

                addToFailQueue(request);
                AirBridgeExecutor.runAfterTime(request.retryDelay, new Runnable() {
                    @Override
                    public void run() {
                        Logger.v("Retrying request after %d seconds.", request.retryDelay);
                        removeFromFailQueue(request);
                        request.callAsync(responseCallback);
                    }
                });
            }
        }
    };

    /**
     * Called when network gets back online.
     */
    private final BroadcastReceiver reconnectReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context c, Intent intent) {
            if (isNetworkConnected()) {
                context.unregisterReceiver(this);

                // send all messages
                Logger.d("Internet reconnected!");
                retrySending();
            }
        }
    };
}
