package io.airbridge.statistics.page;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.ComponentCallbacks2;
import android.content.res.Configuration;
import android.os.Bundle;

import io.airbridge.internal.log.Logger;
import io.airbridge.internal.tasks.AirBridgeExecutor;
import io.airbridge.statistics.Tracker;
import io.airbridge.statistics.events.AppShutdownEvent;
import io.airbridge.statistics.events.BackgroundEvent;

import static io.airbridge.statistics.page.AppActiveState.ACTIVE;
import static io.airbridge.statistics.page.AppActiveState.INACTIVE;
import static io.airbridge.statistics.page.AppActiveState.OFF;

/**
 * 앱의 생명주기를 트래킹한다.
 * @author Hyojun Kim
 */
@SuppressLint("NewApi")
class LifecycleTracker implements Application.ActivityLifecycleCallbacks, ComponentCallbacks2 {

    private PageTracker pageTracker;
    private Tracker tracker;

    /**
     * 앱이 현재 UI상에 떠있는지 여부.
     */
    private boolean isInForeground = false;

    private volatile boolean isBackgroundEventCancelled = false;

    LifecycleTracker(PageTracker pageTracker, Tracker tracker) {
        this.pageTracker = pageTracker;
        this.tracker = tracker;
    }

    /**
     * 앱이 Foreground 상태가 될 때를 인식하고, 현재 Activity를 설정한다.
     */
    @Override
    public void onActivityResumed(Activity activity) {
        if (!isInForeground) {
            isInForeground = true;
            Logger.d("App is on foreground.");

        } else if (activity.getClass().getName().equals(pageTracker.currentPageInfo.componentName)) {
            // same page. don't need to update.
            return;
        }

        pageTracker.setCurrentState(ACTIVE);
        pageTracker.setCurrentPage(activity);
    }

    /**
     * 앱 종료를 감지한다.
     */
    @Override
    public void onActivityDestroyed(final Activity activity) {
        String destroyedActivityName = activity.getClass().getName();
        String currentActivityName = pageTracker.currentPageInfo.componentName;

        if (activity.isTaskRoot() && destroyedActivityName.equals(currentActivityName)) {
            Logger.d("App is shutting down.");

            // 앱 종료시엔 백그라운드 이벤트를 보내면 안된다.
            isBackgroundEventCancelled = true;

            pageTracker.setCurrentState(OFF);
            tracker.send(new AppShutdownEvent());
        }

//        // 앱이 백그라운드에 있고 (TRIM_MEMORY_HIDDEN) 현재 액티비티가 종료된다면 앱 종료이다.
//        if (destroyedActivityName.equals(currentActivityName)) {
//            Logger.d("App is shutting down.");
//        }
//        if (!isInForeground && destroyedActivityName.equals(currentActivityName)) {
//        }
    }

    /**
     * 앱이 내려질 때를 감지한다.
     */
    @Override
    public void onTrimMemory(int level) {
        if (level == TRIM_MEMORY_UI_HIDDEN && isInForeground) {
            isInForeground = false;

            // 2초간 미루면서 앱 종료 이벤트가 들어오는지 감지한다. 만약 들어오면 백그라운드를 보내면 안됨.
            AirBridgeExecutor.runAfterTime(2, new Runnable() {
                @Override
                public void run() {
                    if (isBackgroundEventCancelled) {
                        isBackgroundEventCancelled = false;
                        return;
                    }
                    // 그동안 앱 종료 이벤트가 일어나지 않았으므로 안심하고 백그라운드 이벤트를 보낸다.
                    pageTracker.setCurrentState(INACTIVE);
                    tracker.send(new BackgroundEvent());
                }
            });
            Logger.d("App is on background.");
        }
    }

    // UNUSED METHODS
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {}
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {}
    public void onConfigurationChanged(Configuration newConfig) {}
    public void onActivityStarted(Activity activity) {}
    public void onActivityStopped(Activity activity) {}
    public void onActivityPaused(Activity activity) {}
    public void onLowMemory() {}
}
