package io.airbridge.statistics;

import android.content.Context;
import android.net.Uri;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.airbridge.AirBridge;
import io.airbridge.Constants;
import io.airbridge.Session;
import io.airbridge.integration.IntegrationManager;
import io.airbridge.internal.log.Logger;
import io.airbridge.internal.networking.ABRequest;
import io.airbridge.internal.networking.ABResponse;

/**
 * Attribution 정보를 관리하고 업데이트한다.
 * @author Hyojun Kim
 */
public class Attribution {
    // Attribution 데이터 키 목록. 서버에서 보내주는 param key에 맞춘다.
    private static final List<String> ATTR_KEYS = Arrays.asList("channel", "campaign", "ad_group",
            "ad_creative", "content", "medium", "term");

    private static final String NAMESPACE = "attr:";

    private static Attribution instance;

    public static Attribution getInstance() {
        if (instance == null) instance = new Attribution();
        return instance;
    }

    Session prefs;
    Map<String, String> attrData = new HashMap<>();

    private Attribution() {
        prefs = Session.getCurrent();

        // load from storage
        for (String key : ATTR_KEYS) {
            attrData.put(key, prefs.getString(NAMESPACE + key, null));
        }
    }

    /**
     * 서버로부터 Attribution 정보를 업데이트한다.
     * 비동기 함수이다.
     */
    public void update(final Context context) {
        String url = Constants.STATS_ENDPOINT + AirBridge.appId
                + "/events/mobile-app/9163/attribution-result"
                + "?device_uuid=" + DeviceInfo.getInstance().uuid;

        new ABRequest("GET", url).callAsync(new ABRequest.Callback() {
            @Override
            public void done(ABRequest request, ABResponse response) {
                try {
                    JSONObject result = new JSONObject(response.body).getJSONObject("resource");
                    if (result.has("attributedTouchpoint")) {
                        JSONObject attributionData = result.getJSONObject("attributedTouchpoint");
                        Uri uri = Uri.parse(attributionData.getString("simplelink"));

                        saveOnlyChanged(uri, context);
                    }

                } catch (JSONException e) {
                    Logger.wtf("Failed to update attribution data.", e);
                }
            }
        });
    }

    private void saveOnlyChanged(Uri uri, Context context) {
        boolean changed = false;

        for (String key : ATTR_KEYS) {
            String localData = attrData.get(key);
            String remoteData = uri.getQueryParameter(key);
            if (remoteData != null && !remoteData.equals(localData)) {
                attrData.put(key, remoteData);
                changed = true;
            }
        }
        if (changed) {
            save();
            IntegrationManager.sendAttribution(this, context);
        }
    }

    private void save() {
        for (String key : ATTR_KEYS) {
            prefs.put(NAMESPACE + key, attrData.get(key));
        }
        Logger.d("Updated attribution info");
    }

    /**
     * Attribution 정보를 가져온다.
     * @param key channel, campaign, ad_group, ad_creative, content, medium, term 중 하나
     * @return Attribution 정보 (없을 시 null)
     */
    public String get(String key) {
        if (!ATTR_KEYS.contains(key)) throw new IllegalArgumentException("Unknown key : " + key);
        return attrData.get(key);
    }
}
