package io.airbridge.internal.tasks;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Light, Vanilla, and brief shim of CompletableFuture in Java 8.
 * @author Hyojun Kim
 */
public class CompletableFuture<T> implements Future<T> {
    private CountDownLatch lock = new CountDownLatch(1);
    private T data;
    private boolean done, cancelled;

    /**
     * Completes the task and sets data.
     * @param data Data you want to send to the awaiters.
     */
    public void complete(T data) {
        this.data = data;
        done = true;
        lock.countDown();
    }

    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        cancelled = true;
        return true;
    }

    @Override
    public boolean isCancelled() {
        return cancelled;
    }

    @Override
    public boolean isDone() {
        return done;
    }

    @Override
    public T get() throws InterruptedException, ExecutionException {
        lock.await();
        return data;
    }

    @Override
    public T get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        lock.await(timeout, unit);
        return data;
    }

    /**
     * 작업이 완료될 때 까지 주어진 ms만큼 기다린다. 작업이 이전에 완료되면 즉시 값을 반환한다.
     * 동기식 함수이므로 절대 Main Thread에서 사용하지 말 것.
     * @param timeoutInMilliseconds 시간
     * @return 결과 데이터
     */
    public T await(long timeoutInMilliseconds) {
        try {
            return get(timeoutInMilliseconds, TimeUnit.MILLISECONDS);

        } catch (Exception ignored) {
            // Interrupted or timeout.
            return null;
        }
    }
}
