package cucumber.pro;

import cucumber.pro.results.GroupGenerator;
import cucumber.pro.results.MetadataApi;
import cucumber.pro.results.NullMetadataClient;
import cucumber.pro.results.Response;
import cucumber.pro.results.Status;
import cucumber.pro.results.WebSocketMetadataClient;
import cucumber.pro.scm.WorkingCopy;
import cucumber.pro.scm.WorkingCopyDetector;
import cucumber.runtime.CucumberException;
import cucumber.runtime.Env;
import gherkin.formatter.Formatter;
import gherkin.formatter.Reporter;
import gherkin.formatter.model.Background;
import gherkin.formatter.model.BasicStatement;
import gherkin.formatter.model.Examples;
import gherkin.formatter.model.Feature;
import gherkin.formatter.model.Match;
import gherkin.formatter.model.Result;
import gherkin.formatter.model.Scenario;
import gherkin.formatter.model.ScenarioOutline;
import gherkin.formatter.model.Step;

import javax.websocket.DeploymentException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Future;

public class CucumberProFormatter implements Formatter, Reporter {
    private static Env ENV = new Env("cucumber");

    private static boolean shouldPublish() {
        String[] ciVariables = ENV.get("CUCUMBER_PRO_CI_VARIABLE", "BUILD_NUMBER,CI").split(",");

        for (String ciVariable : ciVariables) {
            String val = ENV.get(ciVariable);
            if (Boolean.parseBoolean(val)) {
                return true;
            }
        }
        return false;
    }

    private static URI resultsUri() throws URISyntaxException {
        return new URI(ENV.get("CUCUMBER_PRO_RESULTS_URL", "wss://results.cucumber.pro/ws"));
    }

    private static String getToken() {
        String token = ENV.get("CUCUMBER_PRO_TOKEN");
        if (token == null) {
            throw new CucumberException("Please define CUCUMBER_PRO_TOKEN");
        }
        return token;
    }

    private final MetadataApi metadataApi;
    private final WorkingCopy workingCopy;
    private final GroupGenerator groupGenerator;

    // These should be cleared out
    private List<Step> steps = new ArrayList<Step>();
    private BasicStatement statement;
    private String uri;
    private boolean headerSent;
    private List<Status> statuses = new ArrayList<>();
    private Future<Response> responseFuture;
    private int currentLine;

    private static MetadataApi createMetadataApi() throws URISyntaxException, IOException, DeploymentException {
        if (shouldPublish()) {
            return new WebSocketMetadataClient(new URI(resultsUri().toString() + "?token=" + getToken()));
        } else {
            return new NullMetadataClient();
        }
    }

    public CucumberProFormatter() throws Exception {
        this(createMetadataApi(), WorkingCopyDetector.detect(), GroupGenerator.DEFAULT);
    }

    CucumberProFormatter(MetadataApi metadataApi, WorkingCopy workingCopy, GroupGenerator groupGenerator) {
        this.metadataApi = metadataApi;
        this.workingCopy = workingCopy;
        this.groupGenerator = groupGenerator;
        if (shouldPublish()) {
            workingCopy.checkClean();
        }
    }

    // Reporter API

    @Override
    public void before(Match match, Result result) {
        // Need to move format() up in CucumberScenario so statement isn't null.
//        metadataApi.publishLineResult(uri, statement.getLine(), result.getStatus(), attachments);
    }

    @Override
    public void result(final Result result) {
        publishResult(result, steps.remove(0));
    }

    @Override
    public void after(Match match, Result result) {
        publishResult(result, statement);
    }

    private void publishResult(Result result, BasicStatement stmt) {
        Status status = getStatus(result);
        statuses.add(status);

        awaitPreviousFuture();
        responseFuture = metadataApi.sendTestStepResult(uri, stmt.getLine() + 1, status);

        String errorMessage = result.getErrorMessage();
        if (errorMessage != null) {
            awaitPreviousFuture();
            responseFuture = metadataApi.sendStackTrace(uri, stmt.getLine() + 1, errorMessage);
        }
    }

    private void awaitPreviousFuture() {
        if (responseFuture != null) {
            Response response = null;
            try {
                response = responseFuture.get();
                RuntimeException e = response.getException();
                if (e != null) {
                    throw e;
                }
            } catch (Exception e) {
                throw new CucumberException(e);
            }
        }
    }

    @Override
    public void match(Match match) {
    }

    @Override
    public void embedding(String mimeType, byte[] bytes) {
        awaitPreviousFuture();
        responseFuture = metadataApi.sendMetadata(uri, currentLine + 1, mimeType);

        awaitPreviousFuture();
        responseFuture = metadataApi.sendBinary(bytes);
    }

    @Override
    public void write(String text) {
    }

    // Formatter

    @Override
    public void uri(String uri) {
        this.uri = uri;
    }

    @Override
    public void feature(Feature feature) {
        if (!headerSent) {
            metadataApi.sendHeader(workingCopy.getRepoUrl(), workingCopy.getRev(), workingCopy.getBranch(), groupGenerator.group());
            headerSent = true;
        }
    }

    @Override
    public void background(Background background) {
    }

    @Override
    public void scenario(Scenario scenario) {
        this.statement = scenario;
    }

    @Override
    public void scenarioOutline(ScenarioOutline scenarioOutline) {
    }

    @Override
    public void examples(Examples examples) {
    }

    @Override
    public void startOfScenarioLifeCycle(Scenario scenario) {
        statuses.clear();
    }

    @Override
    public void step(Step step) {
        currentLine = step.getLine();
        steps.add(step);
    }

    @Override
    public void endOfScenarioLifeCycle(Scenario scenario) {
        awaitPreviousFuture();
        responseFuture = metadataApi.sendTestCaseResult(uri, scenario.getLine() + 1, getStatus());
    }

    private Status getStatus() {
        int pos = 0;
        for (Status status : statuses) {
            int p = Arrays.binarySearch(Status.values(), status);
            pos = Math.max(pos, p);
        }
        return Status.values()[pos];
    }

    private Status getStatus(Result result) {
        return Status.valueOf(result.getStatus());
    }


    @Override
    public void eof() {
    }

    @Override
    public void syntaxError(String state, String event, List<String> legalEvents, String uri, Integer line) {
    }

    @Override
    public void done() {
//        try {
//            metadataApi.finish(TIMEOUT_MILLIS);
//        } catch (InterruptedException e) {
//            throw new CucumberException(e);
//        }
    }

    @Override
    public void close() {
    }
}
