package cucumber.pro.results;

import cucumber.pro.results.metadata.Metadata;
import cucumber.pro.results.metadata.StackTrace;
import cucumber.pro.results.metadata.TestCaseResult;
import cucumber.pro.results.metadata.TestStepResult;
import org.glassfish.tyrus.client.ClientManager;
import org.glassfish.tyrus.core.HandshakeException;

import javax.websocket.ClientEndpoint;
import javax.websocket.DeploymentException;
import javax.websocket.OnMessage;
import javax.websocket.RemoteEndpoint;
import javax.websocket.Session;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

@ClientEndpoint(encoders = JsonEncoder.class, decoders = JsonDecoder.class)
public class ResultsClient implements MetadataApi {

    private static final long TIMEOUT_MILLIS = 15000;
    private final Future<Session> sessionFuture;
    private final BlockingQueue<Response> responseQueue = new SynchronousQueue<>();
    private final URI uri;

    public ResultsClient(URI uri) throws URISyntaxException, IOException, DeploymentException {
        this.uri = uri;
        ClientManager client = ClientManager.createClient();
        sessionFuture = client.asyncConnectToServer(this, uri);
    }

    @Override
    public Future<Response> sendHeader(String repoUrl, String rev, String branch, String group) {
        getEndpoint().sendObject(new Header(repoUrl, rev, branch, group));
        return new ResponseFuture();
    }

    @Override
    public Future<Response> sendTestStepResult(String path, int location, Status status) {
        getEndpoint().sendObject(new TestStepResult(path, location, status));
        return new ResponseFuture();
    }

    @Override
    public Future<Response> sendTestCaseResult(String path, int location, Status status) {
        getEndpoint().sendObject(new TestCaseResult(path, location, status));
        return new ResponseFuture();
    }

    @Override
    public Future<Response> sendStackTrace(String path, int location, String stackTrace) {
        getEndpoint().sendObject(new StackTrace(path, location, stackTrace));
        return new ResponseFuture();
    }

    @Override
    public Future<Response> sendMetadata(String path, int location, String mimeType) {
        getEndpoint().sendObject(new Metadata(path, location, mimeType));
        return new ResponseFuture();
    }

    @Override
    public Future<Response> sendBinary(byte[] attachment) {
        getEndpoint().sendBinary(ByteBuffer.wrap(attachment));
        return new ResponseFuture();
    }

    private RemoteEndpoint.Async getEndpoint() {
        try {
            Session session = sessionFuture.get(TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
            if (session == null) throw new RuntimeException("Couldn't connect to " + uri);
            return session.getAsyncRemote();
        } catch (ExecutionException e) {
            if (e.getCause() instanceof DeploymentException) {
                DeploymentException de = (DeploymentException) e.getCause();
                if (de.getCause() instanceof HandshakeException) {
                    HandshakeException he = (HandshakeException) de.getCause();

                    if (he.getMessage().contains("401")) {
                        throw new RuntimeException("Couldn't authenticate user: " + uri.getQuery());
                    } else if (he.getMessage().contains("403")) {
                        throw new RuntimeException("Access denied: " + uri.getQuery());
                    } else {
                        throw he;
                    }
                    // See https://java.net/jira/browse/TYRUS-321
//                    switch (he.getHttpStatusCode()) {
//                        case 401:
//                            throw new RuntimeException("Couldn't authenticate user: " + uri.getQuery());
//                        case 403:
//                            throw new RuntimeException("Access denied: " + uri.getQuery());
//                        default:
//                            throw he;
//                    }
                } else {
                    throw new RuntimeException(e);
                }
            } else {
                throw new RuntimeException(e);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @OnMessage
    public void onMessage(Response response) throws InterruptedException {
        responseQueue.put(response);
    }

    private class ResponseFuture implements Future<Response> {
        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean isCancelled() {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean isDone() {
            return !responseQueue.isEmpty();
        }

        @Override
        public Response get() throws InterruptedException, ExecutionException {
            return responseQueue.take();
        }

        @Override
        public Response get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            return responseQueue.poll(timeout, unit);
        }
    }
}
