package cucumber.pro.scm;

import java.io.File;
import java.util.List;

public class Git implements WorkingCopy {
    private final File rootDir;
    private final Exec exec;

    public static Git detect(File dir) {
        File workingCopyRoot = null;
        while (dir != null) {
            File dotSvn = new File(dir, ".git");
            if (dotSvn.isDirectory()) {
                workingCopyRoot = dir;
            }
            dir = dir.getParentFile();
        }
        return workingCopyRoot != null ? new Git(workingCopyRoot) : null;
    }

    public Git(File rootDir) {
        this.rootDir = rootDir;
        this.exec = new Exec(rootDir);
    }

    @Override
    public String getRevision() {
        return exec.cmd("git rev-parse HEAD").get(0);
    }

    @Override
    public String getRepoUrl() {
        // First, loop through all remotes
        List<String> remotes = exec.cmd("git ls-remote --get-url");
        for (String remote : remotes) {
            if (remote.contains("github") || remote.contains("bitbucket")) {
                // That's a good one.
                return remote;
            }
        }
        // Fall back to the origin url
        return exec.cmd("git config --get remote.origin.url").get(0);
    }

    @Override
    public String getBranch() {
        // CI servers typically check out a commit, so `git branch` won't work.
        // Output is "* branchname" so we strip the 2 first chars
        List<String> branches = exec.cmd("git branch --contains " + getRevision());
        for (String branch : branches) {
            if (!branch.contains("* (detached from")) {
                return branch.substring(2);
            }
        }
        throw new RuntimeException("Couldn't detect branch:\n" + branches);
    }

    File getRootDir() {
        return rootDir;
    }
}
