package cucumber.pro.publisher;

import cucumber.pro.scm.Scm;

import java.io.File;
import java.net.URL;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * Utility for publishing line results to Cucumber Pro. This class should be used by plugins
 * for various testing tools such as Cucumber-JVM, JUnit, TestNG etc.
 */
public class LineResultPublisher {
    private final ExecutorService executorService = Executors.newFixedThreadPool(5);
    private final URL baseUrl;
    private final Scm scm;
    private final File lineResultsBackupDir;
    private final String run = UUID.randomUUID().toString();

    public LineResultPublisher(URL baseUrl, Scm scm) {
        this.baseUrl = baseUrl;
        this.scm = scm;
        String results = ".cucumber-pro/results";
        lineResultsBackupDir = new File(scm.getRootDirectory(), results);
        scm.ignore(results);
        publishStoredLineResults();
    }

    /**
     * Publishes a line result.
     * <p/>
     * A line result represents the result of a single line in a file. The file can be identified
     * either by a relative path name from the root of the repo, or a path fragment in the repo.
     * The latter is useful when the relative path is unknown. This happens for example in Cucumber-JVM
     * when a feature file is loaded from within a jar file. We then assume that the path inside the jar
     * will also exist somewhere on the filesystem (in the repo) as source. The relative path will be found
     * (and validated for presence in the scm) in that case.
     * <p/>
     * If nothing is found it is considered an error (typically that the file exists, but it's not in the scm).
     * The method then throws a {@link cucumber.runtime.CucumberException}.
     * <p/>
     * No validation is performed to verify that the line is within the bounds of the file.
     *
     * @param pathOrFragment a relative path or a path fragment.
     * @param line           the line of the result.
     * @param status         the status of the result.
     * @param attachments    attachments to the result.
     */
    public void publishLineResult(final String pathOrFragment, final Integer line, final Status status, final List<Attachment> attachments) {
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                LineResultPublisherClient lineResultPublisherClient = new LineResultPublisherClient(baseUrl, run, lineResultsBackupDir, scm.copy());
                lineResultPublisherClient.publishLineResult(pathOrFragment, line, status, attachments);
            }
        });
    }

    private void publishStoredLineResults() {
        StoredLineResults storedLineResults = new StoredLineResults(lineResultsBackupDir);
        for (LineResult lineResult : storedLineResults.readStoredResults()) {
            publishLineResultAsync(lineResult);
        }
    }

    private void publishLineResultAsync(final LineResult lineResult) {
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                LineResultPublisherClient lineResultPublisherClient = new LineResultPublisherClient(baseUrl, run, lineResultsBackupDir, scm.copy());
                lineResultPublisherClient.publishLineResult(lineResult);
            }
        });
    }

    public void finish(long timeoutMillis) throws InterruptedException {
        executorService.shutdown();
        executorService.awaitTermination(timeoutMillis, TimeUnit.MILLISECONDS);
    }
}
