package cucumber.pro.scm;

import cucumber.runtime.CucumberException;
import gherkin.util.FixJava;
import org.eclipse.jgit.errors.AmbiguousObjectException;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.treewalk.TreeWalk;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class GitScm implements Scm {
    private static final Pattern GITHUB_SSH_URL = Pattern.compile("git@github.com:([^/]+)/([^.]+).git");
    private static final Pattern GITHUB_HTTPS_URL = Pattern.compile("https://github.com/([^/]+)/([^.]+).git");

    private final Repository repository;

    public GitScm(File pwd) {
        FileRepositoryBuilder builder = new FileRepositoryBuilder();
        try {
            repository = builder
                    .readEnvironment() // scan environment GIT_* variables
                    .findGitDir(pwd) // scan up the file system tree
                    .setMustExist(true)
                    .build();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public File getRootDirectory() {
        return repository.getDirectory().getParentFile();
    }

    @Override
    public String relativePathFromRoot(String pathOrFragment) {
        try {
            // http://stackoverflow.com/questions/1685228/how-to-cat-a-file-in-jgit
            ObjectId lastCommitId = repository.resolve(Constants.HEAD);
            RevWalk revWalk = new RevWalk(repository);
            RevCommit commit = revWalk.parseCommit(lastCommitId);
            RevTree tree = commit.getTree();
            TreeWalk treeWalk = new TreeWalk(repository);
            treeWalk.addTree(tree);
            treeWalk.setRecursive(true);

            List<String> candidates = new ArrayList<String>();
            while (treeWalk.next()) {
                String pathString = treeWalk.getPathString();
                if (pathString.endsWith(pathOrFragment)) {
                    candidates.add(pathString);
                }
            }
            if (candidates.size() != 1) {
                throw new CucumberException("Expected to find exactly one file for \"" + pathOrFragment + "\", got " + candidates);
            }
            return candidates.get(0);
        } catch (IncorrectObjectTypeException e) {
            throw new RuntimeException(e);
        } catch (AmbiguousObjectException e) {
            throw new RuntimeException(e);
        } catch (CorruptObjectException e) {
            throw new RuntimeException(e);
        } catch (MissingObjectException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getHeadRevision() {
        try {
            return repository.getRef("HEAD").getObjectId().getName();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getName() {
        StoredConfig config = repository.getConfig();
        Set<String> remotes = config.getSubsections("remote");
        for (String remote : remotes) {
            String url = config.getString("remote", remote, "url");

            {
                Matcher matcher = GITHUB_SSH_URL.matcher(url);
                if (matcher.matches()) {
                    return "github/" + matcher.group(1) + "/" + matcher.group(2);
                }
            }

            {
                Matcher matcher = GITHUB_HTTPS_URL.matcher(url);
                if (matcher.matches()) {
                    return "github/" + matcher.group(1) + "/" + matcher.group(2);
                }
            }
        }
        throw new RuntimeException("No github remotes");
    }

    @Override
    public GitScm copy() {
        return new GitScm(getRootDirectory());
    }

    @Override
    public void ignore(String path) {
        File gitignore = new File(getRootDirectory(), ".gitignore");
        try {
            boolean prependNewline = false;
            boolean alreadyIgnored = false;
            if (gitignore.isFile()) {
                String contents = FixJava.readReader(new FileReader(gitignore));
                prependNewline = !contents.endsWith("\n");
                String[] lines = contents.split("\n");
                for (String line : lines) {
                    if (line.equals(path) || line.equals("/" + path) || line.equals(path + "/") || line.equals("/" + path + "/")) {
                        alreadyIgnored = true;
                    }
                }
            }
            if (alreadyIgnored) {
                return;
            }
            FileWriter writer = new FileWriter(gitignore, true);
            if (prependNewline) {
                writer.append("\n");
            }
            writer.append("/").append(path).append("\n");
            writer.close();
        } catch (IOException e) {
            throw new CucumberException(e);
        }
    }
}
