package cucumber.pro.publisher;

import cucumber.pro.scm.Scm;
import cucumber.runtime.CucumberException;
import cucumber.runtime.Env;
import cucumber.runtime.io.URLOutputStream;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

class LineResultPublisherClient {
    private static final Map<String, String> HEADERS = new HashMap<String, String>() {{
        put("Accept", "application/json");
        put("Content-Type", "application/json");
        put("Authorization", "token " + new Env("cucumber-pro").get("GITHUB_TOKEN"));
    }};
    private static AtomicBoolean ERROR_PRINTED = new AtomicBoolean();

    private final URL baseUrl;
    private final File resultBackupDir;
    private final String run;

    final URL resultsUrl;
    final Scm scm;

    private long backupCounter = System.currentTimeMillis();

    public LineResultPublisherClient(URL baseUrl, String run, File resultBackupDir, Scm scm) {
        this.baseUrl = baseUrl;
        this.scm = scm;
        this.run = run;
        this.resultsUrl = getResultsUrl();
        this.resultBackupDir = resultBackupDir;
        if (!resultBackupDir.exists()) {
            this.resultBackupDir.mkdirs();
        }
    }

    private URL getResultsUrl() {
        try {
            return new URL(baseUrl, "api/" + scm.getName() + "/" + scm.getHeadRevision() + "/results");
        } catch (MalformedURLException e) {
            throw new CucumberException(e);
        }
    }

    String relativePathFromProjectRoot(String pathOrFragment) {
        File file = new File(pathOrFragment);
        if (file.isFile() && file.isAbsolute() && file.getAbsolutePath().startsWith(scm.getRootDirectory().getAbsolutePath())) {
            String relativePathFromRoot = file.getAbsolutePath().substring(scm.getRootDirectory().getAbsolutePath().length() + 1);
            return scm.relativePathFromRoot(relativePathFromRoot); // for validation
        } else {
            return scm.relativePathFromRoot(pathOrFragment);
        }
    }

    public void publishLineResult(String pathOrFragment, Integer line, Status status, List<Attachment> attachments) {
        LineResult lineResult = new LineResult(run, relativePathFromProjectRoot(pathOrFragment), line, status, attachments);
        publishLineResult(lineResult);
    }

    public void publishLineResult(LineResult lineResult) {
        String json = GsonHelper.GSON.toJson(lineResult);
        try {
            writeResult(json, new URLOutputStream(resultsUrl, "POST", HEADERS, 201));
            if (lineResult.getSource() != null) {
                // This result had previously failed to be published, but now it was published
                // successfully so we should delete the file.
                lineResult.getSource().delete();
            }
        } catch (IOException publicationFailed) {
            printErrorOnce(publicationFailed);
            if (lineResult.getSource() == null) {
                // This is the first time we're sending this result, so save it to disk
                // so it can be sent again in a later run.
                try {
                    File file = new File(resultBackupDir, String.valueOf(backupCounter++) + ".json");
                    System.err.format("Storing result in %s\n", file.getPath());
                    writeResult(json, new URLOutputStream(file.toURI().toURL()));
                } catch (IOException fileStorageFailed) {
                    throw new CucumberException("FATAL: Failed to publish and then failed to store result locally.", fileStorageFailed);
                }
            }
        }
    }

    private void printErrorOnce(IOException publicationFailed) {
        if (!ERROR_PRINTED.get()) {
            publicationFailed.printStackTrace();
            ERROR_PRINTED.set(true);
        }
    }

    private void writeResult(String json, OutputStream out) throws IOException {
        OutputStreamWriter writer = new OutputStreamWriter(out, Charset.forName("utf-8"));
        writer.write(json);
        writer.flush();
        writer.close();
    }
}
