package im.zego.zegoexpress;

import android.app.Application;
import android.content.Context;
import android.graphics.SurfaceTexture;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import android.os.*;

import org.json.*;

import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.entity.*;
import im.zego.zegoexpress.utils.*;

import java.nio.*;
import java.util.*;

import static im.zego.zegoexpress.ZegoExpressErrorCode.CommonEventHandlerExists;

final class ZegoExpressEngineJni {
    private static Handler mUIHandler;
    static private volatile boolean hasSoLoaded;
    private static volatile ZegoExpressEngine engine = null;
    private static Application context;
    private static boolean mIsTestEnv = true;
    private static boolean enableToastOnTestEnv = true;
    private static boolean enableDebugErrorAlways = false;

    private static ZegoLanguage language = ZegoLanguage.ENGLISH;
    private static IZegoEventHandler eventHandler = null;
    private static HashMap<Integer, IZegoPublisherUpdateCdnUrlCallback> sPublisherUpdateCDNURLHandler = new HashMap<>();
    private static HashMap<Integer, IZegoPublisherSetStreamExtraInfoCallback> sPublisherUpdateStreamExtraInfoHandler = new HashMap<>();
    private static HashMap<Integer, IZegoIMSendBarrageMessageCallback> sIMSendBarragetMssageHandler = new HashMap<>();

    private static HashMap<Integer, IZegoMixerStartCallback> sMixerStartResultHandler = new HashMap<>();
    private static HashMap<Integer, IZegoMixerStopCallback> sMixerStopResultHandler = new HashMap<>();
    private static HashMap<Integer, IZegoIMSendBroadcastMessageCallback> sIMSendBoradcastMssageHandler = new HashMap<>();
    private static HashMap<Integer, IZegoIMSendCustomCommandCallback> sIMSendCustomCommandHandler = new HashMap<>();

    final public static class ZegoInnerModule {
        public static final int COMMON = 0;
        public static final int ENGINE = 1;
        public static final int ROOM = 2;
        public static final int PUBLISHER = 3;
        public static final int PLAYER = 4;
        public static final int MIXER = 5;
        public static final int DEVICE = 6;
        public static final int PREPROCESS = 7;
        public static final int MEDIAPLAYER = 8;
        public static final int IM = 9;
        public static final int RECODER = 10;
        public static final int CUSTOM_VIDEO_IO = 11;
    }

    private static final int MAX_EVENT_HANDLE_COUNT = 16;
    private static final int ZEGO_EXPRESS_MODULE_JNI = 0xc;
    private static final int RESERVE_SEGMENT = 1000000;
    private static final int ERRCODE_OFFSET = 1000;
    private static final int ZEGO_ERRCODE_SUCCESS = 0x0;
    private static final int ZEGO_ERRCODE_COMMON_ENGINE_NOT_CREATED = RESERVE_SEGMENT + ZegoInnerModule.COMMON * ERRCODE_OFFSET + 1;
    private static final int ZEGO_ERRCODE_ENGINE_APPID_ZERO = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET;
    private static final int ZEGO_ERRCODE_ENGINE_APPSIGN_INVALID_LENGTH = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET + 1;
    private static final int ZEGO_ERRCODE_ENGINE_APPSIGN_INVALID_CHARACTER = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET + 2;
    private static final int ZEGO_ERRCODE_ENGINE_APPSIGN_NULL = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET + 3;
    private static final int ZEGO_ERRCODE_ENGINE_EVENT_HANDLER_NULL = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET + 27;
    private static final int ZEGO_ERRCODE_ENGINE_EVENT_HANDLER_COUNT_EXCEED = RESERVE_SEGMENT + ZegoInnerModule.ENGINE * ERRCODE_OFFSET + 28;


    private static Object mCustomVideoRenderHandler;

    private static IZegoDestroyCompletionCallback iZegoDestroyCompletionCallback;

    private static Object mCustomVideoCaptureHandler;

    private static IZegoAudioMixingHandler iZegoAudioMixingHandler;


    static {
        try {
            System.loadLibrary("ZegoExpressEngine");
            hasSoLoaded = true;
        } catch (UnsatisfiedLinkError e) {
            //Log.e("ZEGO", "load ZegoExpressSDK native library failed", e);
            hasSoLoaded = false;
        }
    }

    //TODO 该接口需要做单元测试
    static boolean ensureSoLoaded(Context appContext, String customizeSoPath) {
        if (!hasSoLoaded) {
            //优先加载外部指定的so
            if (!TextUtils.isEmpty(customizeSoPath)) {
                try {
                    hasSoLoaded = ZegoLibraryLoadUtil.loadSpecialLibrary(customizeSoPath, appContext);
                } catch (UnsatisfiedLinkError e) {
                    Log.e("ZEGO", String.format("Load library %s failed", customizeSoPath), e);
                    return false;
                } catch (Exception e) {
                    Log.e("ZEGO", String.format("Load library %s failed", customizeSoPath), e);
                    return false;
                }
            }
        }

        if (!hasSoLoaded) {
            //当加载外部指定so失败时，指定库文件重新加载
            try {
                hasSoLoaded = ZegoLibraryLoadUtil.loadSoFile("libZegoExpressSDK.so", appContext);
                return hasSoLoaded;
            } catch (UnsatisfiedLinkError e) {
                Log.e("ZEGO", "Load library libZegoExpressSDK.so failed", e);
                return false;
            }
        }
        return true;
    }


    private static im.zego.zegoexpress.entity.ZegoEngineConfig mEngineConfig;

    public static void setEngineConfig(ZegoEngineConfig engineConfig) {
        if (engineConfig != null) {
            // 如果外层未设置 engineConfig , 桥接层应new一份实例，传到jni
            if (engineConfig.advancedConfig != null) {
                if (engineConfig.advancedConfig.containsKey("enable_toast_on_test_env")) {
                    String enableToastOnTestEnvStr = engineConfig.advancedConfig.get("enable_toast_on_test_env");
                    if (enableToastOnTestEnvStr != null) {
                        enableToastOnTestEnv = !enableToastOnTestEnvStr.equals("false");
                    }
                }
                if (engineConfig.advancedConfig.containsKey("enable_debug_error_always")) {
                    String enableDebugErrorAlwaysStr = engineConfig.advancedConfig.get("enable_debug_error_always");
                    if (enableDebugErrorAlwaysStr != null) {
                        enableDebugErrorAlways = !enableDebugErrorAlwaysStr.equals("false");
                    }
                }
            }
            mEngineConfig = engineConfig;
        } else {
            mEngineConfig = new ZegoEngineConfig();
        }

        String advanceConfig = "";
        if (mEngineConfig.advancedConfig != null) {
            for (String key : mEngineConfig.advancedConfig.keySet()) {
                advanceConfig += key + "=" + mEngineConfig.advancedConfig.get(key) + ";";
            }
        }

        // 当App层设置了 customVideoRenderConfig , 说明开启了外部渲染
        // 由于jni层处理字段为枚举类型的对象时不方便, 这里转为基础类型来向jni传参
        boolean engineConfig_enableCustomVideoRender;
        boolean engineConfig_isEngineRender;
        int engineConfig_zegoCustomVideoRenderSeries;
        int engineConfig_zegoCustomVideoRenderType;
        if (mEngineConfig.customVideoRenderConfig != null) {
            engineConfig_enableCustomVideoRender = true;
            engineConfig_zegoCustomVideoRenderSeries = mEngineConfig.customVideoRenderConfig.frameFormatSeries.value();
            engineConfig_isEngineRender = mEngineConfig.customVideoRenderConfig.enableEngineRender;
            engineConfig_zegoCustomVideoRenderType = mEngineConfig.customVideoRenderConfig.bufferType.value();
        } else {
            engineConfig_enableCustomVideoRender = false;
            engineConfig_isEngineRender = false;
            engineConfig_zegoCustomVideoRenderSeries = -1;
            engineConfig_zegoCustomVideoRenderType = -1;
        }

        // 当App层设置了 videoCaptureConfig , 说明开启了外部采集
        // 由于jni层处理字段为枚举类型的对象时不方便, 这里转为基础类型来向jni传参
        boolean engineConfig_enableCustomVideoCaptureMain;
        int engineConfig_zegoCustomVideoCaptureMainType;
        if (mEngineConfig.customVideoCaptureMainConfig != null) {
            engineConfig_enableCustomVideoCaptureMain = true;
            engineConfig_zegoCustomVideoCaptureMainType = mEngineConfig.customVideoCaptureMainConfig.bufferType.value();
        } else {
            engineConfig_enableCustomVideoCaptureMain = false;
            engineConfig_zegoCustomVideoCaptureMainType = -1;
        }

        boolean engineConfig_enableCustomVideoCaptureAux;
        int engineConfig_zegoCustomVideoCaptureAuxType;

        if (mEngineConfig.customVideoCaptureAuxConfig != null) {
            engineConfig_enableCustomVideoCaptureAux = true;
            engineConfig_zegoCustomVideoCaptureAuxType = mEngineConfig.customVideoCaptureAuxConfig.bufferType.value();
        } else {
            engineConfig_enableCustomVideoCaptureAux = false;
            engineConfig_zegoCustomVideoCaptureAuxType = -1;
        }

        /*try {
            final Context mContext = context;
            if (ZegoExpressEngineJni.engineConfig.logConfig.logPath.equals("") && mContext != null) {
                ZegoExpressEngineJni.engineConfig.logConfig.logPath = ZegoLogUtil.getLogPath(mContext);
            }
        } catch (Exception ex) {
            switch (language) {
                case ENGLISH:
                    throw new RuntimeException("Input application parameter abnormal");
                case CHINESE:
                    throw new RuntimeException("输入application参数异常");
            }
        }*/

        ZegoExpressEngineJni.setEngineInitConfigToJni(mEngineConfig,
                engineConfig_enableCustomVideoRender, engineConfig_zegoCustomVideoRenderSeries,
                engineConfig_isEngineRender, engineConfig_zegoCustomVideoRenderType,
                engineConfig_enableCustomVideoCaptureMain, engineConfig_zegoCustomVideoCaptureMainType,
                engineConfig_enableCustomVideoCaptureAux, engineConfig_zegoCustomVideoCaptureAuxType,
                mEngineConfig.logConfig.logPath, mEngineConfig.logConfig.logSize,
                advanceConfig);
    }


    public static ZegoExpressEngine createEngine(long appID, String appSign, boolean isTestEnvironment,
                                                 ZegoScenario scenario, Application application, IZegoEventHandler handler) {

        // 防止多次调用在Java层产生多份实例
        if (engine != null) {
            return engine;
        }

        // 防止 application 参数传空
        if (application == null) {
            switch (language) {
                case ENGLISH:
                    throw new RuntimeException("Input application parameter abnormal");
                case CHINESE:
                    throw new RuntimeException("输入application参数异常");
            }
        }
        // setEngineConfig();

        int errorCode = ZegoExpressEngineJni.engineInitJni(appID, appSign, isTestEnvironment, scenario.value(), application);

        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.COMMON, "createEngine", errorCode);
        if (errorCode != 0) {
            return null;
        } else {
            synchronized (ZegoExpressEngineJni.class) {
                mUIHandler = new Handler(Looper.getMainLooper());
                context = application;
                mIsTestEnv = isTestEnvironment;
                engine = new ZegoExpressEngine();
                if (handler != null) {
                    engine.setEventHandler(handler);
                }
                return engine;
            }
        }

    }


    public static ZegoExpressEngine getEngine() {
        return engine;
    }

    public static ZegoAudioConfig getAudioConfig() {
        return getAudioConfigJni();
    }

    public static ZegoVideoConfig getVideoConfig(ZegoPublishChannel main) {
        return getVideoConfigJni(main.value());
    }

    public static void destroyEngine(final IZegoDestroyCompletionCallback callback) {
        synchronized (ZegoExpressEngineJni.class) {
            iZegoDestroyCompletionCallback = callback;
            release();
            int errorCode = ZegoExpressEngineJni.engineUninitAsyncJni();
        }
    }

    public static int setEventHandler(IZegoEventHandler handler) {
        int errorCode = ZEGO_ERRCODE_SUCCESS;
        synchronized (ZegoExpressEngineJni.class) {
            if (engine == null) {
                errorCode = ZEGO_ERRCODE_COMMON_ENGINE_NOT_CREATED;
            } else {
                if (eventHandler != null && handler != null) {
                    errorCode = CommonEventHandlerExists;
                    printDebugInfo(ZegoDebugLevel.ERROR, ZegoInnerModule.ENGINE, "setEventHandler", CommonEventHandlerExists);
                } else {
                    eventHandler = handler;
                }
            }
        }
        return errorCode;
    }


    private static void release() {
        eventHandler = null;
        if (sPublisherUpdateCDNURLHandler != null) {
            sPublisherUpdateCDNURLHandler.clear();
        }
        if (sPublisherUpdateStreamExtraInfoHandler != null) {
            sPublisherUpdateStreamExtraInfoHandler.clear();
        }
        if (sMixerStartResultHandler != null) {
            sMixerStartResultHandler.clear();
        }

        if (sMixerStopResultHandler != null) {
            sMixerStopResultHandler.clear();
        }

        if (sIMSendBoradcastMssageHandler != null) {
            sIMSendBoradcastMssageHandler.clear();
        }
        if (sIMSendCustomCommandHandler != null) {
            sIMSendCustomCommandHandler.clear();
        }
        if (sIMSendBarragetMssageHandler != null) {
            sIMSendBarragetMssageHandler.clear();
        }

        mCustomVideoRenderHandler = null;
        mCustomVideoCaptureHandler = null;

        context = null;
        engine = null;
        mIsTestEnv = true;
        mUIHandler = null;
        language = ZegoLanguage.ENGLISH;

        // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
        ZegoMediaPlayerJni.destroyAllMediaPlayer();

        // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

        setCustomVideoCaptureHandler(null);
        setCustomVideoRenderHandler(null);


    }


    public static String getVersion() {
        return getVersionJni();
    }

    public static void setDebugVerbose(boolean enable, ZegoLanguage language) {
        ZegoExpressEngineJni.language = language;
        ZegoExpressEngineJni.setDebugVerboseJni(enable, language.value());
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ENGINE,
                "setDebugVerbose", ZEGO_ERRCODE_SUCCESS);
    }

    public static void uploadLog() {
        ZegoExpressEngineJni.uploadLogJni();
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ENGINE,
                "uploadLog", ZEGO_ERRCODE_SUCCESS);
    }


    public static int loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config) {
        int maxMemberCount = 0;
        int userStateNotify = 0;
        if (config != null) {
            maxMemberCount = config.maxMemberCount;
            userStateNotify = config.isUserStatusNotify ? 1 : 0;
        }
        String userID = "";
        String userName = "";
        if (user != null) {
            userID = user.userID;
            userName = user.userName;
        }
        if (roomID == null) {
            roomID = "";
        }

        int errorCode = loginRoomJni(user, roomID, config);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ROOM, "loginRoom", errorCode);
        return errorCode;
    }

    public static int loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config, String token) {
        int errorCode = loginRoomJni(user, roomID, config, token);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ROOM, "loginRoom", errorCode);
        return errorCode;
    }

    public static boolean isMicrophoneMuted() {
        return isMicrophoneMutedJni();
    }

    public static boolean isSpeakerMuted() {
        return isSpeakerMutedJni();
    }

    public static void enableHeadphoneMonitor(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableHeadphoneMonitorJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.DEVICE, "enableHeadphoneMonitor", errorCode);
    }

    public static void enableCustomVideoCapture(boolean enable, ZegoCustomVideoCaptureConfig config, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.enableCustomVideoCaptureJni(enable, config, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "enableHeadphoneMonitor", errorCode);
    }

    public static void enableCustomVideoRender(boolean enable, ZegoCustomVideoRenderConfig config) {
        int errorCode = ZegoExpressEngineJni.enableCustomVideoRenderJni(enable, config);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "enableCustomVideoRender", errorCode);
    }

    public static void setHeadphoneMonitorVolume(int volume) {
        int errorCode = ZegoExpressEngineJni.setHeadphoneMonitorVolumeJni(volume);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.DEVICE, "setHeadphoneMonitorVolume", errorCode);
    }

    public static void logoutRoom(String roomID) {
        int errorCode = ZegoExpressEngineJni.logoutRoomJni(roomID);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ROOM, "logoutRoom", errorCode);
    }

    public static void setVideoMirrorMode(ZegoVideoMirrorMode mirrorMode, ZegoPublishChannel channel) {
        int errorCode;
        if (mirrorMode == null) {
            errorCode = ZegoExpressEngineJni.ZegoVideoMirrorModeJni(ZegoVideoMirrorMode.ONLY_PREVIEW_MIRROR.value(), channel.value());
        } else {
            errorCode = ZegoExpressEngineJni.ZegoVideoMirrorModeJni(mirrorMode.value(), channel.value());
        }

        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setVideoMirrorMode", errorCode);
    }

    public static void setAppOrientation(ZegoOrientation orientation, ZegoPublishChannel channel) {
        int errorCode;
        if (orientation == null) {
            errorCode = ZegoExpressEngineJni.setAppOrientationJni(ZegoOrientation.ORIENTATION_0.value(), channel.value());
        } else {
            errorCode = ZegoExpressEngineJni.setAppOrientationJni(orientation.value(), channel.value());
        }
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setAppOrientation", errorCode);
    }


    public static void startPreview(ZegoCanvas canvas, ZegoPublishChannel channel) {
        int errorCode;
        if (canvas == null) {
            errorCode = ZegoExpressEngineJni.startPreviewJni(null, ZegoViewMode.ASPECT_FILL.value(), 0, channel.value());
        } else if (canvas.viewMode == null) {
            errorCode = ZegoExpressEngineJni.startPreviewJni(canvas.view, ZegoViewMode.ASPECT_FILL.value(), canvas.backgroundColor, channel.value());
        } else {
            errorCode = ZegoExpressEngineJni.startPreviewJni(canvas.view, canvas.viewMode.value(), canvas.backgroundColor, channel.value());
        }
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "startPreview", errorCode);
    }

    public static void stopPreview(ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.stopPreviewJni(channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "stopPreview", errorCode);
    }

    public static void setVideoConfig(ZegoVideoConfig videoConfig, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.setVideoConfigJni(videoConfig, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setVideoConfig", errorCode);
    }

    public static int startPublishingStream(String streamID, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.startPublishingStreamJni(streamID, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "startPublish", errorCode);
        return errorCode;
    }

    public static void stopPublishingStream(ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.stopPublishingStreamJni(channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "stopPublishJni", errorCode);
    }

    public static void mutePublishStreamAudio(boolean mute, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.mutePublishStreamAudioJni(mute, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "mutePublishStreamAudio", errorCode);
    }

    public static void mutePublishStreamVideo(boolean mute, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.mutePublishStreamVideoJni(mute, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "mutePublishStreamVideo", errorCode);
    }

    public static void setStreamExtraInfo(String extraInfo, ZegoPublishChannel channel, IZegoPublisherSetStreamExtraInfoCallback handler) {
        int seq = ZegoExpressEngineJni.setStreamExtraInfoJni(extraInfo, channel.value());
        synchronized (ZegoExpressEngineJni.class) {
            sPublisherUpdateStreamExtraInfoHandler.put(new Integer(seq), handler);
        }
    }

    public static void setCaptureVolume(int volume) {
        int errorCode = ZegoExpressEngineJni.setCaptureVolumeJni(volume);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setCaptureVolume", errorCode);
    }

    public static void setAudioConfig(ZegoAudioConfig config) {
        int errorCode = 0;
        if (config != null) {
            errorCode = ZegoExpressEngineJni.setAudioConfigJni(config.bitrate, config.channel.value(), config.codecID.value());
        } else {
            errorCode = ZegoExpressEngineJni.setAudioConfigJni(-1, -1, -1);
        }
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setAudioConfig", errorCode);
    }


    public static void enableTrafficControl(boolean enable, int controlTypeBitMask) {
        int errorCode = ZegoExpressEngineJni.enableTrafficControlJni(enable, controlTypeBitMask);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableTrafficControl", errorCode);
    }

    public static void enablePublishDirectToCDN(boolean enable, ZegoCDNConfig zegoCDNConfig, ZegoPublishChannel channel) {
        ZegoExpressEngineJni.enablePublishDirectToCDNJni(enable, zegoCDNConfig, channel.value());
    }


    public static void addPublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback handler) {
        int seq = ZegoExpressEngineJni.addPublishCdnUrlJni(streamID, targetURL);
        synchronized (ZegoExpressEngineJni.class) {
            sPublisherUpdateCDNURLHandler.put(new Integer(seq), handler);
        }
    }

    public static void removePublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback handler) {
        int seq = ZegoExpressEngineJni.removePublishCdnUrlJni(streamID, targetURL);
        synchronized (ZegoExpressEngineJni.class) {
            sPublisherUpdateCDNURLHandler.put(new Integer(seq), handler);
        }
    }

    public static void setCapturePipelineScaleMode(ZegoCapturePipelineScaleMode mode) {
        int errorCode = ZegoExpressEngineJni.setCapturePipelineScaleModeJni(mode.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setCapturePipelineScaleMode", errorCode);
    }

    public static void setPublishWatermark(ZegoWatermark watermark, boolean isPreviewVisible, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.setPublishWatermarkJni(watermark, isPreviewVisible, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setPublishWatermark", errorCode);
    }

    public static void enableHardwareEncoder(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableHardwareEncoderJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableHardwareEncoder", errorCode);
    }

    public static void enableDTX(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableDTXJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableDTX", errorCode);
    }

    public static void enableVAD(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableVADJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableVAD", errorCode);
    }

    public static int startPlayingStream(String streamID, ZegoCanvas canvas, ZegoPlayerConfig config) {
        //todo 后面让c层修改, 如果重复传入不同的view，会有什么影响？应该要提示用户或者帮客户换View

        int videoLayer = ZegoPlayerVideoLayer.AUTO.value();
        if (config != null && config.videoLayer != null) {
            videoLayer = config.videoLayer.value();
        }

        ZegoCDNConfig zegoCDNConfig = null;
        if (config != null && config.cdnConfig != null) {
            zegoCDNConfig = config.cdnConfig;
        }

        int errorCode;
        if (canvas == null) {
            errorCode = ZegoExpressEngineJni.startPlayingStreamJni(streamID, null, 0, 0, zegoCDNConfig, videoLayer);

        } else {
            errorCode = ZegoExpressEngineJni.startPlayingStreamJni(streamID, canvas.view, canvas.viewMode == null ? ZegoViewMode.ASPECT_FIT.value() : canvas.viewMode.value(), canvas.backgroundColor, zegoCDNConfig, videoLayer);
        }

        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "startPlayingStream", errorCode);
        return errorCode;
    }

    public static void stopPlayingStream(String streamID) {
        int errorCode = ZegoExpressEngineJni.stopPlayingStreamJni(streamID);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "stopPlayingStream", errorCode);
    }

    public static void setPlayVolume(String streamID, int volume) {
        int errorCode = ZegoExpressEngineJni.setPlayVolumeJni(streamID, volume);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "setPlayVolume", errorCode);
    }

    public static void mutePlayStreamAudio(String streamID, boolean mute) {
        int errorCode = ZegoExpressEngineJni.mutePlayStreamAudioJni(streamID, mute);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "mutePlayStreamAudio", errorCode);
    }

    public static void mutePlayStreamVideo(String streamID, boolean mute) {
        int errorCode = ZegoExpressEngineJni.mutePlayStreamVideoJni(streamID, mute);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "mutePlayStreamVideo", errorCode);
    }

    public static void setMinVideoBitrateForTrafficControl(int bitrate, ZegoTrafficControlMinVideoBitrateMode mode) {
        int errorCode = ZegoExpressEngineJni.setMinVideoBitrateForTrafficControlJni(bitrate, mode.value());
    }

    // opt_content_start: audio_or_video = video
    public static void enableHardwareDecoder(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableHardwareDecoderJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PLAYER, "enableHardwareDecoder", errorCode);
    }
    // opt_content_end: audio_or_video = video

    public static void startMixerTask(ZegoMixerTask task, IZegoMixerStartCallback handler) {
        int seq = ZegoExpressEngineJni.startMixerJni(task);
        synchronized (ZegoExpressEngineJni.class) {
            sMixerStartResultHandler.put(seq, handler);
        }
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.MIXER, "startMixerTask", 0);
    }

    public static void enableAudioMixing(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableAudioMixingJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "enableAudioMixing", errorCode);
    }


    public static void setAudioMixingHandler(IZegoAudioMixingHandler handler) {
        iZegoAudioMixingHandler = handler;
    }

    public static void muteLocalAudioMixing(boolean mute) {
        int errorCode = ZegoExpressEngineJni.muteLocalAudioMixingJni(mute);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "muteLocalAudioMixing", errorCode);
    }


    public static void setAudioMixingVolume(int volume, ZegoVolumeType type) {
        int errorCode = ZegoExpressEngineJni.setAudioMixingVolumeJniWithType(volume, type.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "setAudioMixingVolume", errorCode);
    }

    public static void setAudioMixingVolume(int volume) {
        int errorCode = ZegoExpressEngineJni.setAudioMixingVolumeJni(volume);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.ENGINE, "setAudioMixingVolume", errorCode);
    }

    public static void stopMixerTask(ZegoMixerTask task, IZegoMixerStopCallback handler) {
        int seq = ZegoExpressEngineJni.stopMixerJni(task);
        synchronized (ZegoExpressEngineJni.class) {
            sMixerStopResultHandler.put(seq, handler);
        }
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.MIXER, "stopMixerTask", 0);
    }

    public static void muteMicrophone(boolean mute) {
        int errorCode = ZegoExpressEngineJni.muteMicrophoneJni(mute);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "muteMicrophone", errorCode);
    }

    public static void muteAudioOutput(boolean enable) {
        int errorCode = ZegoExpressEngineJni.muteAudioOutputJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "muteAudioOutput", errorCode);
    }

    public static void muteSpeaker(boolean enable) {
        int errorCode = ZegoExpressEngineJni.muteSpeakerJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "muteSpeaker", errorCode);
    }

    // opt_content_start: audio_or_video = video
    public static void enableCamera(boolean enable, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.enableCameraJni(enable, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableCamera", errorCode);
    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void useFrontCamera(boolean enable, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.useFrontCameraJni(enable, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "useFrontCamera", errorCode);
    }
    // opt_content_end: audio_or_video = video

    public static void startSoundLevelMonitor() {
        int errorCode = ZegoExpressEngineJni.startSoundLevelMonitorJni();
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "startSoundLevelMonitor", errorCode);
    }

    public static void stopSoundLevelMonitor() {
        int errorCode = ZegoExpressEngineJni.stopSoundLevelMonitorJni();
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "stopSoundLevelMonitor", errorCode);
    }

    public static void startAudioSpectrumMonitor() {
        int errorCode = ZegoExpressEngineJni.startFrequencySpectrumMonitorJni();
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "startFrequencySpectrumMonitor", errorCode);
    }

    public static void stopAudioSpectrumMonitor() {
        int errorCode = ZegoExpressEngineJni.stopFrequencySpectrumMonitorJni();
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "stopFrequencySpectrumMonitor", errorCode);
    }

    public static void enableAudioCaptureDevice(boolean enable) {
        //todo 名称修改为audio
        int errorCode = ZegoExpressEngineJni.enableAudioCaptureDeviceJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableCaptureDevice", errorCode);
    }


    public static void enableAEC(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableAECJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableAEC", errorCode);
    }

    public static void setAECMode(ZegoAECMode mode) {
        int errorCode = ZegoExpressEngineJni.setAECModeJni(mode.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setAECMode", errorCode);
    }

    // opt_content_start: audio_or_video = video
    public static void enableCheckPoc(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableCheckPocJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableCheckPoc", errorCode);
    }
    // opt_content_end: audio_or_video = video

    public static void enableAGC(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableAGCJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableAGC", errorCode);
    }

    public static void enableANS(boolean enable) {
        int errorCode = ZegoExpressEngineJni.enableANSJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableANS", errorCode);
    }

    // opt_content_start: audio_or_video = video
    public static void enableBeautify(int feature, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.enableBeautifyJni(feature, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "enableBeautify", errorCode);
    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void setBeautifyOption(ZegoBeautifyOption option, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.setBeautifyOptionJni(option, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "setBeautifyOption", errorCode);

    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_SIDE_INFO
    public static void sendSEI(byte[] data, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.sendSEIJni(data, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.PUBLISHER, "sendSEI", errorCode);
    }
    // opt_content_end: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_SIDE_INFO

    public static void sendBarrageMessage(String roomID, String message, IZegoIMSendBarrageMessageCallback handler) {
        int seq = ZegoExpressEngineJni.sendBarrageMessageJni(roomID, message);
        synchronized (ZegoExpressEngineJni.class) {
            sIMSendBarragetMssageHandler.put(seq, handler);
        }
    }


    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static ZegoMediaPlayer createMediaplayer() {
        return ZegoMediaPlayerJni.createMediaPlayer();
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    public static void setCustomVideoRenderHandler(Object handler) {
        mCustomVideoRenderHandler = handler;
    }

    public static void setCustomVideoCaptureHandler(final Object handler) {
        mCustomVideoCaptureHandler = handler;
    }

    static private boolean isCustomVideoCapturing = false;

    // opt_content_start: audio_or_video = video
    static void onCustomVideoCaptureWillStart(final int channel) {
        if (mUIHandler == null) {
            return;
        }
        mUIHandler.post(new Runnable() {
            @Override
            public void run() {
                final Object customVideoCaptureHandler = mCustomVideoCaptureHandler;
                if (customVideoCaptureHandler != null) {
                    // 经讨论，外部采集的通知不切线程
                    ZegoCallbackHelpers.callCustomVideoCaptureOnStartMethod(customVideoCaptureHandler, ZegoPublishChannel.getZegoPublishChannel(channel));
                    isCustomVideoCapturing = true;

                } else {
                    //todo: 这里需要增加日志打印
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void onCustomVideoCaptureWillStop(final int channel) {
        if (mUIHandler == null) {
            return;
        }
        mUIHandler.post(new Runnable() {
            @Override
            public void run() {
                final Object customVideoCaptureHandler = mCustomVideoCaptureHandler;
                if (customVideoCaptureHandler != null && isCustomVideoCapturing) {
                    // 经讨论，外部采集的通知不切线程
                    ZegoCallbackHelpers.callCustomVideoCaptureOnStopMethod(customVideoCaptureHandler, ZegoPublishChannel.values()[channel]);
                    isCustomVideoCapturing = false;
                } else {
                    //todo: 这里需要增加日志打印
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    public static void sendBroadcastMessage(String roomID, String message, IZegoIMSendBroadcastMessageCallback callback) {
        int seq = ZegoExpressEngineJni.sendBroadcastMessageJni(roomID, message);
        synchronized (ZegoExpressEngineJni.class) {
            sIMSendBoradcastMssageHandler.put(seq, callback);
        }
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.IM, "sendBroadcastMessage", 0);
    }

    public static void sendCustomCommand(String roomID, String command, ArrayList<ZegoUser> toUserList, IZegoIMSendCustomCommandCallback callback) {

        int size = toUserList.size();
        ZegoUser[] userList = toUserList.toArray(new ZegoUser[size]);
        int seq = ZegoExpressEngineJni.sendCustomerMessageJni(command, userList, roomID);
        synchronized (ZegoExpressEngineJni.class) {
            sIMSendCustomCommandHandler.put(seq, callback);
        }
        ZegoExpressEngineJni.printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.IM, "sendCustomCommand", 0);
    }

    public static void sendCustomVideoCaptureRawData(ByteBuffer data, int dataLength, ZegoVideoFrameParam params, long referenceTimeMillisecond, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.sendCustomVideoCaptureRawDataJni(data, dataLength, params.format.value(), params.strides, params.width, params.height, referenceTimeMillisecond, channel.value(), params.rotation);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.CUSTOM_VIDEO_IO, "sendCustomVideoCaptureRawData", errorCode);
    }

    public static void sendCustomVideoCaptureEncodedData(ByteBuffer data, int dataLength, ZegoVideoEncodedFrameParam params, long referenceTimeMillisecond, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.sendCustomVideoCaptureEncodedDataJni(data, dataLength, params.format.value(), params.isKeyFrame, params.width, params.height, params.SEIData, params.SEIDataLength, referenceTimeMillisecond, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.CUSTOM_VIDEO_IO, "sendCustomVideoCaptureEncodedData", errorCode);
    }

    public static void setCustomVideoCaptureFillMode(int value) {
        int errorCode = ZegoExpressEngineJni.setCustomVideoCaptureFillModeJni(value);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.CUSTOM_VIDEO_IO, "setCustomVideoCaptureFillMode", errorCode);
    }

    public static void sendCustomVideoCaptureTextureData(int textureID, int width, int height, double referenceTimeMillisecond, ZegoPublishChannel channel) {
        int errorCode = ZegoExpressEngineJni.sendCustomVideoCaptureTextureDataJni(textureID, width, height, referenceTimeMillisecond, channel.value());
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.CUSTOM_VIDEO_IO, "sendCustomVideoCaptureTextureData", errorCode);
    }

    public static SurfaceTexture getCustomVideoCaptureSurfaceTexture() {
        SurfaceTexture surfaceTexture = ZegoExpressEngineJni.getCustomVideoCaptureSurfaceTextureJni();
        return surfaceTexture;
    }

    public static void setBuiltInSpeakerOn(boolean enable) {
        int errorCode = ZegoExpressEngineJni.setBuiltInSpeakerOnJni(enable);
        ZegoExpressEngineJni.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                ZegoInnerModule.DEVICE, "setBuiltInSpeakerOn", errorCode);
    }


    private native static int setBuiltInSpeakerOnJni(boolean enable);

    private native static int sendBarrageMessageJni(String roomID, String message);

    private native static int sendCustomVideoCaptureTextureDataJni(int textureID, int width, int height, double referenceTimeMillisecond, int channel);

    private native static int sendCustomVideoCaptureRawDataJni(ByteBuffer data, int dataLength, int format, int[] strides, int width, int height, long referenceTimeMillisecond, int channel, int rotation);

    private native static int setCustomVideoCaptureFillModeJni(int mode);

    private native static SurfaceTexture getCustomVideoCaptureSurfaceTextureJni();

    /* zego-express-engine-jni */
    private native static int engineInitJni(long appID, String appSign, boolean isTestEnvironment, int scenario, Context context);

    private native static void setEngineInitConfigToJni(ZegoEngineConfig engineConfig,
                                                        boolean engineConfig_enableCustomVideoRender,
                                                        int engineConfig_zegoCustomVideoRenderSeries,
                                                        boolean engineConfig_isEngineRender,
                                                        int engineConfig_zegoCustomVideoRenderType,
                                                        boolean engineConfig_enableCustomVideoMainCapture,
                                                        int engineConfig_zegoCustomVideoCaptureMainType,
                                                        boolean engineConfig_enableCustomVideoAuxCapture,
                                                        int engineConfig_zegoCustomVideoCaptureAuxType,
                                                        String logPath, long logSize, String advanceConfig);

    private static native int engineUninitAsyncJni();

    private static native String getVersionJni();

    private static native void setDebugVerboseJni(boolean enable, int language);

    private static native void uploadLogJni();

    /* zego-express-room.h */
    private static native int loginRoomJni(ZegoUser user, String roomID, ZegoRoomConfig config);

    private static native int loginRoomJni(ZegoUser user, String roomID, ZegoRoomConfig config, String token);

    public static native int logoutRoomJni(String roomID);

    /* zego-express-publisher.h */

    public static native int ZegoVideoMirrorModeJni(int viewMode, int channel);

    public static native int setCapturePipelineScaleModeJni(int zegoCapturePipelineScaleMode);

    public static native int enableCheckPocJni(boolean enable);

    public static native int setAppOrientationJni(int orientation, int channel);

    public static native int startPreviewJni(Object view, int viewMode, int backgroundColor, int channel);

    public static native int stopPreviewJni(int channel);

//    public static native int setLatencyModeJni(int latencyMode);
//
//    public static native int setAudioBitrateJni(int bitrate);
//
//    public static native int setAudioChannelCountJni(int channelCount);

    public static native int setVideoConfigJni(ZegoVideoConfig videoConfig, int channel);

    public static native int startPublishingStreamJni(String streamID, int channel);

    public static native int stopPublishingStreamJni(int channel);

    public static native int mutePublishStreamAudioJni(boolean mute, int channel);

    public static native int mutePublishStreamVideoJni(boolean mute, int channel);

    public static native int setStreamExtraInfoJni(String extraInfo, int channel);

    public static native int setCaptureVolumeJni(int volume);

    public static native int enableTrafficControlJni(boolean enable, int controlTypeBitMask);

    public static native int addPublishCdnUrlJni(String streamID, String targetURL);

    public static native int enablePublishDirectToCDNJni(boolean enable, ZegoCDNConfig zegoCDNConfig, int value);

    public static native int removePublishCdnUrlJni(String streamID, String targetURL);

    public static native int setPublishWatermarkJni(ZegoWatermark watermark, boolean isPreviewVisible, int channel);

    public static native int enableHardwareEncoderJni(boolean enable);

    public static native int enableDTXJni(boolean enable);

    public static native int enableVADJni(boolean enable);

    native static int sendSEIJni(byte[] data, int channel);

    /* zego-express-player.h */

    public static native int startPlayingStreamJni(String streamID, Object view, int viewMode, int bacgroundColor, ZegoCDNConfig zegoCDNConfig, int layer);

    public static native int stopPlayingStreamJni(String streamID);

    public static native int setPlayVolumeJni(String streamID, int volume);

    public static native int mutePlayStreamAudioJni(String streamID, boolean mute);

    public static native int mutePlayStreamVideoJni(String streamID, boolean mute);

    public static native int enableHardwareDecoderJni(boolean enable);

    /* zego-express-mixer.h */
    public static native int startMixerJni(ZegoMixerTask task);

    public static native int stopMixerJni(ZegoMixerTask task);

    /* zego-express-device.h */
    public static native int muteMicrophoneJni(boolean mute);

    public static native int muteAudioOutputJni(boolean enable);

    public static native int muteSpeakerJni(boolean mute);

    public static native int enableCameraJni(boolean enable, int channel);

    public static native int useFrontCameraJni(boolean enable, int channel);

    public static native int startSoundLevelMonitorJni();

    public static native int stopSoundLevelMonitorJni();

    public static native boolean isMicrophoneMutedJni();

    public static native int startFrequencySpectrumMonitorJni();

    public static native int stopFrequencySpectrumMonitorJni();

    public static native int enableAudioCaptureDeviceJni(boolean enable);

    public static native int setMinVideoBitrateForTrafficControlJni(int bitrate, int value);

    /* zego-express-preprocess.h */
    public static native int enableAECJni(boolean enable);

    public static native int setAECModeJni(int mode);

    public static native int enableAGCJni(boolean enable);

    public static native int enableANSJni(boolean enable);

    public static native int enableVirtualStereoJni(boolean enable, int angle);

    public static native int enableReverbJni(boolean enable, int reverbMode);

    public static native int enableBeautifyJni(int feature, int channel);

    public static native int setBeautifyOptionJni(ZegoBeautifyOption option, int channel);

    public static native int setWatermarkImagePathJni(String imagePath);

    public static native boolean isSpeakerMutedJni();

    public static native int setPreviewWatermarkPositionJni(int left, int top, int right, int bottom);

    public static native int setPublishWatermarkPositionJni(int left, int top, int right, int bottom);

    /* zego-express-im.h */
    public static native int sendBroadcastMessageJni(String roomID, String message);

    public static native int sendCustomerMessageJni(String command, ZegoUser[] toUserList, String roomID);


    /* Callback */
    public static void printDebugInfo(ZegoDebugLevel level, int module, final String funcName, final int errorCode, Object... args) {

        final String msg;

        msg = ZegoExpressEngineJni.printDebugInfoJni(level.value(), module, funcName, errorCode, args);

        if (!msg.equals("")) {
            if (errorCode == 0 || !mIsTestEnv) {
                //只有在测试环境下，才弹窗提示
            } else {
                if (enableToastOnTestEnv) {
                    showToastMsg(msg, context);
                }
            }

            if (errorCode != 0 || enableDebugErrorAlways) {
                if (mUIHandler == null) {
                    //todo: 这里需要增加日志打印
                    return;
                }

                onDebugError(errorCode, funcName, msg);

            }
        }
    }

    //TODO:确认ZegoLiveRoom.java和ZegoLiveRoomJNI.java中的onLoginRoom接口处理的差别
    //TODO:这里数组不指定长度，是否会有问题？
    public static void onRoomStreamUpdate(final String roomID, final int updateType, final ZegoStream[] streamList) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ROOM, "onRoomStreamUpdate", 0);
        final ArrayList<ZegoStream> streamArrayList = new ArrayList<ZegoStream>();
        for (int i = 0; i < streamList.length; i++) {
            streamArrayList.add(streamList[i]);
        }

        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRoomStreamUpdate(roomID, ZegoUpdateType.values()[updateType], streamArrayList);
                }

            }
        });

    }

    public static ZegoAudioMixingData onAudioMixingCopyData(int dataLength) {
        final IZegoAudioMixingHandler mZegoAudioMixingHandle = iZegoAudioMixingHandler;
        if (mZegoAudioMixingHandle != null) {
            return mZegoAudioMixingHandle.onAudioMixingCopyData(dataLength);
        }
        return null;
    }

    public static void onRoomStateUpdate(final String roomID, final int state, final int errorCode, final String extendedData) {
        if (state == ZegoRoomState.DISCONNECTED.value() && errorCode != 0) {
            printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ROOM, "onRoomStreamUpdate", errorCode);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRoomStateUpdate(roomID, ZegoRoomState.values()[state], errorCode, getJsonObject(extendedData));
                }
            }
        });
    }

    public static void onStreamExtraInfoUpdate(final ZegoStream[] streamList, final int streamInfoCount, final String roomID) {
        //printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ROOM, "onStreamExtraInfoUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                synchronized (ZegoExpressEngineJni.class) {

                    //eventHandler.onStreamExtraInfoUpdate(streamList, streamInfoCount, roomID);

                }
            }
        });
    }

    public static void onRoomUserUpdate(final String roomID, final int updateType, final ZegoUser[] userList) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ROOM, "onRoomUserUpdate", 0);
        final ArrayList<ZegoUser> userArrayList = new ArrayList<ZegoUser>();
        for (int i = 0; i < userList.length; i++) {
            userArrayList.add(userList[i]);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRoomUserUpdate(roomID, ZegoUpdateType.values()[updateType], userArrayList);

                }
            }
        });

    }

    public static void onPublisherStateUpdate(final String streamID, final int state, final int errorCode, final String extendedData) {
        if (state == ZegoPublisherState.NO_PUBLISH.value() && errorCode != 0) {
            printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherStateUpdate", errorCode);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {

                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPublisherStateUpdate(streamID, ZegoPublisherState.values()[state], errorCode, getJsonObject(extendedData));

                }

            }
        });

    }

    private static JSONObject getJsonObject(String extendedData) {
        try {
            return new JSONObject(extendedData);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return new JSONObject();
    }

    public static void onPublisherUpdateCdnUrlResult(final String streamID, final int errorCode, final int seq) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoPublisherUpdateCdnUrlCallback handler = sPublisherUpdateCDNURLHandler.get(seq);
                if (handler != null) {
                    handler.onPublisherUpdateCdnUrlResult(errorCode);
                    sPublisherUpdateCDNURLHandler.remove(seq);
                }
            }
        });

    }

    // opt_content_start: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_SIDE_INFO
    public static void onPlayerRecvSEI(final String streamID, final byte[] data, final int dataLength) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPlayerRecvSEIMethodMethod(mEventHandler, streamID, data);
                }
            }
        });

    }
    // opt_content_end: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_SIDE_INFO

    public static void onPublisherUpdateStreamExtraInfoResult(final int errorCode, final int seq) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoPublisherSetStreamExtraInfoCallback handler = sPublisherUpdateStreamExtraInfoHandler.get(seq);
                if (handler != null) {
                    handler.onPublisherSetStreamExtraInfoResult(errorCode);
                    sPublisherUpdateStreamExtraInfoHandler.remove(seq);
                }
            }
        });
    }


    public static void onPublisherQualityUpdate(final String streamID, final ZegoPublishStreamQuality quality) {
        //printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherQualityUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPublisherQualityUpdate(streamID, quality);
                }
            }
        });
    }

    public static void onPublisherMediaEvent(final int media_event, final int reason, final String streamID) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherMediaEvent", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    //eventHandler.onPublisherMediaEvent(media_event, reason, streamID);

                }
            }
        });

    }

    public static void onPublisherRecvAudioFirstFrame() {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherCapturedAudioFirstFrame", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPublisherCapturedAudioFirstFrame();
                }
            }
        });

    }

    // opt_content_start: audio_or_video = video
    public static void onPublisherRecvVideoFirstFrame(final int zegoPublishChannel) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherCapturedAudioFirstFrame", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPublisherCapturedVideoFirstFrameMethod(mEventHandler, ZegoPublishChannel.values()[zegoPublishChannel]);
                }
            }
        });

    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void onPublisherVideoSizeChanged(final int width, final int height, final int channel) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PUBLISHER, "onPublisherVideoSizeChanged", 0, channel);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPublisherVideoSizeChangedMethod(mEventHandler, width, height, ZegoPublishChannel.values()[channel]);
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    public static void onPublisherRelayCDNStateUpdate(final String streamID, ZegoStreamRelayCDNInfo[] infoList) {
        final ArrayList<ZegoStreamRelayCDNInfo> infoArrayList = new ArrayList<ZegoStreamRelayCDNInfo>();
        for (int i = 0; i < infoList.length; i++) {
            infoArrayList.add(infoList[i]);
        }
        final Handler uiHandler = mUIHandler;

        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPublisherRelayCDNStateUpdate(streamID, infoArrayList);
                }
            }
        });
    }


    public static void onPlayerStateUpdate(final String streamID, final int state, final int errorCode, final String extendedData) {
        if (state == ZegoPlayerState.NO_PLAY.value() && errorCode != 0) {
            printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerStateUpdate", errorCode);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    eventHandler.onPlayerStateUpdate(streamID, ZegoPlayerState.values()[state], errorCode, getJsonObject(extendedData));
                }
            }
        });
    }

    public static void onPlayerQualityUpdate(final String streamID, final ZegoPlayStreamQuality quality) {
        //printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerQualityUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPlayerQualityUpdate(streamID, quality);
                }
            }
        });
    }

    public static void onPlayerMediaEvent(final String streamID, final int mediaEvent) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerMediaEvent", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPlayerMediaEvent(streamID, ZegoPlayerMediaEvent.values()[mediaEvent]);
                }
            }
        });
    }

    public static void onPlayerRecvAudioFirstFrame(final String streamID) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerRecvAudioFirstFrame", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onPlayerRecvAudioFirstFrame(streamID);
                }
            }
        });
    }

    // opt_content_start: audio_or_video = video
    public static void onPlayerRecvVideoFirstFrame(final String streamID) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerRecvVideoFirstFrame", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPlayerRecvVideoFirstFrameMethod(mEventHandler, streamID);
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void onPlayerRenderVideoFirstFrame(final String streamID) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerRecvVideoFirstFrame", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPlayerRenderVideoFirstFrameMethod(mEventHandler, streamID);
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    // opt_content_start: audio_or_video = video
    public static void onPlayerVideoSizeChanged(final String streamID, final int width, final int height) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.PLAYER, "onPlayerVideoSizeChanged", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnPlayerVideoSizeChangedMethod(mEventHandler, streamID, width, height);
                }
            }
        });
    }
    // opt_content_end: audio_or_video = video

    public static void onRoomStreamExtraInfoUpdate(final String roomID, ZegoStream[] streamList) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.ROOM, "onRoomStreamExtraInfoUpdate", 0);
        final ArrayList<ZegoStream> streamArrayList = new ArrayList<>();
        for (int i = 0; i < streamList.length; i++) {
            streamArrayList.add(streamList[i]);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRoomStreamExtraInfoUpdate(roomID, streamArrayList);
                }
            }
        });

    }

    public static void onIMRecvBroadcastMessage(final String roomID, ZegoBroadcastMessageInfo[] messageList) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.IM, "onIMRecvBroadcastMessage", 0);
        final ArrayList<ZegoBroadcastMessageInfo> messageArrayList = new ArrayList<>();
        for (int i = 0; i < messageList.length; i++) {
            messageArrayList.add(messageList[i]);
        }
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onIMRecvBroadcastMessage(roomID, messageArrayList);
                }
            }
        });
    }

    public static void onIMRecvCustomCommand(final String roomID, final ZegoUser fromUser, final String command) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.IM, "onIMRecvCustomCommand", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onIMRecvCustomCommand(roomID, fromUser, command);
                }
            }
        });

    }

    public static void onIMSendBroadcastMessageResult(final String roomID, final int errorCode, final int seq, final long messageID) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                synchronized (ZegoExpressEngineJni.class) {
                    final IZegoIMSendBroadcastMessageCallback callback = sIMSendBoradcastMssageHandler.get(seq);
                    if (callback != null) {
                        callback.onIMSendBroadcastMessageResult(errorCode, messageID);
                    }
                    sIMSendBoradcastMssageHandler.remove(seq);
                }
            }
        });
    }


    public static void onIMRecvBarrageMessage(final String roomID, ZegoBarrageMessageInfo[] messageList) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.IM, "onIMRecvBarrageMessage", 0);
        final ArrayList<ZegoBarrageMessageInfo> messageArrayList = new ArrayList<>();
        messageArrayList.addAll(Arrays.asList(messageList));
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onIMRecvBarrageMessage(roomID, messageArrayList);
                }
            }
        });
    }

    public static void onIMSendBarrageMessageResult(final int seq, final int errorCode, final String messageID) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {

                final IZegoIMSendBarrageMessageCallback callback = sIMSendBarragetMssageHandler.get(seq);
                if (callback != null) {
                    callback.onIMSendBarrageMessageResult(errorCode, messageID);
                }
                sIMSendBarragetMssageHandler.remove(seq);

            }
        });
    }

    public static void onIMSendCustomCommandResult(final String roomID, final int errorCode, final int seq) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoIMSendCustomCommandCallback callback = sIMSendCustomCommandHandler.get(seq);
                if (callback != null) {
                    callback.onIMSendCustomCommandResult(errorCode);
                }
                sIMSendCustomCommandHandler.remove(seq);
            }
        });
    }

    public static void onRemoteCameraStateUpdate(final String streamID, final int state) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    ZegoCallbackHelpers.callOnRemoteCameraStateUpdateMethod(mEventHandler, streamID, ZegoRemoteDeviceState.values()[state]);

                }
            }
        });
    }

    public static void onRemoteMICStateUpdate(final String streamID, final int state) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    eventHandler.onRemoteMicStateUpdate(streamID, ZegoRemoteDeviceState.values()[state]);
                }
            }
        });
    }


    public static void onEngineStateUpdate(final int zegoEngineState) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    eventHandler.onEngineStateUpdate(ZegoEngineState.getZegoEngineState(zegoEngineState));
                }
            }
        });
    }


    public static void onEngineUninitUpdate() {
        final IZegoDestroyCompletionCallback mIZegoDestroyCompletionCallback = iZegoDestroyCompletionCallback;
        if (mIZegoDestroyCompletionCallback != null) {
            mIZegoDestroyCompletionCallback.onDestroyCompletion();
            iZegoDestroyCompletionCallback = null;
        }
    }

    public static void onDeviceError(final int errorCode, final String deviceName) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.DEVICE, "onDeviceError", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onDeviceError(errorCode, deviceName);
                }
            }
        });
    }

    public static void onPlayerSoundLevelUpdate(final HashMap<String, Float> soundLevelHashMap) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.DEVICE, "onPlayerSoundLevelUpdate", 0);

        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRemoteSoundLevelUpdate(soundLevelHashMap);
                }
            }
        });
    }

    public static void onCapturedSoundLevelUpdate(final float soundLevel) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.DEVICE, "onCapturedSoundLevelUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onCapturedSoundLevelUpdate(soundLevel);
                }
            }
        });
    }

    public static void onCapturedFrequencySpectrumUpdate(final float[] frequencySpectrum) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.DEVICE, "onCapturedFrequencySpectrumUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onCapturedAudioSpectrumUpdate(frequencySpectrum);
                }
            }
        });

    }

    public static void onPlayerFrequencySpectrumUpdate(final HashMap<String, float[]> frequencySpectrums) {
        // printDebugInfo(ZegoDebugLevel.INFO, ZegoInnerModule.DEVICE, "onPlayerFrequencySpectrumUpdate", 0);
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRemoteAudioSpectrumUpdate(frequencySpectrums);
                }
            }
        });

    }

    public static void onMixerStartResult(final int seq, final int errorCode, final String extendedData) {
        //todo 混流不走通用事件通知
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoMixerStartCallback callback = sMixerStartResultHandler.get(seq);
                if (callback != null) {
                    callback.onMixerStartResult(errorCode, getJsonObject(extendedData));
                }
                sMixerStartResultHandler.remove(seq);
            }
        });
    }

    public static void onMixerStopResult(final int seq, final int errorCode) {
        //todo 混流不走通用事件通知
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoMixerStopCallback callback = sMixerStopResultHandler.get(seq);
                if (callback != null) {
                    callback.onMixerStopResult(errorCode);
                }
                sMixerStartResultHandler.remove(seq);

            }
        });

    }

    public static void onRoomOnlineUserCountUpdate(final String roomID, final int count) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onRoomOnlineUserCountUpdate(roomID, count);
                }
            }
        });
    }


    public static void onMixerRelayCDNStateUpdate(final ZegoStreamRelayCDNInfo[] infoList, final String taskID) {
        final ArrayList<ZegoStreamRelayCDNInfo> infoArrayList = new ArrayList<ZegoStreamRelayCDNInfo>();
        for (int i = 0; i < infoList.length; i++) {
            infoArrayList.add(infoList[i]);
        }

        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onMixerRelayCDNStateUpdate(taskID, infoArrayList);
                }
            }
        });
    }

    // opt_content_start: audio_or_video = video ; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_EXTERNAL_VIDEO_RENDER
    public static void onCustomVideoRenderCapturedFrameData(ByteBuffer[] buffers, int[] dataLen, int[] strides, int width, int height, int videoPixelFormat, int videoFlipMode, int channel) {
        final Object iZegoCustomVideoRenderHandler = mCustomVideoRenderHandler;
        if (iZegoCustomVideoRenderHandler != null) {
            ZegoVideoFrameParam zegoVideoFrameParam = new ZegoVideoFrameParam();
            for (int i = 0; i < strides.length; i++) {
                zegoVideoFrameParam.strides[i] = strides[i];
            }
            zegoVideoFrameParam.format = ZegoVideoFrameFormat.values()[videoPixelFormat];
            zegoVideoFrameParam.width = width;
            zegoVideoFrameParam.height = height;

            ZegoCallbackHelpers.callCustomVideoRenderOnCapturedVideoFrameRawDataMethod(iZegoCustomVideoRenderHandler, buffers, dataLen, zegoVideoFrameParam, ZegoVideoFlipMode.values()[videoFlipMode], ZegoPublishChannel.values()[channel]);
        }
    }
    // opt_content_end: audio_or_video = video ; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_EXTERNAL_VIDEO_RENDER

    // opt_content_start: audio_or_video = video ; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_EXTERNAL_VIDEO_RENDER
    public static void onCustomVideoRenderRemoteFrameData(ByteBuffer[] buffers, int[] dataLen, int[] strides, int width, int height, int videoPixelFormat, String streamID) {
        final Object iZegoCustomVideoRenderHandler = mCustomVideoRenderHandler;
        if (iZegoCustomVideoRenderHandler != null) {
            ZegoVideoFrameParam zegoVideoFrameParam = new ZegoVideoFrameParam();
            for (int i = 0; i < strides.length; i++) {
                zegoVideoFrameParam.strides[i] = strides[i];
            }
            zegoVideoFrameParam.format = ZegoVideoFrameFormat.values()[videoPixelFormat];
            zegoVideoFrameParam.width = width;
            zegoVideoFrameParam.height = height;
            ZegoCallbackHelpers.callCustomVideoRenderOnCustomVideoRenderRemoteFrameDataMethod(iZegoCustomVideoRenderHandler, buffers, dataLen, zegoVideoFrameParam, streamID);
        }
    }
    // opt_content_end: audio_or_video = video ; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_EXTERNAL_VIDEO_RENDER


    public static void showToastMsg(final String msg, final Context context) {
        //todo 是否有可能线程不安全
        if (context != null) {
            try {
                if (Thread.currentThread() == Looper.getMainLooper().getThread()) {
                    Toast.makeText(context, msg, Toast.LENGTH_LONG).show();
                } else {
                    final Handler uiHandler = mUIHandler;
                    uiHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            Toast.makeText(context, msg, Toast.LENGTH_LONG).show();
                        }
                    });
                }
            } catch (Exception e) {
                //do nothing
            }
        }
    }

    public static void onDebugError(final int errorCode, final String funcName, final String info) {
        final Handler uiHandler = mUIHandler;
        if (uiHandler == null) {
            //todo: 这里需要增加日志打印
            return;
        }

        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                final IZegoEventHandler mEventHandler = eventHandler;
                if (mEventHandler != null) {
                    mEventHandler.onDebugError(errorCode, funcName, info);
                }
            }
        });
    }

    public static native ZegoAudioConfig getAudioConfigJni();

    public static native ZegoVideoConfig getVideoConfigJni(int value);

    public static native String printDebugInfoJni(int level, int module, String funcName, int errorCode, Object... args);

    public static native void logInfoJni(String fileName, String funcName, int line, int level, int layer, int module, String format, Object... args);

    public static native int setTestCase(int group, int id, String desc, boolean enableStub);

    public static native int triggerCallback();

    public static native int sendCustomVideoCaptureEncodedDataJni(ByteBuffer data, int dataLength, int value, boolean isKeyFrame, int width, int height, ByteBuffer seiData, int seiDataLength, long referenceTimeMillisecond, int value1);

    public static native int setHeadphoneMonitorVolumeJni(int volume);

    public static native int enableHeadphoneMonitorJni(boolean enable);

    public static native int enableCustomVideoCaptureJni(boolean enable, ZegoCustomVideoCaptureConfig config, int value);

    public static native int enableCustomVideoRenderJni(boolean enable, ZegoCustomVideoRenderConfig config);

    public static native int setAudioConfigJni(int bitrate, int channel, int codecID);

    public static native int enableAudioMixingJni(boolean enable);

    public static native int muteLocalAudioMixingJni(boolean mute);

    public static native int setAudioMixingVolumeJniWithType(int volume, int type);

    public static native int setAudioMixingVolumeJni(int volume);
}
