package com.zego.zegoavkit2.receiver;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothHeadset;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.hardware.usb.UsbConfiguration;
import android.hardware.usb.UsbConstants;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbInterface;
import android.hardware.usb.UsbManager;
import android.media.AudioManager;
import android.os.Handler;
import android.os.Looper;
import android.os.Bundle;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

public final class AudioRouteMonitor extends BroadcastReceiver {
    private long mThis;
    private Context mContext;
    private int mBluetoothOpSeq;
    private AtomicBoolean mIsInited = new AtomicBoolean(false);

    private final int AUDIO_ROUTE_SPEAKER = 0;
    private final int AUDIO_ROUTE_HEADSET = 1;
    private final int AUDIO_ROUTE_BLUETOOTH = 2;
    private final int AUDIO_ROUTE_RECEIVER = 3;
    private final int AUDIO_ROUTE_USB_AUDIO = 4;

    public void setThis(long pThis) {
        mThis = pThis;
    }

    public int init(Context ctx) {
        mContext = ctx;
        if (mContext == null) {
            return -1;
        }

        IntentFilter intentHeadSetPlug = new IntentFilter();
        intentHeadSetPlug.addAction("android.intent.action.HEADSET_PLUG"); // * AudioManager.ACTION_HEADSET_PLUG
        intentHeadSetPlug.addAction(BluetoothAdapter.ACTION_STATE_CHANGED);
        intentHeadSetPlug.addAction(BluetoothHeadset.ACTION_CONNECTION_STATE_CHANGED);
        intentHeadSetPlug.addAction("android.hardware.usb.action.USB_DEVICE_ATTACHED");
        intentHeadSetPlug.addAction("android.hardware.usb.action.USB_DEVICE_DETACHED");
        mContext.registerReceiver(this, intentHeadSetPlug);

        final AudioManager audioManager = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
        final UsbManager usbManager = (UsbManager) mContext.getSystemService(Context.USB_SERVICE);

        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                mBluetoothOpSeq = 0;
                mIsInited.set(true);

                boolean headsetConnected = false;
                boolean bluetoothConnected = false;
                boolean usbAudioConnected = false;

                headsetConnected = audioManager.isWiredHeadsetOn();

                // * init state
                try {
                    int state = BluetoothAdapter.getDefaultAdapter().getProfileConnectionState(android.bluetooth.BluetoothProfile.HEADSET);
                    bluetoothConnected = state == BluetoothProfile.STATE_CONNECTED;
                } catch (Exception e) {
                    e.printStackTrace();
                }

                HashMap<String, UsbDevice> deviceHashMap = usbManager.getDeviceList();
                for (Map.Entry entry : deviceHashMap.entrySet()) {
                    UsbDevice device = (UsbDevice) entry.getValue();
                    if (null == device) continue;

                    for (int i = 0; !usbAudioConnected && i < device.getConfigurationCount(); i++) {
                        UsbConfiguration configuration = device.getConfiguration(i);
                        if (null == configuration) continue;

                        for (int j = 0; j < configuration.getInterfaceCount(); j++) {
                            UsbInterface usbInterface = configuration.getInterface(j);
                            if (null == usbInterface) continue;

                            if (UsbConstants.USB_CLASS_AUDIO == usbInterface.getInterfaceClass()) {
                                usbAudioConnected = true;
                                break;
                            }
                        }
                    }
                }

                onDeviceStateInited(mThis, headsetConnected, bluetoothConnected, usbAudioConnected);
            }
        });

        return 0;
    }

    public int uninit() {
        mIsInited.set(false);
        if (mContext == null) {
            return -1;
        }

        mContext.unregisterReceiver(this);
        mContext = null;
        return 0;
    }

    @Override
    public void onReceive(Context context, Intent intent) {

        if (!mIsInited.get()) {
            return;
        }

        String action = intent.getAction();
        Bundle extras = intent.getExtras();
        final String strInent = "action: " + action + ((extras != null && extras.size() > 0) ? ", " + extras.toString() : "");
        
        if ("android.intent.action.HEADSET_PLUG".equals(action)) {

            if (intent.hasExtra("state")) {
                // * wired head set: 0 - no headset; 1 - has headset
                int state = intent.getIntExtra("state", 0);
                onDeviceStateChanged(mThis, AUDIO_ROUTE_HEADSET, state == 1 ? true : false, strInent);
            }

        } else if (BluetoothAdapter.ACTION_STATE_CHANGED.equals(action)) {

            int state = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, BluetoothAdapter.ERROR);
            if (state ==  BluetoothAdapter.STATE_OFF) {
                // * bluetooth removed
                mBluetoothOpSeq += 1;
                onDeviceStateChanged(mThis, AUDIO_ROUTE_BLUETOOTH, false, strInent);
            }

        } else if (BluetoothHeadset.ACTION_CONNECTION_STATE_CHANGED.equals(action)) {

            int state = intent.getIntExtra(BluetoothHeadset.EXTRA_STATE, BluetoothAdapter.ERROR);
            if (state == BluetoothHeadset.STATE_CONNECTED) {
                mBluetoothOpSeq += 1;
                final int currentOpSeq = mBluetoothOpSeq;
                new Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (!mIsInited.get()) {
                            return;
                        }
                        // * bluetooth added
                        if (mBluetoothOpSeq == currentOpSeq) {
                            onDeviceStateChanged(mThis, AUDIO_ROUTE_BLUETOOTH, true, strInent);
                        }
                    }
                }, 1500);
            } else if (state == BluetoothHeadset.STATE_DISCONNECTED) {
                // * bluetooth removed
                mBluetoothOpSeq += 1;
                onDeviceStateChanged(mThis, AUDIO_ROUTE_BLUETOOTH, false, strInent);
            }
        }
        else if ("android.hardware.usb.action.USB_DEVICE_ATTACHED".equals(action)) {
            UsbDevice device = (UsbDevice) intent.getParcelableExtra(UsbManager.EXTRA_DEVICE);
            if (null == device) {
                return;
            }

            boolean hasAudio = false;
            int count = device.getConfigurationCount();
            for (int i = 0; !hasAudio && i < count; i++) {
                UsbConfiguration configuration = device.getConfiguration(i);
                if (null == configuration) continue;

                int interfaceCount = configuration.getInterfaceCount();
                for (int j = 0; j < interfaceCount; j++) {
                    UsbInterface usbInterface = configuration.getInterface(j);
                    if (null != usbInterface && usbInterface.getInterfaceClass() == UsbConstants.USB_CLASS_AUDIO) {
                        hasAudio = true;
                        break;
                    }
                }
            }

            if (hasAudio) {
                onDeviceStateChanged(mThis, AUDIO_ROUTE_USB_AUDIO, true, strInent);
            }
        }
        else if ("android.hardware.usb.action.USB_DEVICE_DETACHED".equals(action)) {
            UsbDevice device = (UsbDevice) intent.getParcelableExtra(UsbManager.EXTRA_DEVICE);
            if (null == device) {
                return;
            }

            boolean hasAudio = false;
            int count = device.getConfigurationCount();
            for (int i = 0; !hasAudio && i < count; i++) {
                UsbConfiguration configuration = device.getConfiguration(i);
                if (null == configuration) continue;

                int interfaceCount = configuration.getInterfaceCount();
                for (int j = 0; j < interfaceCount; j++) {
                    UsbInterface usbInterface = configuration.getInterface(j);
                    if (null != usbInterface && usbInterface.getInterfaceClass() == UsbConstants.USB_CLASS_AUDIO) {
                        hasAudio = true;
                        break;
                    }
                }
            }

            if (hasAudio) {
                onDeviceStateChanged(mThis, AUDIO_ROUTE_USB_AUDIO, false, strInent);
            }
        }
    }

    static native void onDeviceStateInited(long pThis, boolean headsetConnected, boolean bluetoothConnected, boolean usbAudioConnected);
    static native void onDeviceStateChanged(long pThis, int device, boolean connected, String intent);
}
