package im.zego.zegoexpress.utils;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.Log;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * 加载非系统目录下的 .so 文件，根据CPU架构自动搜索最合适的 .so <br>
 * 同时负责维护 .so 的释放、更新。
 *
 * <p>Copyright © 2017 Zego. All rights reserved.</p>
 *
 * @author realuei on 2017/7/21.
 */

public final class ZegoLibraryLoadUtil {

    /**
     * 加载指定 so 文件。
     *
     * <p>需要确保指定 so 文件可被应用加载</p>
     *
     * @param customizeSoPath .so 文件绝对路径
     * @param context 应用上下文
     * @return true: 加载成功；false: 加载失败
     *
     * @throws Exception
     */
    static public boolean loadSpecialLibrary(String customizeSoPath, Context context) throws Exception {
        File soFile = new File(customizeSoPath);
        if (!soFile.exists()) {
            return false;
        }

        try {
            System.load(customizeSoPath);
        } catch (UnsatisfiedLinkError e) {
            // 加载失败，拷贝到 libs 目录看看能否正常加载
            File soPath = getCustomizeLibDir(context, "ext");
            File targetSo = new File(soPath, soFile.getName());
            if (!targetSo.exists() || targetSo.length() != soFile.length()) {
                copyFile(soFile, targetSo);
            }
            if (targetSo.exists()) {
                System.load(targetSo.getAbsolutePath());
            }
        }
        return true;
    }

    /**
     * 加载指定的库文件，如果当前文件不存在，则自动从 .apk 中释放最适合当前平台的库文件；<br>
     * 若当前文件不是最新的，则重新从 .apk 文件中释放。
     *
     * @param fileName 不带路径 .so 文件名
     * @param context 应用上下文
     *
     * @return true: 成功加载；false: 加载失败
     */
    static public boolean loadSoFile(String fileName, Context context) {
        File soPath = getCustomizeLibDir(context, "apk");
        int currentVersionCode = getVersionCode(context);
        File versionFile = new File(soPath, versionFileName(currentVersionCode));

        File targetSoFile = new File(soPath, fileName);
        int errorCode = 0;
        if (!versionFile.exists() || !targetSoFile.exists()) {
            errorCode = unzipSo(context, fileName, targetSoFile, versionFile);
        }

        if (errorCode == 0 && versionFile.exists() && targetSoFile.exists()) {
            System.load(targetSoFile.getPath());
            return true;
        } else {
            Log.e("ZegoSoLoadUtil", "unzip " + fileName + " from apk failed, errorCode: " + errorCode);
        }
        return false;
    }

    static private File getCustomizeLibDir(Context context, String subDirName) {
        File rootDir = context.getDir("libs", Context.MODE_PRIVATE);
        return new File(rootDir, subDirName);
    }

    static private int getVersionCode(Context context) {
        PackageManager pm = context.getPackageManager();
        String pkgName = context.getPackageName();
        try {
            return pm.getPackageInfo(pkgName, PackageManager.GET_CONFIGURATIONS).versionCode;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 解压最适合当前CPU架构的 so 库。
     *
     * @param context Context 上下文
     * @param libName so 库文件名
     * @param targetPath 解压目标目录
     * @param versionFile 版本号文件
     * @return  0: success;
     *         -1: create version file failed;
     *         -2: decompress failed;
     *         -3: not found support so file
     */
    static private int unzipSo(Context context, String libName, File targetFile, File versionFile) {
        String codePath = context.getPackageCodePath();
        String[] supportAbis = getSupportABIs();

        for (String abiType : supportAbis) {
            try {
                boolean success = unzipSpecialABISo(codePath, libName, abiType, targetFile);
                if (success) {
                    success = versionFile.createNewFile();
                    return success ? 0 : -1;
                }
            } catch (Exception e) {
                e.printStackTrace();
                return -2;
            }
        }
        return -3;
    }

    static final private int BUFFER_SIZE = 1024 * 64;;
    static private boolean unzipSpecialABISo(String apkPath, String libName, String ABIType, File targetFile) throws IOException {
        ZipInputStream zis = null;
        try {
            boolean found = false;
            ZipEntry entry;
            zis = new ZipInputStream(new BufferedInputStream(new FileInputStream(apkPath)));
            while ((entry = zis.getNextEntry()) != null) {
                String entryName = entry.getName();
                if (entryName.contains("../")) {
                    throw new SecurityException("unsecurity zip file!");
                }
                if (entryName.startsWith("lib/") && entryName.startsWith(ABIType, 4) && entryName.endsWith(libName)) {
                    found = true;
                    break;
                }
            }

            if (found) {
                byte[] buffer = new byte[BUFFER_SIZE];
                File parentDir = targetFile.getParentFile();
                if (!parentDir.exists()) {
                    parentDir.mkdirs();
                }

                BufferedOutputStream targetStream = null;
                try {
                    int count;
                    targetStream = new BufferedOutputStream(new FileOutputStream(targetFile), BUFFER_SIZE);
                    while ((count = zis.read(buffer, 0, BUFFER_SIZE)) != -1) {
                        targetStream.write(buffer, 0, count);
                    }
                    targetStream.flush();
                    return true;
                } finally {
                    if (targetStream != null) {
                        targetStream.close();
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (zis != null) {
                zis.close();
            }
        }
        return false;
    }

    static private boolean copyFile(File srcFile, File targetFile) throws IOException {
        File parentDir = targetFile.getParentFile();
        if (!parentDir.exists()) {
            parentDir.mkdirs();
        }

        BufferedInputStream bis = null;
        BufferedOutputStream bos = null;
        try {
            bis = new BufferedInputStream(new FileInputStream(srcFile));
            bos = new BufferedOutputStream(new FileOutputStream(targetFile, false));
            byte[] buffer = new byte[BUFFER_SIZE];
            int readSize;
            int loopCount = 0;
            while ((readSize = bis.read(buffer, 0, BUFFER_SIZE)) != -1) {
                bos.write(buffer, 0, readSize);
                loopCount ++;
                if (loopCount % 10 == 0) {
                    bos.flush();
                }
            }
            bos.flush();
        } finally {
            if (bis != null) {
                bis.close();
            }
            if (bos != null) {
                bos.close();
            }
        }
        return true;
    }

    static final private String VERSION_FILE_NAME_TEMPLATE = "cur_ver_%d.txt";
    static private String versionFileName(int versionCode) {
        return String.format(VERSION_FILE_NAME_TEMPLATE, versionCode);
    }

    static private String[] getSupportABIs() {
        String[] supportedAbis;
        if (Build.VERSION.SDK_INT >= 21) {
            supportedAbis = Build.SUPPORTED_ABIS;
        } else {
            supportedAbis = new String[] { Build.CPU_ABI, Build.CPU_ABI2 };
        }

        return supportedAbis;
    }
}
