package im.zego.zegoexpress.module.mediaplayer;

import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;
import im.zego.zegoexpress.constants.ZegoVideoFrameFormat;
import im.zego.zegoexpress.entity.ZegoAudioFrameParam;
import im.zego.zegoexpress.entity.ZegoCanvas;
import im.zego.zegoexpress.entity.ZegoVideoFrameParam;
import im.zego.zegoexpress.module.mediaplayer.callback.IZegoMediaplayerAudioHandler;
import im.zego.zegoexpress.module.mediaplayer.callback.IZegoMediaplayerEventHandler;
import im.zego.zegoexpress.module.mediaplayer.callback.IZegoMediaplayerVideoHandler;
import im.zego.zegoexpress.module.mediaplayer.callback.ZegoMediaplayerLoadResourceCallback;
import im.zego.zegoexpress.module.mediaplayer.callback.ZegoMediaplayerSeekToCallback;
import im.zego.zegoexpress.module.mediaplayer.entity.ZegoMediaplayerNetworkEvent;
import im.zego.zegoexpress.module.mediaplayer.entity.ZegoMediaplayerState;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public final class ZegoMediaplayerJni {

    private static Handler mUIHandler = new Handler(Looper.getMainLooper());


    /* 在 ZegoMediaplayerJni 里同时管理对于JNI的调用回调与 Mediaplayer 多实例的管理*/

    static class IdxAndHandler {
        int kMediaplayerIdx;
        HashMap<Integer, ZegoMediaplayerSeekToCallback> seekToTimeCallbackHashMap;
        ZegoMediaplayerLoadResourceCallback loadResourceCallback;
        IZegoMediaplayerEventHandler eventHandler;
        IZegoMediaplayerAudioHandler audioHandler;
        IZegoMediaplayerVideoHandler videoHandler;
        IdxAndHandler(int idx){
            this.kMediaplayerIdx = idx;
        }
    }

    private static HashMap<ZegoMediaplayer, IdxAndHandler> mediaplayerToIdxAndEventhandler = new HashMap<>();

    static void setEventHandler(@NonNull ZegoMediaplayer mediaplayer, IZegoMediaplayerEventHandler handler){
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null){
            idxAndHandler.eventHandler = handler;
        }
    }

    static void onMediaPlayerStateUpdateCallback(int state, final int errorcode, int idx){

        final ZegoMediaplayerState zegoMediaplayerState = ZegoMediaplayerState.values()[state];
        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().eventHandler == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().eventHandler.onMediaPlayerStateUpdate(item.getKey(), zegoMediaplayerState, errorcode);
                            }
                        });
                    }
                }
            }
        }
    }

    static void onMediaPlayerNetWorkEventCallback(int event, int idx){

        final ZegoMediaplayerNetworkEvent zegoMediaplayerNetworkEvent = ZegoMediaplayerNetworkEvent.values()[event];

        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().eventHandler == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().eventHandler.onMediaPlayerNetworkEvent(item.getKey(), zegoMediaplayerNetworkEvent);
                            }
                        });
                    }
                }
            }
        }
    }

    static void onMediaPlayerPlayingProgressCallback(int idx, final long millisecond){
        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().eventHandler == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().eventHandler.onMediaPlayerPlayingProgress(item.getKey(), millisecond);
                            }
                        });
                    }
                }
            }
        }
    }


    static void setVideoHandler(ZegoMediaplayer mediaplayer, ZegoVideoFrameFormat frameFormat, IZegoMediaplayerVideoHandler handler){
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null){
            idxAndHandler.videoHandler = handler;
            if(handler == null){
                ZegoMediaplayerJni.enableVideoDataJni(frameFormat.value(), false, idxAndHandler.kMediaplayerIdx);
            }else {
                ZegoMediaplayerJni.enableVideoDataJni(frameFormat.value(), true, idxAndHandler.kMediaplayerIdx);
            }
        }
    }

    static void onMediaPlayerVideoHandlerCallback(int idx,final ByteBuffer[] buffer,final int[] datalength, int[] strides, int format, int width, int height){
        final ZegoVideoFrameParam videoFrameParam = new ZegoVideoFrameParam();
        videoFrameParam.height = height;
        videoFrameParam.strides = strides;
        videoFrameParam.width = width;

        videoFrameParam.format = ZegoVideoFrameFormat.values()[format];

        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().videoHandler == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().videoHandler.onVideoFrame(item.getKey(), buffer, datalength, videoFrameParam);
                            }
                        });
                    }
                }
            }
        }
    }

    static void setAudioHandler(ZegoMediaplayer mediaplayer, IZegoMediaplayerAudioHandler handler) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            idxAndHandler.audioHandler = handler;
            if (handler == null) {
                ZegoMediaplayerJni.enableAudioDataJni(false, idxAndHandler.kMediaplayerIdx);
            } else {
                ZegoMediaplayerJni.enableAudioDataJni(true, idxAndHandler.kMediaplayerIdx);
            }
        }
    }

    static void onMediaPlayerAudioHandlerCallback(int idx, final ByteBuffer buffer, final int bufferLength, int channels, int samplesRate){

        final ZegoAudioFrameParam audioFrame = new ZegoAudioFrameParam();
        audioFrame.channels = channels;
        audioFrame.sampleRate = samplesRate;

        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().audioHandler == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().audioHandler.onAudioFrame(item.getKey(), buffer, bufferLength, audioFrame);
                            }
                        });
                    }
                }
            }
        }
    }

    static void loadResource(ZegoMediaplayer mediaplayer, String path, ZegoMediaplayerLoadResourceCallback callback) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.loadResourceJni(idxAndHandler.kMediaplayerIdx, path);
            idxAndHandler.loadResourceCallback = callback;
        }
    }

    static void onLoadResourceCallback(int idx, final int errorcode){
        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    if(item.getValue().loadResourceCallback == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                item.getValue().loadResourceCallback.onLoadResourceCallback(errorcode);
                            }
                        });
                    }
                }
            }
        }
    }

    static void start(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.startJni(idxAndHandler.kMediaplayerIdx);
        }
    }

    static void stop(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.stopJni(idxAndHandler.kMediaplayerIdx);
        }
    }

    static void pause(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.pauseJni(idxAndHandler.kMediaplayerIdx);
        }
    }

    static void resume(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.resumeJni(idxAndHandler.kMediaplayerIdx);
        }
    }

    static void seekTo(ZegoMediaplayer mediaplayer, long millisecond, ZegoMediaplayerSeekToCallback callback) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            int seq = ZegoMediaplayerJni.seekToJni(idxAndHandler.kMediaplayerIdx, millisecond);
            synchronized (ZegoMediaplayerJni.class) {
                idxAndHandler.seekToTimeCallbackHashMap.put(seq, callback);
            }
        }
    }

    static void onSeekToTimeCallback(final int seq, int idx, final int errorcode){
        synchronized (ZegoMediaplayerJni.class){
            for(final Map.Entry<ZegoMediaplayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()){
                if(item.getValue().kMediaplayerIdx == idx){
                    final ZegoMediaplayerSeekToCallback seekToTimeCallback = item.getValue().seekToTimeCallbackHashMap.get(seq);
                    if(seekToTimeCallback == null){
                        //todo: 这里需要增加日志打印
                        return;
                    }else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                seekToTimeCallback.onSeekToTimeCallback(errorcode);
                                item.getValue().seekToTimeCallbackHashMap.remove(seq);
                            }
                        });
                    }
                }
            }
        }
    }

    static void enableRepeat(ZegoMediaplayer mediaplayer, boolean enable) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.enableRepeatJni(idxAndHandler.kMediaplayerIdx, enable);
        }
    }

    public static void enableAux(ZegoMediaplayer mediaplayer, boolean enable) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.enableAuxJni(idxAndHandler.kMediaplayerIdx, enable);
        }
    }

    public static void muteLocal(ZegoMediaplayer mediaplayer, boolean mute) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.muteLocalJni(idxAndHandler.kMediaplayerIdx, mute);
        }
    }

    static void setPlayerCanvas(ZegoMediaplayer mediaplayer, ZegoCanvas canvas) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            if(canvas != null){
                ZegoMediaplayerJni.setPlayerCanvasJni(idxAndHandler.kMediaplayerIdx, canvas.view, canvas.viewMode.value(), canvas.backgroundColor);
            }else {
                ZegoMediaplayerJni.setPlayerCanvasJni(idxAndHandler.kMediaplayerIdx, null, 0, 0);

            }
        }
    }

    static void setVolume(ZegoMediaplayer mediaplayer, int volume) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.setVolumeJni(idxAndHandler.kMediaplayerIdx, volume);
        }
    }

    static int getVolume(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            return ZegoMediaplayerJni.getVolumeJni(idxAndHandler.kMediaplayerIdx);
        }
        return -1;
    }

    static void setProgressInterval(ZegoMediaplayer mediaplayer, long millisecond) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            ZegoMediaplayerJni.setProgressIntervalJni(idxAndHandler.kMediaplayerIdx, millisecond);
        }
    }

    static long getTotalDuration(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            return ZegoMediaplayerJni.getTotalDurationJni(idxAndHandler.kMediaplayerIdx);
        }
        return -1;
    }

    static long getCurrentProgress(ZegoMediaplayer mediaplayer) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            return ZegoMediaplayerJni.getCurrentProgressJni(idxAndHandler.kMediaplayerIdx);
        }
        return -1;
    }

    public static ZegoMediaplayer createMediaplayer() {
        int index = ZegoMediaplayerJni.createMediaplayerJni();
        if(index >= 0){
            ZegoMediaplayer zegoMediaplayer = ZegoMediaplayer.createMediaplayer();
            IdxAndHandler idxAndHandler = new IdxAndHandler(index);
            idxAndHandler.seekToTimeCallbackHashMap = new HashMap<>();
            mediaplayerToIdxAndEventhandler.put(zegoMediaplayer, idxAndHandler);
            return zegoMediaplayer;
        }else {
            return null;
        }
    }

    static void destroyMediaplayer(ZegoMediaplayer mediaplayer) {
        synchronized (ZegoMediaplayerJni.class){
            for (Iterator<Map.Entry<ZegoMediaplayer, IdxAndHandler>> it = mediaplayerToIdxAndEventhandler.entrySet().iterator(); it.hasNext();){
                Map.Entry<ZegoMediaplayer, IdxAndHandler> item = it.next();
                if(item.getKey() == mediaplayer){
                    item.getValue().videoHandler = null;
                    item.getValue().audioHandler = null;
                    item.getValue().eventHandler = null;
                    item.getValue().loadResourceCallback = null;
                    Iterator<Map.Entry<Integer, ZegoMediaplayerSeekToCallback>> itSeekToTimeCallbackHashMap =  item.getValue().seekToTimeCallbackHashMap.entrySet().iterator();
                    for(; itSeekToTimeCallbackHashMap.hasNext();){
                        itSeekToTimeCallbackHashMap.remove();
                    }
                    item.getValue().seekToTimeCallbackHashMap = null;
                    int idx = item.getValue().kMediaplayerIdx;
                    ZegoMediaplayerJni.destroyMediaplayerJni(idx);
                    it.remove();
                }
            }
        }
    }

    public static void destroyAllMediaplayer() {
        synchronized (ZegoMediaplayerJni.class){
            for (Iterator<Map.Entry<ZegoMediaplayer, IdxAndHandler>> it = mediaplayerToIdxAndEventhandler.entrySet().iterator(); it.hasNext();){
                Map.Entry<ZegoMediaplayer, IdxAndHandler> item = it.next();
                item.getValue().videoHandler = null;
                item.getValue().audioHandler = null;
                item.getValue().eventHandler = null;
                item.getValue().loadResourceCallback = null;
                Iterator<Map.Entry<Integer, ZegoMediaplayerSeekToCallback>> itSeekToTimeCallbackHashMap =  item.getValue().seekToTimeCallbackHashMap.entrySet().iterator();
                for(; itSeekToTimeCallbackHashMap.hasNext();){
                    itSeekToTimeCallbackHashMap.remove();
                }
                item.getValue().seekToTimeCallbackHashMap = null;
                int idx = item.getValue().kMediaplayerIdx;
                ZegoMediaplayerJni.destroyMediaplayerJni(idx);
                it.remove();
            }
        }
    }

    static ZegoMediaplayerState getCurrentState(ZegoMediaplayer mediaplayer) {

        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if(idxAndHandler != null) {
            int state = ZegoMediaplayerJni.getCurrentStateJni(idxAndHandler.kMediaplayerIdx);
            final ZegoMediaplayerState zegoMediaplayerState;
            if(state == ZegoMediaplayerState.NO_PLAY.value()){
                zegoMediaplayerState = ZegoMediaplayerState.NO_PLAY;
            } else if (state == ZegoMediaplayerState.PLAYING.value()) {
                zegoMediaplayerState = ZegoMediaplayerState.PLAYING;
            }else if (state == ZegoMediaplayerState.PAUSING.value()) {
                zegoMediaplayerState = ZegoMediaplayerState.PAUSING;
            }else if (state == ZegoMediaplayerState.PLAY_ENDED.value()) {
                zegoMediaplayerState = ZegoMediaplayerState.PLAY_ENDED;
            }else {
                zegoMediaplayerState = ZegoMediaplayerState.NO_PLAY;
            }
            return zegoMediaplayerState;
        }
        return ZegoMediaplayerState.NO_PLAY;
    }

    native static int getCurrentStateJni(int idx);

    native static int destroyMediaplayerJni(int idx);

    native static int createMediaplayerJni();

    native static int startJni(int idx);

    native static int stopJni(int idx);

    native static int pauseJni(int idx);

    native static int resumeJni(int idx);

    native static int enableRepeatJni(int idx, boolean enable);

    native static int enableAuxJni(int idx, boolean enable);

    native static int muteLocalJni(int idx, boolean mute);

    native static int setPlayerCanvasJni(int idx, Object view, int viewMode, int backgroundColor);

    native static int setVolumeJni(int idx, int volume);

    native static int setProgressIntervalJni(int idx, long millisecond);

    native static long getTotalDurationJni(int idx);

    native static long getCurrentProgressJni(int idx);

    native static int seekToJni(int idx, long millisecond);

    native static int loadResourceJni(int idx, String path);

    native static int enableAudioDataJni(boolean enable, int idx);

    native static int enableVideoDataJni(int value, boolean enable, int idx);

    native static int getVolumeJni(int idx) ;
}

