package im.zego.zegoexpress;

import java.nio.ByteBuffer;
import android.app.Application;

import java.util.ArrayList;
import im.zego.zegoexpress.callback.IZegoEventHandler;
import im.zego.zegoexpress.callback.IZegoExternalVideoCapturer;
import im.zego.zegoexpress.callback.IZegoPublisherUpdateCDNURLCallback;
import im.zego.zegoexpress.constants.ZegoAECMode;
import im.zego.zegoexpress.constants.ZegoCapturePipelineScaleMode;
import im.zego.zegoexpress.constants.ZegoLanguage;
import im.zego.zegoexpress.constants.ZegoOrientation;
import im.zego.zegoexpress.constants.ZegoScenario;
import im.zego.zegoexpress.constants.ZegoVideoMirrorMode;
import im.zego.zegoexpress.callback.IZegoExternalVideoRenderer;
import im.zego.zegoexpress.entity.ZegoBeautifyOption;
import im.zego.zegoexpress.entity.ZegoEngineConfig;
import im.zego.zegoexpress.entity.ZegoRoomConfig;
import im.zego.zegoexpress.entity.ZegoUser;
import im.zego.zegoexpress.entity.ZegoCanvas;
import im.zego.zegoexpress.entity.ZegoVideoConfig;
import im.zego.zegoexpress.callback.IZegoMixerStartCallback;
import im.zego.zegoexpress.entity.ZegoMixerTask;
import im.zego.zegoexpress.callback.IZegoIMSendBroadcastMessageCallback;
import im.zego.zegoexpress.callback.IZegoPublisherSetStreamExtraInfoCallback;
import im.zego.zegoexpress.entity.ZegoAudioConfig;
import im.zego.zegoexpress.entity.ZegoWatermark;
import im.zego.zegoexpress.callback.IZegoIMSendCustomCommandCallback;
import im.zego.zegoexpress.module.mediaplayer.ZegoMediaplayer;


public class ZegoExpressEngine {
    protected ZegoExpressEngine() {
    }

    /**
     * 创建引擎实例对象
     * 在调用其他 API 前需要先创建并初始化引擎；SDK 只支持创建一个 ZegoExpressEngine 实例，多次调用此接口返回的都是同一个对象。
     * @param appID ZEGO 为开发者签发的应用 ID，请从 ZEGO 管理控制台申请
     * @param appSign 每个 AppID 对应的应用签名，请从 ZEGO 管理控制台申请
     * @param isTestEnv 选择使用测试环境还是正式商用环境，正式环境需要在 ZEGO 管理控制台提交工单配置
     * @param scenario 所属的应用场景，当前请选择为 ZegoScenarioGeneral
     * @param application 应用上下文
     * @param eventHandler 事件通知回调。传空则意味着不接收任何回调通知。之后也可通过 addEventHandler/RemoveHandler 进行管理
     * @return (ZegoExpressEngine), 引擎实例对象
     */
    public static ZegoExpressEngine createEngine(long appID, String appSign, boolean isTestEnv, ZegoScenario scenario, Application application, IZegoEventHandler eventHandler) {
        return ZegoExpressEngineJni.createEngine(appID, appSign, isTestEnv, scenario, application, eventHandler);
    }

    /**
     * 销毁引擎实例对象<br>
     * 用于释放 ZegoExpressEngine 使用的资源。
     */
    public static void destroyEngine() {
        ZegoExpressEngineJni.destroyEngine();
    }

    /**
     * 获取 {@link ZegoExpressEngine} 的单例对象, 必须在 Zego Express Engine 的声明周期里调用本接口, 即必须在调用 <br/>
     * {@link ZegoExpressEngine#createEngine(long, String, boolean, ZegoScenario, Application, IZegoEventHandler)} 之后, 调用 {@link ZegoExpressEngine#destroyEngine()} 之前调用本接口
     *
     * @return 参考 {@link ZegoExpressEngine}
     */
    public static ZegoExpressEngine getEngine(){
        return ZegoExpressEngineJni.getEngine();
    }

    /**
     * 添加事件通知回调，最多支持添加 16 个
     * 
     * @param eventHandler 事件通知回调
     */
    public void addEventHandler(IZegoEventHandler eventHandler) {
        ZegoExpressEngineJni.addEventHandler(eventHandler);
    }

    /**
     * 移除事件通知回调
     * 
     * @param eventHandler 事件通知回调
     */
    public void removeEventHandler(IZegoEventHandler eventHandler) {
        ZegoExpressEngineJni.removeEventHandler(eventHandler);
    }

    /**
     * 获取 SDK 版本号
     * 
     * @return (String), SDK版本号
     */
    public static String getVersion() {
        return ZegoExpressEngineJni.getVersion();
    }

    /**
     * 上传日志到 ZEGO 服务器
     * 
     */
    public void uploadLog() {
        ZegoExpressEngineJni.uploadLog();
    }

    /**
     * 设置调试详细信息开关以及语言
     * 默认开启且调试信息的语言为英文。
     * @param enable 详细调试信息开关
     * @param language 调试信息语种
     */
    public void setDebugVerbose(boolean enable, ZegoLanguage language) {
        ZegoExpressEngineJni.setDebugVerbose(enable, language);
    }

    /**
     * 登录房间
     * 推拉流前必须登录房间，在同一个房间内的用户可以相互通话
     * 如果由于网络质量原因导致网络临时中断，SDK 内部会自动进行重连
     * 可通过监听 onRoomStateUpdate 代理方法获取本端当前房间连接状态的变化情况，同时同房间其他用户会接收到 onRoomUserUpdate 回调通知。
     * @param roomID 房间ID，最大长度为256字节的字符串
     * @param user 用户对象实例，配置用户 ID、用户名
     * @param config 房间高级配置，传空将使用默认配置
     */
    public void loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config) {
        ZegoExpressEngineJni.loginRoom(roomID, user, config);
    }

    /**
     * 登录房间
     * 如果有较高的安全需求请使用此接口
     * 推拉流前必须登录房间，在同一个房间内的用户可以相互通话
     * 如果由于网络质量原因导致网络临时中断，SDK 内部会自动进行重连
     * 可通过监听 onRoomStateUpdate 代理方法获取本端当前房间连接状态的变化情况，同时同房间其他用户会接收到 onRoomUserUpdate 回调通知。
     * @param roomID 房间ID，最大长度为256字节的字符串
     * @param user 用户对象实例，配置用户 ID、用户名
     * @param config 房间高级配置，传空将使用默认配置
     * @param token 由开发者业务服务器下发的 token，用以保证安全性，生成规则详见 https://doc.zego.im/CN/565.html
     */
    public void loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config, String token) {
        ZegoExpressEngineJni.loginRoom(roomID, user, config, token);
    }

    /**
     * 退出房间
     * 退出房间会停止该用户的所有推拉流，调用该接口后会收到  onRoomStateUpdate 回调通知成功退出房间，同时同房间其他用户会接收到 onRoomUserUpdate  回调通知。
     * @param roomID 用户已登录的房间 ID
     */
    public void logoutRoom(String roomID) {
        ZegoExpressEngineJni.logoutRoom(roomID);
    }

    /**
     * 开始推流
     * 可通过此接口让用户将自己本地的音视频流推送到 ZEGO 实时音视频云，同一房间的其他用户通过 streamID 就可以拉取该音视频流进行互通。
     * 在开始推流前，需要先加入房间，同房间内其他用户可通过监听 onRoomStreamUpdate 事件回调来获取该 streamID 新增。
     * 在网络质量不佳的情况下，用户推流可能出现中断， SDK 会尝试重新连接，可通过监听  onPublisherStateUpdate 事件来获知当前推流状态以及错误信息。
     * @param streamID 流 ID，长度不超过256的字符串，需要在整个 AppID 内全局唯一，若出现在同一个 AppID 内，不同的用户各推了一条流且流名相同 ，将会导致后推流的用户推流失败
     */
    public void startPublishing(String streamID) {
        ZegoExpressEngineJni.startPublishing(streamID);
    }

    /**
     * 停止推流
     * 可通过此接口让用户停止发送本地的音视频流，结束通话。如果用户已经启动推流，在推新流（新的streamID）之前，必须要调用此接口停止当前流的推送，否则新流推送会返回失败。
     */
    public void stopPublishing() {
        ZegoExpressEngineJni.stopPublishing();
    }

    /**
     * 设置流附加信息
     * 可通过此接口设置当前推流的流附加信息，设置结果会通过 IZegoPublisherSetStreamExtraInfoCallback 回调通知。
     * @param extraInfo 流附加信息，长度不超过1024的字符串。
     * @param callback 更新流附加信息执行结果通知
     */
    public void setStreamExtraInfo(String extraInfo, IZegoPublisherSetStreamExtraInfoCallback callback) {
        ZegoExpressEngineJni.setStreamExtraInfo(extraInfo, callback);
    }

    /**
     * 启动/更新本地预览
     * 用户通过调用此接口可以看到自己本地的画面。预览功能不需要先登陆房间或推流。
     * 可以通过再次调用该接口来更新本地视图和视图预览模式。
     * 可以通过调用 setVideoMirrorMode 接口来设置预览的镜像模式，默认为开启预览镜像。
     * @param canvas 启动预览时用于显示画面的视图，视图设置为空则不进行预览。
     */
    public void startPreview(ZegoCanvas canvas) {
        ZegoExpressEngineJni.startPreview(canvas);
    }

    /**
     * 停止本地预览
     * 当本地不需要看到预览画面时可调用此接口停止预览。
     */
    public void stopPreview() {
        ZegoExpressEngineJni.stopPreview();
    }

    /**
     * 设置视频配置
     * 可通过此接口设置视频帧率、码率，视频采集分辨率，视频编码输出分辨率。如果不调用此接口，默认分辨率为 360p，码率为 600 kbps，帧率为 15 fps。需要在推流前设置好相关视频配置，在推流后仅支持编码分辨率和码率的修改。
     * @param videoConfig 视频配置，SDK 提供常用的分辨率、帧率和码率的组合值，也可自定义分辨率、帧率和码率
     */
    public void setVideoConfig(ZegoVideoConfig videoConfig) {
        ZegoExpressEngineJni.setVideoConfig(videoConfig);
    }

    /**
     * 设置镜像模式
     * 可调用此接口来设置本地预览视频以及推送的视频是否开启镜像模式。
     * @param mirrorMode 预览或推流的镜像模式
     */
    public void setVideoMirrorMode(ZegoVideoMirrorMode mirrorMode) {
        ZegoExpressEngineJni.setVideoMirrorMode(mirrorMode);
    }

    /**
     * 设置采集视频的朝向
     * 此接口可设置视频的朝向方位，相比与手机正立的正向，将采集到的数据向逆时针方向分别旋转90，180或270度。旋转后会自动进行调整，以适配编码后的图像分辨率。
     * @param orientation 视频朝向
     */
    public void setAppOrientation(ZegoOrientation orientation) {
        ZegoExpressEngineJni.setAppOrientation(orientation);
    }

    /**
     * 设置音频质量配置
     * 可通过此接口设置音频编码类型、码率，声道数的组合值。如果不调用此接口，默认为“普通延迟-标准音质”模式。仅支持推流前设置。
     * @param config 音频质量配置
     */
    public void setAudioConfig(ZegoAudioConfig config) {
        ZegoExpressEngineJni.setAudioConfig(config);
    }

    /**
     * 停止或恢复发送音频流
     * 推流时可调用此接口实现只推视频流不推音频，本地仍会采集和处理音频，但不向网络发送音频数据。可以在推流前设置。
     * 如果在本地设置了停止发送音频流，对端可以通过监听 onRemoteMicStateUpdate 回调收到 REMOTE_DEVICE_STATE_MUTE 的状态变更通知。
     * @param mute 是否停止发送音频流，true 表示只发送视频流不发送音频流，false 表示同时发送音频和视频流。默认为false
     */
    public void mutePublishStreamAudio(boolean mute) {
        ZegoExpressEngineJni.mutePublishStreamAudio(mute);
    }

    /**
     * 停止或恢复发送视频流
     * 推流时可调用此接口实现只推音频流不推视频流，本地摄像头仍能正常工作，能正常采集，预览和处理视频画面，但不向网络发送视频数据。可以在推流前设置。
     * 如果在本地设置了停止发送视频流，对端可以通过监听 onRemoteCameraStateUpdate 回调收到 REMOTE_DEVICE_STATE_MUTE 的状态变更通知。
     * @param mute 是否停止发送视频流，true 表示只发送音频流不发送视频流，false 表示同时发送音频和视频流。默认为false。
     */
    public void mutePublishStreamVideo(boolean mute) {
        ZegoExpressEngineJni.mutePublishStreamVideo(mute);
    }

    /**
     * 设置推流端采集音量
     * 此接口用于设置音频的采集音量，本端用户可控制往远端发送音频流的声音大小。可以在推流前设置。
     * @param volume 音量百分比，默认值为100
     */
    public void setCaptureVolume(int volume) {
        ZegoExpressEngineJni.setCaptureVolume(volume);
    }

    /**
     * 增加转推至CDN的URL
     * 当需要将音视频流转推到其它指定的 CDN 时需要调用此接口进行设置。
     * @param streamID 需要转推的流 ID
     * @param targetURL CDN 转推地址，支持的转推地址格式有 rtmp，flv，hls
     * @param callback 配置转推CDN操作的执行结果通知，可根据执行结果进行下一步处理。
     */
    public void addPublishCDNURL(String streamID, String targetURL, IZegoPublisherUpdateCDNURLCallback callback) {
        ZegoExpressEngineJni.addPublishCDNURL(streamID, targetURL, callback);
    }

    /**
     * 删除转推至CDN的URL 
     * 当已经添加了某个CDN转推地址，需要停止将流停止转推至该CDN时调用此接口。
     * @param streamID 需要停止转推的流 ID
     * @param targetURL CDN 转推地址，支持的转推地址格式有 rtmp，flv，hls
     * @param callback 移除 CDN 转推结果通知
     */
    public void removePublishCDNURL(String streamID, String targetURL, IZegoPublisherUpdateCDNURLCallback callback) {
        ZegoExpressEngineJni.removePublishCDNURL(streamID, targetURL, callback);
    }

    /**
     * 设置推流水印
     * 推流前设置，水印的布局不能超出推流的视频编码分辨率。
     * @param watermark 水印布局左上角为坐标系原点，区域不能超过编码分辨率设置的大小
     * @param isPreviewVisible 是否本地预览能看见水印
     */
    public void setPublishWatermark(ZegoWatermark watermark, boolean isPreviewVisible) {
        ZegoExpressEngineJni.setPublishWatermark(watermark, isPreviewVisible);
    }

    /**
     * 发送媒体增强补充信息
     * 此接口可在开发者推流传输音视频流数据同时，发送流媒体增强补充信息来同步一些其他附加信息。
     * 一般如同步音乐歌词或视频画面精准布局等场景，可选择使用发送 SEI。
     * 当推流方发送 SEI 后，拉流方可通过监听 onPlayerRecvSEI 的回调获取SEI内容。
     * @param data SEI 内容
     * @param dataLength SEI 内容长度
     */
    public void sendSEI(ByteBuffer data, int dataLength) {
        ZegoExpressEngineJni.sendSEI(data, dataLength);
    }

    /**
     * 开/关硬件编码
     * 推流时是否采用硬件编码的开关，开启硬解编码后会使用GPU进行编码，降低CPU使用率。在推流前设置才能生效，如果在推流后设置，停推后重新推流可以生效。
     * @param enable 是否开启硬件编码。true 表示开启硬件编码，false 表示关闭硬件编码。
     */
    public void enableHardwareEncoder(boolean enable) {
        ZegoExpressEngineJni.enableHardwareEncoder(enable);
    }

    /**
     * 设置采集时机
     * 
     * @param mode 采集时机
     */
    public void setCapturePipelineScaleMode(ZegoCapturePipelineScaleMode mode) {
        ZegoExpressEngineJni.setCapturePipelineScaleMode(mode);
    }

    /**
     * 开始拉流
     * 可通过此接口让用户可以从 ZEGO 实时音视频云拉取远端用户的音视频流进行互通。
     * 在开始拉流前，需要先加入房间，可通过监听 onRoomStreamUpdate 事件回调来获取该房间内 streamID 的新增。
     * 在网络质量不佳的情况下，用户拉流可能出现中断， SDK 会尝试重新连接，可通过监听 onPlayerStateUpdate 事件来获知当前拉流状态以及错误信息。
     * 拉取不存在的流 ID，执行本接口后 SDK 持续尝试拉取，在该流 ID 被成功推送后，音视频流可以真正被拉取到。
     * 开发者可通过再次调用此接口实现更换拉流 Canvas 的操作（流 ID 必须一样）。
     * @param streamID 流 ID
     * @param canvas （ \link com::zego::zegoexpress::entity::ZegoCanvas ZegoCanvas \endlink ）用于显示拉流画面的视图，视图设置为空则不进行显示。
     */
    public void startPlayingStream(String streamID, ZegoCanvas canvas) {
        ZegoExpressEngineJni.startPlayingStream(streamID, canvas);
    }

    /**
     * 停止拉流
     * 可通过此接口让用户停止拉取远端的音视频流。停止拉流后对此条流此前设置的属性如 setPlayVolume、mutePlayStreamAudio、mutePlayStreamVideo等都会失效，需要在下次拉流时重新设置。
     * @param streamID 流 ID
     */
    public void stopPlayingStream(String streamID) {
        ZegoExpressEngineJni.stopPlayingStream(streamID);
    }

    /**
     * 设置拉流音量
     * 此接口用于设置拉流的声音大小，本端用户可控制音频流的播放音量。需要在调用拉流接口后调用。
     * @param streamID 流 ID
     * @param volume 音量百分比
     */
    public void setPlayVolume(String streamID, int volume) {
        ZegoExpressEngineJni.setPlayVolume(streamID, volume);
    }

    /**
     * 停止或恢复拉取音频流
     * 拉流时可调用此接口实现不拉取远端用户的音频数据，需要在调用拉流接口后调用。
     * @param streamID 流 ID
     * @param mute 禁用标识。true:禁止拉取；false:恢复拉取。
     */
    public void mutePlayStreamAudio(String streamID, boolean mute) {
        ZegoExpressEngineJni.mutePlayStreamAudio(streamID, mute);
    }

    /**
     * 停止或恢复拉取视频流
     * 拉流时可调用此接口实现不拉取远端用户的视频数据，需要在调用拉流接口后调用。
     * @param streamID 流 ID
     * @param mute 禁用标识。true:禁止拉取；false:恢复拉取。
     */
    public void mutePlayStreamVideo(String streamID, boolean mute) {
        ZegoExpressEngineJni.mutePlayStreamVideo(streamID, mute);
    }

    /**
     * 开/关硬件解码
     * 拉流时是否使用硬件解码，开启硬件解码后SDK会使用GPU进行解码，降低CPU使用率。在拉流前设置才能生效，如果在拉流后设置，停止拉流后重新拉流可以生效。
     * @param enable 是否开启硬解开关。true:启用硬解；false:关闭硬解。
     */
    public void enableHardwareDecoder(boolean enable) {
        ZegoExpressEngineJni.enableHardwareDecoder(enable);
    }

    /**
     * 开/关帧顺序检测
     * 
     * @param enable true  检测帧顺序，不支持B帧； false 不检测帧顺序，支持B帧，可能出现短暂花屏
     */
    public void enableCheckPoc(boolean enable) {
        ZegoExpressEngineJni.enableCheckPoc(enable);
    }

    /**
     * 开始混流任务
     * 
     * @param task 混流任务对象
     * @param callback 开始混流任务结果通知
     */
    public void startMixerTask(ZegoMixerTask task, IZegoMixerStartCallback callback) {
        ZegoExpressEngineJni.startMixerTask(task, callback);
    }

    /**
     * 停止混流任务
     * 
     * @param taskID 混流任务 ID
     */
    public void stopMixerTask(String taskID) {
        ZegoExpressEngineJni.stopMixerTask(taskID);
    }

    /**
     * 开/关麦克风
     * 此接口用于控制是否使用采集到的音频数据，关闭麦克风即采集到数据后就丢弃，此时仍然会占用麦克风
     * @param mute 是否开启麦克风。true 表示关闭麦克风；false 表示开启麦克风
     */
    public void muteMicrophone(boolean mute) {
        ZegoExpressEngineJni.muteMicrophone(mute);
    }

    /**
     * 开/关音频输出至设备
     * 关闭后SDK将不会再传递音频数据给输出设备，以达到静音的效果
     * @param mute 是否关闭音频输出到设备，true表示关闭音频输出；false 表示开启音频输出
     */
    public void muteAudioOutput(boolean mute) {
        ZegoExpressEngineJni.muteAudioOutput(mute);
    }

    /**
     * 开/关音频采集设备
     * 此接口用于控制是否释放音频采集设备，关闭音频采集设备，则 SDK 不会再占用音频设备，当然如果此时正在推流，也不会产生音频数据。
     * @param enable 是否使能音频采集设备。true 表示开启音频采集设备，false 表示关闭音频采集设备
     */
    public void enableAudioCaptureDevice(boolean enable) {
        ZegoExpressEngineJni.enableAudioCaptureDevice(enable);
    }

    /**
     * 开/关摄像头
     * 此接口用于控制是否启动摄像头的采集，关闭摄像头后，将不会进行视频采集，此时推流也将没有视频数据
     * @param enable 是否打开摄像头。false 表示关闭摄像头；true 表示开启摄像头
     */
    public void enableCamera(boolean enable) {
        ZegoExpressEngineJni.enableCamera(enable);
    }

    /**
     * 切换前后摄像头
     * 此接口用于控制使用前摄像头或者后摄像头
     * @param enable 是否采用前置摄像头。false 表示使用后置摄像头；true 表示使用前置摄像头
     */
    public void useFrontCamera(boolean enable) {
        ZegoExpressEngineJni.useFrontCamera(enable);
    }

    /**
     * 启动声浪监控
     * 
     */
    public void startSoundLevelMonitor() {
        ZegoExpressEngineJni.startSoundLevelMonitor();
    }

    /**
     * 停止声浪监控
     * 
     */
    public void stopSoundLevelMonitor() {
        ZegoExpressEngineJni.stopSoundLevelMonitor();
    }

    /**
     * 启动音频频谱监控
     * 
     */
    public void startAudioSpectrumMonitor() {
        ZegoExpressEngineJni.startAudioSpectrumMonitor();
    }

    /**
     * 停止音频频谱监控
     * 
     */
    public void stopAudioSpectrumMonitor() {
        ZegoExpressEngineJni.stopAudioSpectrumMonitor();
    }

    /**
     * 开/关回声消除
     * 打开回声消除， SDK 会对采集到的音频数据进行过滤以降低音频中的回音成分。需要在开始推流之前设置，推流开始后设置无效。
     * @param enable 是否开启回声消除，false 表示关闭回声消除，true 表示开启回声消除
     */
    public void enableAEC(boolean enable) {
        ZegoExpressEngineJni.enableAEC(enable);
    }

    /**
     * 设置回声消除模式
     * 切换不同的回声消除模式以控制消除回声数据的程度。需要在开始推流之前设置。
     * @param mode （ \link com::zego::zegoexpress::constants::ZegoAECMode ZegoAECMode \endlink ）回声消除模式
     */
    public void setAECMode(ZegoAECMode mode) {
        ZegoExpressEngineJni.setAECMode(mode);
    }

    /**
     * 开/关自动增益
     * 开启自动增益后声音会被放大，但会一定程度上影响音质。需要在开始推流之前设置。
     * @param enable 是否开启自动增益，false 表示关闭自动增益，true 表示开启自动增益
     */
    public void enableAGC(boolean enable) {
        ZegoExpressEngineJni.enableAGC(enable);
    }

    /**
     * 开/关噪声抑制
     * 开启噪声抑制可以使人声更加清晰。需要在开始推流之前设置，推流开始后设置无效。
     * @param enable 是否开启噪声抑制，false 表示关闭噪声抑制，true 表示开启噪声抑制
     */
    public void enableANS(boolean enable) {
        ZegoExpressEngineJni.enableANS(enable);
    }

    /**
     * 开/关美颜
     * 识别视频中的人像进行美颜。开始推流前后都可以设置。
     * @param featureBitmask 比特掩码格式，可选择同时开启 ZegoBeautifyFeature 中的几个特性
     */
    public void enableBeautify(int featureBitmask) {
        ZegoExpressEngineJni.enableBeautify(featureBitmask);
    }

    /**
     * 设置美颜参数
     * 
     * @param option 美颜选项参数
     */
    public void setBeautifyOption(ZegoBeautifyOption option) {
        ZegoExpressEngineJni.setBeautifyOption(option);
    }

    /**
     * 发送房间广播消息
     * 
     * @param message 消息内容，长度不超过256字节
     * @param roomID 房间 ID
     * @param callback 发送广播消息结果通知
     */
    public void sendBroadcastMessage(String message, String roomID, IZegoIMSendBroadcastMessageCallback callback) {
        ZegoExpressEngineJni.sendBroadcastMessage(message, roomID, callback);
    }

    /**
     * 发送自定义信令
     * 
     * @param command 自定义信令内容，长度不超过256字节
     * @param toUserList 信令的接收者
     * @param roomID 房间 ID
     * @param callback 发送信令结果通知
     */
    public void sendCustomCommand(String command, ArrayList<ZegoUser> toUserList, String roomID, IZegoIMSendCustomCommandCallback callback) {
        ZegoExpressEngineJni.sendCustomCommand(command, toUserList, roomID, callback);
    }

    /**
     * 创建媒体播放器实例, 当前最多可以创建4个实例,
     *
     * @return (ZegoMediaplayer) 媒体播放器实例
     */
    public ZegoMediaplayer createMediaplayer(){
        return ZegoExpressEngineJni.createMediaplayer();
    }

//    /**
//     * 设置外部渲染器对象, 用户传入自己构造的渲染器对象 IZegoExternalVideoRenderer
//     *
//     * @param renderer 渲染器对象, 开发者须自行实现渲染器对象须实现的方法并自行渲染视频到UI上
//     */
//    public void setExternalVideoRenderer(IZegoExternalVideoRenderer renderer){
//        ZegoExpressEngineJni.setExternalVideoRenderer(renderer);
//    }
//
//    /**
//     * 设置外部视频采集器对象, 用户传入自己构造的渲染器对象 IZegoExternalVideoCapturer
//     *
//     * @param capturer 视频采集器对象, 开发者须自行传染并实现采集器对象须实现的方法
//     */
//    public void setExternalVideoCapturer(IZegoExternalVideoCapturer capturer){
//        ZegoExpressEngineJni.setExternalVideoCapturer(capturer);
//    }
//
//    /**
//     * 设置初始化配置, 高级功能需求的客户, 若无相关需求无须关注该方法 <br/>
//     * 该方法需要创建引擎 {@link ZegoExpressEngine#createEngine(long, String, boolean, ZegoScenario, Application, IZegoEventHandler)} 之前调用有效
//     *
//     * @param engineConfig 初始化配置
//     */
//    public static void setEngineConfig(ZegoEngineConfig engineConfig){
//        ZegoExpressEngineJni.setEngineConfig(engineConfig);
//    }

}