package im.zego.zegoexpress;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.internal.*;
import android.app.Application;
import android.graphics.*;

/**
 * Zego MediaPlayer.
 *
 * Yon can use ZegoMediaPlayer to play media resource files on the local or remote server, and can mix the sound of the media resource files that are played into the publish stream to achieve the effect of background music.
 *
 */
public abstract class ZegoMediaPlayer {

    /**
     * Set event callback handler for media player.
     *
     * Developers can change the player UI widget according to the related event callback of the media player
     *
     * @param handler Media player event callback object
     */
    public abstract void setEventHandler(IZegoMediaPlayerEventHandler handler);

    /**
     * Set audio callback handler.
     *
     * You can set this callback to throw the audio data of the media resource file played by the media player
     *
     * @param handler Audio event callback object for media player
     */
    public abstract void setAudioHandler(IZegoMediaPlayerAudioHandler handler);

    /**
     * Load media resource.
     *
     * Yon can pass the absolute path of the local resource or the URL of the network resource
     *
     * @param path the absolute path of the local resource or the URL of the network resource
     * @param callback Notification of resource loading results
     */
    public abstract void loadResource(String path, IZegoMediaPlayerLoadResourceCallback callback);

    /**
     * Start playing.
     *
     * You need to load resources before playing
     */
    public abstract void start();

    /**
     * Stop playing.
     */
    public abstract void stop();

    /**
     * Pause playing.
     */
    public abstract void pause();

    /**
     * Resume playing.
     */
    public abstract void resume();

    /**
     * Set the specified playback progress.
     *
     * Unit is millisecond
     *
     * @param millisecond Point in time of specified playback progress
     * @param callback the result notification of set the specified playback progress
     */
    public abstract void seekTo(long millisecond, IZegoMediaPlayerSeekToCallback callback);

    /**
     * Whether to repeat playback.
     *
     * @param enable repeat playback flag. The default is [false].
     */
    public abstract void enableRepeat(boolean enable);

    /**
     * Whether to mix the player's sound into the stream being published.
     *
     * @param enable Aux audio flag. The default is [false].
     */
    public abstract void enableAux(boolean enable);

    /**
     * Whether to play locally silently.
     *
     * If [enableAux] switch is turned on, there is still sound in the publishing stream. The default is [false].
     *
     * @param mute Mute local audio flag, The default is [false].
     */
    public abstract void muteLocal(boolean mute);

    /**
     * Set mediaplayer volume. Both the local play volume and the publish volume are set.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setVolume(int volume);

    /**
     * Set mediaplayer local play volume.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setPlayVolume(int volume);

    /**
     * Set mediaplayer publish volume.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setPublishVolume(int volume);

    /**
     * Set playback progress callback interval.
     *
     * This function can control the callback frequency of [onMediaPlayerPlayingProgress]. When the callback interval is set to 0, the callback is stopped. The default callback interval is 1s
     * This callback are not returned exactly at the set callback interval, but rather at the frequency at which the audio or video frames are processed to determine whether the callback is needed to call
     *
     * @param millisecond Interval of playback progress callback in milliseconds
     */
    public abstract void setProgressInterval(long millisecond);

    /**
     * Gets the current local playback volume of the mediaplayer, the range is 0 ~ 200, with the default value of 60.
     */
    public abstract int getPlayVolume();

    /**
     * Gets the current publish volume of the mediaplayer, the range is 0 ~ 200, with the default value of 60.
     */
    public abstract int getPublishVolume();

    /**
     * Get the total progress of your media resources.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     *
     * @return Unit is millisecond
     */
    public abstract long getTotalDuration();

    /**
     * Get current playing progress.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     */
    public abstract long getCurrentProgress();

    /**
     * Get the number of audio tracks of the playback file.
     */
    public abstract int getAudioTrackCount();

    /**
     * Set the audio track of the playback file.
     *
     * @param index Audio track index, the number of audio tracks can be obtained through the [getAudioTrackCount] function.
     */
    public abstract void setAudioTrackIndex(int index);

    /**
     * Setting up the specific voice changer parameters.
     *
     * @param audioChannel The audio channel to be voice changed
     * @param param Voice changer parameters
     */
    public abstract void setVoiceChangerParam(ZegoMediaPlayerAudioChannel audioChannel, ZegoVoiceChangerParam param);

    /**
     * Get the current playback status.
     */
    public abstract ZegoMediaPlayerState getCurrentState();

    /**
     * Get media player index.
     */
    public abstract int getIndex();

}

