package im.zego.zegoexpress;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.internal.*;
import android.app.Application;
import android.graphics.*;

public abstract class ZegoAudioEffectPlayer {

    /**
     * set audio effect player event handler.
     *
     * @param handler event handler for audio effect player
     */
    public abstract void setEventHandler(IZegoAudioEffectPlayerEventHandler handler);

    /**
     * Start playing audio effect.
     *
     * The default is only played once and is not mixed into the publishing stream, if you want to change this please modify [config].
     *
     * @param audioEffectID ID for the audio effect. The SDK uses audioEffectID to control the playback of sound effects. The SDK does not force the user to pass in this parameter as a fixed value. It is best to ensure that each sound effect can have a unique id. The recommended methods are static self-incrementing id or the hash of the incoming sound effect file path.
     * @param path The absolute path of the local resource. "assets://"、"ipod-library://" and network url are not supported. Set path as [null] or "" if resource is loaded already using [loadResource]
     * @param config Audio effect playback configuration. Set [null] will only be played once, and will not be mixed into the publishing stream.
     */
    public abstract void start(int audioEffectID, String path, ZegoAudioEffectPlayConfig config);

    /**
     * Stop playing audio effect.
     *
     * @param audioEffectID ID for the audio effect
     */
    public abstract void stop(int audioEffectID);

    /**
     * Pause playing audio effect.
     *
     * @param audioEffectID ID for the audio effect
     */
    public abstract void pause(int audioEffectID);

    /**
     * Resume playing audio effect.
     *
     * @param audioEffectID ID for the audio effect
     */
    public abstract void resume(int audioEffectID);

    /**
     * Stop playing all audio effect.
     */
    public abstract void stopAll();

    /**
     * Pause playing all audio effect.
     */
    public abstract void pauseAll();

    /**
     * Resume playing all audio effect.
     */
    public abstract void resumeAll();

    /**
     * Set the specified playback progress.
     *
     * Unit is millisecond
     *
     * @param audioEffectID ID for the audio effect
     * @param millisecond Point in time of specified playback progress
     * @param callback seek to result
     */
    public abstract void seekTo(int audioEffectID, long millisecond, IZegoAudioEffectPlayerSeekToCallback callback);

    /**
     * Set volume for the audio effect. Both the local play volume and the publish volume are set.
     *
     * @param audioEffectID ID for the audio effect
     * @param volume The range is 0 ~ 200. The default is 100.
     */
    public abstract void setVolume(int audioEffectID, int volume);

    /**
     * Set volume for all audio effect. Both the local play volume and the publish volume are set.
     *
     * @param volume The range is 0 ~ 200. The default is 100.
     */
    public abstract void setVolumeAll(int volume);

    /**
     * Get the total progress of your media resources.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     *
     * @param audioEffectID ID for the audio effect
     *
     * @return Unit is millisecond
     */
    public abstract long getTotalDuration(int audioEffectID);

    /**
     * Get current playing progress.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     *
     * @param audioEffectID ID for the audio effect
     */
    public abstract long getCurrentProgress(int audioEffectID);

    /**
     * Load audio effect resource.
     *
     * In a scene where the same sound effect is played frequently, the SDK provides the function of preloading the sound effect file into the memory in order to optimize the performance of repeatedly reading and decoding the file. Preloading supports loading up to 15 sound effect files at the same time, and the duration of the sound effect files cannot exceed 30s, otherwise an error will be reported when loading
     *
     * @param audioEffectID ID for the audio effect
     * @param path the absolute path of the audio effect resource.
     * @param callback load audio effect resource result
     */
    public abstract void loadResource(int audioEffectID, String path, IZegoAudioEffectPlayerLoadResourceCallback callback);

    /**
     * Unload audio effect resource.
     *
     * After the sound effects are used up, related resources can be released through this function; otherwise, the SDK will release the loaded resources when the AudioEffectPlayer instance is destroyed.
     *
     * @param audioEffectID ID for the audio effect loaded
     */
    public abstract void unloadResource(int audioEffectID);

    /**
     * Get audio effect player index.
     *
     * @return Audio effect player index
     */
    public abstract int getIndex();

}

