package com.zego.zegoavkit2.receiver;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;


public final class BackgroundMonitor implements Background.Listener {
    public static final String TAG = "BackgroundMonitor";
    private long mThis;
    private Background.Binding mListenerBinding;

    public void setThis(long pThis) {
        mThis = pThis;
    }

    public int init(Context context) {

        if (context == null) {
            return -1;
        }

        if (context instanceof Application) {
            mListenerBinding = Background.get().init((Application) context).addListener(this);
            return 0;
        }

        return -1;
    }

    public int uninit() {

        if (mListenerBinding != null) {
            mListenerBinding.unbind();
            //do not uninit
//            Background.get().uninit();
        }

        return 0;
    }

    public boolean isBackground() {
        return Background.get().isBackground();
    }

    @Override
    public void onBecameForeground() {
        onBackgroundStatusChanged(mThis, false);
    }

    @Override
    public void onBecameBackground() {
        onBackgroundStatusChanged(mThis, true);
    }

    static native void onBackgroundStatusChanged(long pThis, boolean isBackground);
}


class Background implements Application.ActivityLifecycleCallbacks {

    public static final String TAG = "BackgroundMonitor";

    public static final long CHECK_DELAY = 2000;

    public interface Listener {
        public void onBecameForeground();

        public void onBecameBackground();
    }

    public interface Binding {
        public void unbind();
    }

    private interface Callback {
        public void invoke(Listener listener);
    }

    private static class Listeners {
        private List<WeakReference<Listener>> listeners = new CopyOnWriteArrayList<>();

        public Binding add(Listener listener) {
            final WeakReference<Listener> wr = new WeakReference<>(listener);
            listeners.add(wr);
            return new Binding() {
                public void unbind() {
                    listeners.remove(wr);
                }
            };
        }

        public void each(Callback callback) {
            List<WeakReference<Listener>> toRemove = new ArrayList<>();
            for (Iterator<WeakReference<Listener>> it = listeners.iterator(); it.hasNext(); ) {
                try {
                    WeakReference<Listener> wr = it.next();
                    Listener l = wr.get();
                    if (l != null) {
                        callback.invoke(l);
                    } else {
                        toRemove.add(wr);
                    }
                } catch (Exception exc) {
                    Log.e(TAG, "Listener threw exception!", exc);
                }
            }
            if (!toRemove.isEmpty()) {
                listeners.removeAll(toRemove);
            }
        }
    }

    private static Callback becameForeground = new Callback() {
        @Override
        public void invoke(Listener listener) {
            listener.onBecameForeground();
        }
    };

    private static Callback becameBackground = new Callback() {
        @Override
        public void invoke(Listener listener) {
            listener.onBecameBackground();
        }
    };

    private static Background instance = new Background();

    private boolean background;
    private WeakReference<Activity> currentActivity;
    private Listeners listeners = new Listeners();
    private Handler handler = new Handler(Looper.getMainLooper());
    private Runnable check;
    private Application application;

    public Background init(Application app) {
        uninit();
        application = app;
        application.registerActivityLifecycleCallbacks(this);
        return this;
    }

    public Background uninit() {
        if (application != null) {
            application.unregisterActivityLifecycleCallbacks(this);
            application = null;
        }
        return this;
    }

    public static Background get() {
        return instance;
    }

    public boolean isBackground() {
        return background;
    }

    public boolean isForeground() {
        return !background;
    }

    public Binding addListener(Listener listener) {
        return listeners.add(listener);
    }

    @Override
    public void onActivityResumed(Activity activity) {
    }

    @Override
    public void onActivityPaused(Activity activity) {
        // if we're changing configurations we aren't going background so
        // no need to schedule the check
        if (!activity.isChangingConfigurations()) {
            // don't prevent activity being gc'd
            final WeakReference<Activity> ref = new WeakReference<>(activity);
            handler.postDelayed(check = new Runnable() {
                @Override
                public void run() {
                    onActivityCeased(ref.get());
                }
            }, CHECK_DELAY);
        }
    }

    @Override
    public void onActivityStarted(Activity activity) {
        currentActivity = new WeakReference<>(activity);
        // remove any scheduled checks since we're starting another activity
        // we're definitely not going background
        if (check != null) {
            handler.removeCallbacks(check);
        }

        // check if we're becoming foreground and notify listeners
        if (background && (activity != null && !activity.isChangingConfigurations())) {
            background = false;
            Log.i(TAG, "became foreground");
            listeners.each(becameForeground);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        if (check != null) {
            handler.removeCallbacks(check);
        }
        onActivityCeased(activity);
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
    }


    private void onActivityCeased(Activity activity) {
        if (!background) {
            if ((currentActivity == null || (activity == currentActivity.get()))
                    && (activity != null && !activity.isChangingConfigurations())) {
                background = true;
                Log.i(TAG, "became background");
                listeners.each(becameBackground);
            }
        }
    }
}
