package id.unum.crossPlatformInterfaces

import id.unum.Base58
import id.unum.crossPlatformInterfaces.PemUtil.getPublicKeyFromPem
import id.unum.protos.crypto.v1.PublicKeyInfo
import org.bouncycastle.jce.provider.BouncyCastleProvider
import java.security.KeyFactory
import java.security.PrivateKey
import java.security.Security
import java.security.Signature
import java.security.spec.X509EncodedKeySpec

internal object JvmSignature {
    fun verifySignature(data: ByteArray, signature: ByteArray, key: ByteArray): Boolean {
        val keyFactory = KeyFactory.getInstance(EC_KEYSTORE)
        val keySpec = X509EncodedKeySpec(key)
        val list: List<String> = listOf(
            "NONEwithECDSA",
            "SHA1withECDSA",
            "SHA224withECDSA",
            "SHA256withECDSA",
            "SHA384withECDSA",
            "SHA512withECDSA"
        )
        for (each in list) {
            try {
                val result = Signature.getInstance(each).run {
                    initVerify(keyFactory.generatePublic(keySpec))
                    update(data)
                    verify(signature)
                }

                if (result) {
                    return result
                }
            } catch (e: Exception) {
                println("Error trying $each algorithm")
                println(e.localizedMessage)
            }
        }

        return false
    }

    fun signData(data: ByteArray, privateKey: PrivateKey, type: String): String {
        try {
            privateKey.let {
                val signature: ByteArray? = Signature.getInstance(type).run {
                    initSign(it)
                    update(data)
                    sign()
                }

                if (signature != null) {
                    return Base58.encode(signature)
                }
            }
        } catch (e: Exception) {
            println("error signing data:$e") // TODO add a real logging framework
            throw e; // Why does it seem like this actually doesn't throw? returns an empty string
        }

        throw Exception("Error singingData");
    }

    fun verifyDataWithProvidedKey(key: PublicKeyInfo, data: ByteArray, signatureResults: String): Boolean {
        val platformProvider = PlatformProviderFactory.createPlatformProvider()
        try {
            val encoded: ByteArray? = when {
                key.encoding.lowercase() == "pem" -> {
                    getPublicKeyFromPem(key.publicKey)
                }
                key.encoding.lowercase() == "base58" -> {
                    Base58.decode(key.publicKey)
                }
                else -> {
                    byteArrayOf()
                }
            }
            val signature: ByteArray = Base58.decode(signatureResults)
            encoded?.let {
                return platformProvider.verifySignature(data, signature, it)
            }
            return false
        } catch (e: Exception) {
            return false
        }
    }
}