@file:Suppress("JAVA_MODULE_DOES_NOT_EXPORT_PACKAGE")

package id.unum.crossPlatformInterfaces

import id.unum.Base58
import id.unum.crossPlatformInterfaces.PemUtil.getPrivateKeyFromPem
import id.unum.crossPlatformInterfaces.PemUtil.getPublicKeyFromPem
import id.unum.crossPlatformInterfaces.PemUtil.toPem
import id.unum.protos.crypto.v1.KeyPair
import java.security.KeyFactory
import java.security.PrivateKey
import java.security.PublicKey
import java.security.spec.PKCS8EncodedKeySpec
import java.security.spec.X509EncodedKeySpec


internal object JvmKeys {

    fun generateEcKey(encoding: Encoding): KeyPair? {
        val generator = KeyGenerator(EC_KEYSTORE)
        generator.generate(512)
        generator.privateKey?.let { private ->
            generator.publicKey?.let { public ->
                return createKeyPair(encoding, private, public)
            }
        }

        return null
    }

    fun generateRsaKey(encoding: Encoding): KeyPair? {
        val generator = KeyGenerator(RSA_KEYSTORE)
        generator.generate(2048)
        generator.privateKey?.let { private ->
            generator.publicKey?.let { public ->
                return createKeyPair(encoding, private, public)
            }
        }

        return null
    }

    private fun createKeyPair(encoding: Encoding, private: PrivateKey, public: PublicKey): KeyPair {
        val privateString = if (encoding == Encoding.PEM) {
            toPem("PRIVATE KEY", private.encoded)
        } else {
            toBase58(private.encoded)
        }

        val publicString = if (encoding == Encoding.PEM) {
            toPem("PUBLIC KEY", public.encoded)
        } else {
            toBase58(public.encoded)
        }
        return KeyPair.newBuilder().setPublicKey(publicString)
            .setPrivateKey(privateString).build()
    }

    private fun toBase58(byteArray: ByteArray): String {
        return Base58.encode(byteArray)
    }


    fun getJvmKeyPair(keyPair: KeyPair, type: KeyType): java.security.KeyPair {
        val publicKey = getPublicKey(keyPair, type)
        val privateKey = getPrivateKey(keyPair, type)
        return java.security.KeyPair(publicKey, privateKey)
    }

    fun getPublicKey(keyPair: KeyPair, type: KeyType): PublicKey {
        val public = keyPair.publicKey
        val encoded: ByteArray? = if (public.contains("PUBLIC")) {
            // this is pem
            getPublicKeyFromPem(public)
        } else {
            // this is base 58
            Base58.decode(public)
        }

        val typeString = if (type == KeyType.RSA) {
            "RSA"
        } else {
           EC_KEYSTORE
        }

        val keyFactory = KeyFactory.getInstance(typeString)
        val keySpec = X509EncodedKeySpec(encoded)
        return keyFactory.generatePublic(keySpec) as PublicKey
    }

    // needs to support pem and base 58
    fun getPrivateKey(keyPair: KeyPair, type: KeyType): PrivateKey {
        val privateKey = keyPair.privateKey
        val encoded: ByteArray? = if (privateKey.contains("PRIVATE")) {
            // this is pem
            getPrivateKeyFromPem(privateKey)
        } else {
            // this is base 58
            Base58.decode(privateKey)
        }

        val typeString = if (type == KeyType.RSA) {
            RSA_KEYSTORE
        } else {
            EC_KEYSTORE
        }

        val keyFactory = KeyFactory.getInstance(typeString)
        val keySpec = PKCS8EncodedKeySpec(encoded)
        return keyFactory.generatePrivate(keySpec) as PrivateKey
    }
}

enum class KeyType {
    RSA,
    ECC
}

enum class Encoding {
    PEM,
    BASE58
}