package id.unum.crossPlatformInterfaces

import id.unum.Base58
import id.unum.protos.crypto.v1.EncryptedData
import id.unum.protos.crypto.v1.EncryptedKey
import org.bouncycastle.jce.provider.BouncyCastleProvider
import java.security.*
import java.security.interfaces.RSAPublicKey
import java.security.spec.X509EncodedKeySpec
import java.util.*
import javax.crypto.Cipher
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

internal object JvmEncryption {

    fun encryptData(data: ByteArray, publicKey: PublicKey?): String {
        val encoded = getCipher(publicKey = publicKey).doFinal(data)
        return Base58.encode(encoded)
    }

    fun decryptData(data: String, privateKey: PrivateKey?): ByteArray? {
        try {
            val array = Base58.decode(data)
            val decrypted = getCipher(privateKey = privateKey).doFinal(array)
            var zeroes = 0
            while (decrypted[zeroes] == 0.toByte()) {
                ++zeroes
            }
            return Arrays.copyOfRange(decrypted, zeroes, decrypted.size)
        } catch (e: Exception) {
            println(e.localizedMessage)
            println(e.stackTrace)
            throw e;
        }
    }

    fun decryptWithOptions(data: String, iv: ByteArray, algorithm: String, key: ByteArray): ByteArray? {
        Security.addProvider(
            BouncyCastleProvider()
        )
        val specs = IvParameterSpec(iv)
        val secretKey = SecretKeySpec(key, algorithm)
        val decryptCipher = Cipher.getInstance("AES/CBC/PKCS5Padding")
        decryptCipher.init(Cipher.DECRYPT_MODE, secretKey, specs)
        val results = decryptCipher.doFinal(Base58.decode(data))

        return results
    }

    private fun getCipher(privateKey: PrivateKey? = null, publicKey: PublicKey? = null): Cipher {
        val cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding")
        val encryptMode = publicKey != null
        val mode = if (encryptMode) Cipher.ENCRYPT_MODE else Cipher.DECRYPT_MODE
        if (encryptMode) {
            cipher.init(mode, publicKey)
        } else {
            cipher.init(mode, privateKey)
        }
        return cipher
    }



    private fun encryptWithProvidedKey(
        key: String,
        data: ByteArray
    ): String {
        val cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding")
        cipher.init(Cipher.ENCRYPT_MODE, getPublicKeyFromString(key))
        val encrypted = cipher.doFinal(data)
        return Base58.encode(encrypted)
    }

    internal fun getPublicKeyFromString(originalKey: String): Key {
       val encoded = if (originalKey.contains("PUBLIC KEY")) {
            val publicKeyPEM: String = originalKey
                // Note: Pem encoded keys with RSA in the header/footer have strange behaviors.
                .replace("-----BEGIN RSA PUBLIC KEY-----", "")
                .replace("-----BEGIN PUBLIC KEY-----", "")
                .replace(System.lineSeparator(), "")
                .replace("-----END RSA PUBLIC KEY-----", "")
                .replace("-----END PUBLIC KEY-----", "")

            Base64.getDecoder().decode(publicKeyPEM)
        } else {
            Base58.decode(originalKey)
        }

        val keyFactory = KeyFactory.getInstance(RSA_KEYSTORE)

        val keySpec = X509EncodedKeySpec(encoded)
        return keyFactory.generatePublic(keySpec) as RSAPublicKey

    }

    fun encryptWithOptions(
        data: ByteArray,
        iv: ByteArray,
        algorithm: String,
        key: ByteArray,
        did: String,
        pemKey: String
    ): EncryptedData {
        Security.addProvider(
            BouncyCastleProvider()
        )

        val encryptedIv = encryptWithProvidedKey(pemKey, iv)
        val encryptedAlgorithm = encryptWithProvidedKey(pemKey, algorithm.toByteArray())
        val encryptedKey = encryptWithProvidedKey(pemKey, key)
        val encryptedDid = encryptWithProvidedKey(pemKey, did.toByteArray())
        val finalKey = EncryptedKey.newBuilder().setKey(encryptedKey).setAlgorithm(encryptedAlgorithm)
            .setIv(encryptedIv).setDid(encryptedDid).build()

        val specs = IvParameterSpec(iv)
        val secretKey = SecretKeySpec(key, algorithm)
        val encrypt = Cipher.getInstance("AES/CBC/PKCS5Padding")
        encrypt.init(Cipher.ENCRYPT_MODE, secretKey, specs)
        val results = encrypt.doFinal(data)
        val encryptedResults = Base58.encode(results)
        return EncryptedData.newBuilder()
            .setData(encryptedResults)
            .setKey(finalKey)
            .build()
    }
}