package com.nobleworks_software.gradle.gitflow

import java.util.regex.Pattern
import java.util.regex.Matcher

import org.gradle.api.Project

class Version {
    int major
    int minor
    int hotfix

    boolean isSnapshot

    private static Pattern versionRegex = ~/(\d+)\.(\d+)\.(\d+)(-SNAPSHOT)?/

    static Version parseVersion(String versionStr) {
        Matcher m = versionStr =~ versionRegex
        if (!m.matches()) {
            throw new Exception("Failed to parse version from \"${versionStr}\"")
        }

        int major = Integer.parseInt(m.group(1))
        int minor = Integer.parseInt(m.group(2))
        int hotfix = Integer.parseInt(m.group(3))
        boolean isSnapshot = m.group(4) == "-SNAPSHOT"

        return new Version(
            major: major,
            minor: minor,
            hotfix: hotfix,
            isSnapshot: isSnapshot
        )
    }

    static File getVersionFile(Project project) {
        return new File(project.projectDir, project.gitflow.versionFile)
    }

    static Version loadProjectVersion(Project project) {
        return load(getVersionFile(project))
    }

    static Version load(String filename) {
        return load(new File(filename))
    }

    static Version load(File versionFile) {

        if (!versionFile.exists() || !versionFile.isFile()) {
            throw new Exception("Version file ${versionFile.toString()} does not exist. Please add it to your project")
        }

        try {
            def props = new Properties()
            versionFile.withInputStream {
                stream -> props.load(stream)
            }
            return new Version(
                major: Integer.parseInt(props['major']),
                minor: Integer.parseInt(props['minor']),
                hotfix: Integer.parseInt(props['hotfix']),
                isSnapshot: Boolean.parseBoolean(props['isSnapshot'])
            )
        }
        catch (Exception e) {
            throw new Exception(
                "Failed to parse ${versionFile.toString()} as a valid Version file",
                e
            )
        }

    }

    String toString() {
        String result = "${major}.${minor}.${hotfix}"
        if (isSnapshot) {
            result += "-SNAPSHOT"
        }
        return result
    }

    Version removeSnapshot() {
        return new Version(
            major: major,
            minor: minor,
            hotfix: hotfix,
            isSnapshot: false
        )
    }

    Version bumpHotfix() {
        return new Version (
            major: major,
            minor: minor,
            hotfix: hotfix+1,
            isSnapshot: isSnapshot
        )
    }

    Version bumpMinor() {
        return new Version (
            major: major,
            minor: minor+1,
            hotfix: 0,
            isSnapshot: isSnapshot
        )
    }

    Version bumpMajor() {
        return new Version (
            major: major+1,
            minor: 0,
            hotfix: 0,
            isSnapshot: isSnapshot
        )
    }

    static void changeProjectVersion(Project project, Version oldVersion, Version newVersion) {
        File versionFile = getVersionFile(project)
        newVersion.writeToFile(versionFile)
        project.versionChangeEvent.fire(
            new VersionChangeEvent(oldVersion, newVersion)
        )
    }

    private void writeToFile(String filename) {
        writeToFile(new File(filename))
    }

    private void writeToFile(File propFile) {
        def props = new Properties()
        props.setProperty('major', ''+major)
        props.setProperty('minor', ''+minor)
        props.setProperty('hotfix', ''+hotfix)
        props.setProperty('isSnapshot', ''+isSnapshot)

        OutputStream out = new FileOutputStream(propFile)
        props.store(out, "suppress inspection \"UnusedProperty\" for whole file")
    }
}
