package com.nobleworks_software.gradle.gitflow

import org.gradle.api.tasks.TaskAction

class MergeToMasterTask extends BaseGitflowTask {

    @TaskAction
    def run() {
        GitHelper.ensureBranchMatches('(release/.*)|(hotfix/.*)', "You must be on a release or hotfix branch to run this task")
        GitHelper.ensureClean()

        String originalBranch = GitHelper.getBranchName()
        String branchToMerge = originalBranch
        Version version = getVersion()
        if (version.isSnapshot) {
            // release branches typically will not have the SNAPSHOT flag,
            // but hotfix branches typically will not

            branchToMerge = "gitflow-tmp-" + makeRandomString(8)
            GitHelper.createAndCheckout(branchToMerge)

            Version newVersion = version.removeSnapshot()
            Version.changeProjectVersion(project, version, newVersion)
            GitHelper.addAll()
            GitHelper.commit("gitflow-plugin: removing SNAPSHOT flag before merge-to-master")

            version = newVersion
        }
        String tagName = computeTagName(version)

        // Checkout master and attempt to merge in the branch
        // If there are merge conflicts, run `git merge --abort`,
        // return to the original branch, and throw a (hopefully
        // useful) exception.
        GitHelper.checkoutExisting('master')
        GitHelper.ensureNotBehind() // make sure master is at least caught-up with origin/master. It's ok if we're ahead though (it'll all get pushed later)

        try {
            // Couldn't come up with a short name that showcased all of the options
            // we're using here. Decided to just leave the entire command out
            // in the open instead wrapping it up in a nice GitHelper command.
            //
            // http://git-scm.com/docs/git-merge
            //
            // The --strategy-option=theirs option is particularly important here.
            // From the docs:
            //
            // > ours
            // > This option forces conflicting hunks to be auto-resolved cleanly by
            // > favoring our version. Changes from the other tree that do not conflict
            // > with our side are reflected to the merge result. For a binary file,
            // > the entire contents are taken from our side.
            // >
            // > This should not be confused with the ours merge strategy, which does
            // > not even look at what the other tree contains at all. It discards
            // > everything the other tree did, declaring our history contains all that
            // > happened in it.
            // >
            // > theirs
            // > This is the opposite of ours.
            //
            // Here's an example of why this is necessary:
            // A project uses the onVersionChange event to update their README.md
            // every time the version changes. Since there will always be a merge
            // conflict on the versionFile in a mergeToMaster process, now
            // there will always be a merge conflict on README.md as well. Previous
            // solutions solved this by using the entire README.md file from
            // the incoming branch ('theirs', since we're in master when we do
            // the merge). However, the project could have changed README.md in
            // a hotfix branch and this would overwrite their changes (ideally
            // they would merge those changes into develop as well, but I don't want
            // to preclude users from keeping persistent changes in master)
            GitHelper.runCmd([
                'git', 'merge',
                '--no-ff',
                '--strategy-option=theirs',
                '-m', "gitflow-plugin: merge $originalBranch to master",
                branchToMerge
            ])
        }
        catch (Exception e) {
            GitHelper.mergeAbort()
            GitHelper.checkoutExisting(originalBranch)
            writeFormattedMessage(
                'MERGE FAILED:',
                "Failed to merge $originalBranch into master.",
                'Merge has been aborted and the previous state has been restored to the best of our ability.',
                'When you have fixed the merge issues/conflicts you should re-run this task. It is not',
                'recommended to manually merge into master. For more details, see:',
                'http://stash.lasp.colorado.edu/projects/WEBAPPS/repos/gradle-gitflow-plugin/browse',
                '',
                e.getMessage()
            )
            throw new Exception("Failed to merge $originalBranch into master", e)
        }
        finally {
            // If we made a tmp branch, delete it
            if (branchToMerge != originalBranch) {
                GitHelper.deleteBranch(branchToMerge)
            }
        }

        GitHelper.tagAnnotated(tagName, "gitflow-plugin: tagging release ${version.toString()}")

        project.masterMergeEvent.fire(new MasterMergeEvent())

        String pushCommand = 'git push origin master && git push origin --tags'
        String undoCommand = "git reset --hard HEAD~1; git tag -d $tagName; git checkout $originalBranch"
        String returnCommand = "git checkout $originalBranch"
        writeFormattedMessage(
            "IMPORTANT:",
            'You are now on branch master. A new merge commit has been added',
            'and tagged but not pushed. No merge conflicts were found, but',
            'you still must manually review the changes. If you approve, you',
            'must push to origin. Remember that pushing tags requires a ',
            'special command.',
            '',
            "To push: `${pushCommand}`",
            "To undo: `${undoCommand}`",
            "To return to release branch: `${returnCommand}`"
        )
    }

    protected String computeTagName(Version version) {
        return computeVersionTemplate(version, project.gitflow.tagNameTemplate)
    }

    protected String makeRandomString(int len) {
        Random rand = new Random()
        String alphabet =
            //noinspection SpellCheckingInspection
            "abcdefghijklmnopqrstuvwxyz"
        return (1..len).collect { alphabet[ rand.nextInt( alphabet.length() ) ] }.join()
    }
}

