package com.nobleworks_software.gradle.gitflow

import org.gradle.api.tasks.TaskAction

class MakeReleaseBranchTask extends BaseGitflowTask {

    @TaskAction
    def run() {
        // Ensure the current branch is in the proper state
        GitHelper.ensureDevBranch()
        GitHelper.ensureClean()

        // Get original version, compute new versions for
        // the release branch, and for the develop branch
        // going forward
        Version origVersion = getVersion()
        Version releaseVersion = origVersion.removeSnapshot()
        Version nextDevVersion = origVersion.bumpMinor()

        // While we're still on this commit, make the release
        // branch (but don't check it out yet; we'll do that later)
        String branchName = computeReleaseBranchName(releaseVersion)
        GitHelper.createBranch(branchName)

        // Add a new commit to the develop branch that updates the version
        // file.
        Version.changeProjectVersion(project, origVersion, nextDevVersion)
        GitHelper.addAll() // project may change files during onVersionChange event
        GitHelper.commit("gitflow-plugin: automatically bumping develop version to ${nextDevVersion.toString()}")

        // Checkout the release branch and add a commit that updates
        // the version file there, too
        GitHelper.checkoutExisting(branchName)
        Version.changeProjectVersion(project, origVersion, releaseVersion)
        GitHelper.addAll() // project may change files during onVersionChange event
        GitHelper.commit("gitflow-plugin: making release branch ${branchName}")

        // Notify the user that we're not done yet. Provide them
        // with the appropriate git commands to run, depending
        // on how they'd like to proceed.
        String pushCommand = "git push -u origin develop $branchName"
        String undoCommand = "git checkout develop; git reset --hard origin/develop; git branch -D $branchName"
        writeFormattedMessage(
            "IMPORTANT:",
            "Release branch $branchName has been created, and you are now on",
            'that branch. The develop branch also has a new commit that bumps its version',
            "to ${nextDevVersion.toString()}. If you would prefer a different version",
            'number for the next release, you can commit a change to the develop branch\'s',
            'version.properties file at any time.',
            '',
            'You must manually review the changes in both branches. If you approve,',
            'then you must push *both* branches to origin.',
            '',
            "To push both branches: `${pushCommand}`",
            "To undo everything: `${undoCommand}`"
        )
    }

    protected String computeReleaseBranchName(Version version) {
        return computeVersionTemplate(version, project.gitflow.releaseBranchNameTemplate)
    }
}