package com.nobleworks_software.gradle.gitflow

import org.gradle.api.tasks.TaskAction

class MakeHotfixBranchTask extends BaseGitflowTask {

    @TaskAction
    def run() {
        // Ensure the current branch is in the proper state
        GitHelper.ensureMasterBranch()
        GitHelper.ensureClean()

        // Get original version
        Version origVersion = getVersion()

        Version hotfixVersion = origVersion.bumpHotfix()
        hotfixVersion.isSnapshot = true
        String branchName = computeHotfixBranchName(hotfixVersion)
        GitHelper.createAndCheckout(branchName)

        Version.changeProjectVersion(project, origVersion, hotfixVersion)
        GitHelper.addAll() // project may change files in version-change helper
        GitHelper.commit("gitflow-plugin: making hotfix branch ${hotfixVersion.toString()}")

        String undoCommand = "git checkout master; git branch -D $branchName"
        String afterFixCommand = './gradlew mergeToMaster'
        writeFormattedMessage(
            "IMPORTANT:",
            "You are now on branch $branchName and your version has been",
            "bumped to ${hotfixVersion.toString()}. Make any changes you need to fix the bug,",
            "then commit, push, run `${afterFixCommand}`",
            'from this branch. Don\'t forget to merge to develop as well,',
            'when you\'re done, and to the current release branch if there',
            'is one.',
            '',
            "To undo: `${undoCommand}`",
            "After committing your fix: `${afterFixCommand}`"
        )
    }

    protected String computeHotfixBranchName(Version version) {
        Version unSnapshotVersion = version.removeSnapshot()
        return computeVersionTemplate(unSnapshotVersion, project.gitflow.hotfixBranchNameTemplate)
    }
}