package com.nobleworks_software.gradle.gitflow

import org.gradle.api.Project
import org.gradle.api.Plugin

import groovy.text.SimpleTemplateEngine

class GitflowPlugin implements Plugin<Project> {

    private Project project

    private Version getVersion() {
        String filename = project.gitflow.versionFile
        return Version.load(filename)
    }

    void apply(Project project) {

        this.project = project

        project.extensions.create('gitflow', GitflowExtension)

        project.ext.versionChangeEvent = new GitflowEvent<VersionChangeEvent>(name: 'version-change')
        project.ext.masterMergeEvent = new GitflowEvent<MasterMergeEvent>(name: 'master-merge')

        // Only add the tasks if we're the root project: we don't want to accidentally
        // change the versionFile on a sub-project if we're actually trying to change
        // the versionFile on the root project.
        //
        // Note, however, that we do need to add the non-task properties because
        // the project's build script may fail to execute if they're missing/null
        if (project.rootProject == project) {
            addProjectTasks(project)
        }
    }

    private void addProjectTasks(project) {
        project.task('__ensureClean__').doLast {
            GitHelper.ensureClean()
        }

        project.task('__ensureDev__').doLast {
            GitHelper.ensureDevBranch()
        }

        project.task('makeReleaseBranch', type: MakeReleaseBranchTask) {
            description 'Make a new release branch. Update and commit version.properties in both develop and release branch.'
        }

        project.task('makeHotfixBranch', type: MakeHotfixBranchTask) {
            description 'Make a new hotfix branch from master. Add commit that sets version to \'major.minor.(hotfix+1)-SNAPSHOT\''
        }

        project.task('mergeToMaster', type: MergeToMasterTask) {
            description 'Merge the current hotfix or release branch to master, and tag the resulting commit. Will remove the SNAPSHOT flag if set.'
        }

        project.task('bumpMajor', type: WriteVersionTask) {
            description 'Update the version to (major+1).0.0-SNAPSHOT. Repo must be on develop, up-to-date, and clean'
            dependsOn: ['__ensureClean__', '__ensureDev__']
            doFirst {
                newVersion = getVersion().bumpMajor()
            }
        }

        project.task('version').doLast {
            project.logger.quiet("\nPROJET VERSION (via gitflow plugin): ${getVersion().toString()}")
        }

        project.tasks.addRule("Pattern: setVersion<Version>") { String taskName ->
            if (taskName.startsWith("setVersion")) {
                project.task(taskName, type: WriteVersionTask) {
                    doFirst {
                        File versionFile = new File(project.gitflow.versionFile)
                        if (!versionFile.exists()) {
                            versionFile.createNewFile()
                        }

                        String branch = GitHelper.getBranchName()
                        if (branch == "develop" || branch.startsWith("feature/")) {
                            newVersion.isSnapshot = true
                        }
                        else {
                            throw new Exception("You must be on develop or a feature branch to run this task")
                        }
                    }
                    newVersion = Version.parseVersion(taskName - "setVersion")
                    commitMessageTemplate = 'setting version to $newVersion'
                    dependsOn: ['__ensureClean__', '__ensureDev__']
                }
            }
        }
    }
}