package com.nobleworks_software.gradle.gitflow

import java.util.List
import java.util.regex.Pattern

class GitHelper {

    private static String trimSuffix(String str, String suffix) {
        if (str.endsWith(suffix)) {
            return str.substring(0, str.length() - suffix.length())
        }
        else {
            return str
        }
    }

    /**
     * Run the command specified by the passed strings
     * (which will be concatenated with spaces before
     * being executed)
     *
     * An exception will be thrown if the exit code of
     * the command is not 0 (means the command had an
     * error).
     *
     * @param cmd
     * @return
     */
    private static String runCmd(List<String> cmd) {
        def stdout = new StringBuffer(), stderr = new StringBuffer()
        def process = cmd.execute()
        process.consumeProcessOutput(stdout, stderr)
        process.waitForOrKill(1000)
        int exitVal = process.exitValue()

        def stdOutStr = trimSuffix(stdout.toString(), '\n')
        def stdErrStr = trimSuffix(stderr.toString(), '\n')

        if (exitVal != 0) {
            throw new Exception("Error executing command:\n'${cmd}'\nstdout:\n'${stdOutStr}'\nstderr:\n'${stdErrStr}'\n----------")
        }
        else {
            return stdOutStr
        }
    }

    static void deleteBranch(String name) {
        runCmd(['git', 'branch', '-D', name])
    }

    static void resolveConflictTheirs(String filename) {
        // http://gitready.com/advanced/2009/02/25/keep-either-file-in-merge-conflicts.html
        runCmd(['git', 'checkout', '--theirs', filename])
    }

    static void resolveConflictOurs(String filename) {
        // http://gitready.com/advanced/2009/02/25/keep-either-file-in-merge-conflicts.html
        runCmd(['git', 'checkout', '--ours', filename])
    }

    static String[] getConflictedFiles() {
        String result = runCmd(['git', 'diff', '--name-only', '--diff-filter=U']).trim()
        if (result == "") {
            return []
        }
        else {
            return result.split("\n")
        }
    }

    static void createBranch(String branchName) {
        runCmd(['git', 'branch', branchName])
    }

    static void createBranchAndTrackOrigin(String branchName) {
        runCmd(['git', 'branch', branchName, '--track', "origin/${branchName}"])
    }

    static void createAndCheckout(String branchName) {
        runCmd(['git', 'checkout', '-b', branchName])
    }

    static void checkoutExisting(String branchName) {
        runCmd(['git', 'checkout', branchName])
    }

    static void add(File file) {
        runCmd(['git', 'add', file.toString()])
    }

    static void add(String fileName) {
        runCmd(['git', 'add', fileName])
    }

    static void addAll() {
        runCmd(['git', 'add', '-A'])
    }

    /**
     * This variant of commit is only used
     * when dealing with merge conflicts (AFAICT)
     */
    static void commit() {
        runCmd(['git', 'commit', '--no-edit'])
    }

    static void commit(String commitMessage) {
        runCmd(['git', 'commit', '-m', commitMessage])
    }

    static void mergeNoFf(String branchName, String msg = null) {
        List<String> cmd = ['git', 'merge', '--no-edit', '--no-ff', branchName]
        if (msg != null) {
            cmd += ['-m', msg]
        }
        runCmd(cmd)
    }

    static void mergeAbort() {
        runCmd(['git', 'merge', '--abort'])
    }

    static void tagAnnotated(String tagName, String tagMessage) {
        runCmd(['git', 'tag', '-a', tagName, '-m', tagMessage])
    }

    static String getBranchName() {
        return runCmd(['git', 'rev-parse', '--abbrev-ref', 'HEAD'])
    }

    static void ensureMasterBranch() {
        ensureBranchMatches('master', 'You must be on branch \'master\' to run this task')
    }

    static void ensureDevBranch() {
        ensureBranchMatches('develop', 'You must be on branch \'develop\' to run this task')
    }

    static void ensureBranchMatches(String regex, String errMsg) {
        ensureBranchMatches(~regex, errMsg)
    }

    static void ensureBranchMatches(Pattern regex, String errMsg) {
        def branchName = getBranchName()
        if (!(branchName =~ regex)) {
            throw new Exception(errMsg)
        }
    }

    static int numCommitsBehind() {
        String currentBranch = getBranchName()
        String isBehindOutput = runCmd(['git', 'log', '--oneline', "${currentBranch}..origin/${currentBranch}"])
        if (isBehindOutput == "") {
            return 0
        }
        else {
            return isBehindOutput.split('\n').length
        }
    }

    static int numCommitsAhead() {
        String currentBranch = getBranchName()
        String isAheadOutput = runCmd(['git', 'log', '--oneline', "origin/${currentBranch}..${currentBranch}"])
        if (isAheadOutput == "") {
            return 0
        }
        else {
            return isAheadOutput.split('\n').length
        }
    }

    static void ensureNotBehind() {
        int numBehind = numCommitsBehind()
        if (numBehind > 0) {
            int numAhead = numCommitsAhead()
            String branch = getBranchName()
            if (numAhead > 0) {
                throw new Exception("Unable to proceed: $branch is $numAhead commits ahead and $numBehind behind origin/${branch}")
            }
            else {
                throw new Exception("$branch is $numBehind commits behind origin/$branch and can be fast-forwarded. Consider running `git pull`")
            }
        }
    }

    static void ensureNotAhead() {
        int numAhead = numCommitsAhead()
        if (numAhead > 0) {
            int numBehind = numCommitsBehind()
            String branch = getBranchName()
            if (numBehind > 0) {
                throw new Exception("Unable to proceed: $branch is $numAhead commits ahead and $numBehind behind origin/${branch}")
            }
            else {
                throw new Exception("$branch is $numAhead commits ahead of origin/$branch. Consider running `git push`")
            }
        }
    }

    static void ensureClean() {

        String nothingToCommitOutput = runCmd(['git', 'status', '--porcelain'])
        boolean nothingToCommit = nothingToCommitOutput == ''

        if (!nothingToCommit) {
            throw new Exception('You must commit and push your changes before running this task')
        }

        String currentBranch = getBranchName()

        try {
            // Attempts to print the current remote tracking branch.
            // If it fails, it will return a non-zero status code.
            // The non-zero status code will trigger an Exception
            // in our code, which we will catch here and format
            // nicely
            runCmd(['git', 'rev-parse', '--abbrev-ref', '--symbolic-full-name', '@{u}'])
        }
        catch (Exception ignored) {
            throw new Exception(
                "Unable to find remote tracking branch for $currentBranch. Try running `git push -u origin ${currentBranch}`"
            )
        }

        int numBehind = numCommitsBehind()
        boolean isBehind = numBehind > 0

        int numAhead = numCommitsAhead()
        boolean isAhead = numAhead > 0

        if (isBehind || isAhead) {
            List<String> msgs = []
            if (isAhead) {
                msgs.add("ahead by $numAhead commits")
            }

            if (isBehind) {
                msgs.add("behind by $numBehind commits")
            }

            String subMsgs = msgs.join(' and ')
            String msg = "Branch $currentBranch must be synced with origin. You are $subMsgs"
            throw new Exception(msg)
        }
    }
}
