package uk.co.mruoc.postman.newman;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import uk.co.mruoc.postman.settings.NewmanSettings;
import org.gradle.api.Project;

import java.io.File;
import java.util.Map;

import static uk.co.mruoc.postman.task.util.Json.array;
import static uk.co.mruoc.postman.task.util.Json.empty;
import static uk.co.mruoc.postman.task.util.Json.object;
import static uk.co.mruoc.postman.task.util.Json.primitive;

class NewmanConfig {

    private final Project project;
    private final NewmanSettings settings;

    NewmanConfig(Project project, NewmanSettings settings) {
        this.project = project;
        this.settings = settings;
    }

    String toJsonFor(File collection) {
        return new JsonBuilder(collection).build();
    }

    private class JsonBuilder {

        private final JsonObject params = object();
        private final File collection;

        JsonBuilder(File collection) {
            this.collection = collection;
        }

        private String build() {
            buildParameters();
            return params.toString();
        }

        private void buildParameters() {
            addCollection();
            addEnvironment();
            addGlobals();
            addReporters();
            addBail();
            addNoColor();
            addDisableUnicode();
            addSecure();
            addIgnoreRedirects();
            addEnvVars();
            addGlobalVars();
        }

        private void addCollection() {
            params.add("collection", primitive(collection.toString()));
        }

        private void addReporters() {
            JsonArray reporters = array();
            JsonObject reporter = object();

            addCli(reporters);
            addJunit(reporters, reporter);
            addJson(reporters, reporter);
            addHtml(reporters, reporter);

            if (!empty(reporters)) {
                params.add("reporters", reporters);
            }

            if (!empty(reporter)) {
                params.add("reporter", reporter);
            }
        }

        private void addJunit(JsonArray reporters, JsonObject reporter) {
            if (settings.getXmlReportDir() != null) {
                reporters.add(primitive("junit"));
                reporter.add(
                    "junit",
                    object(
                        "export",
                        projectFile(settings.getXmlReportDir(), "TEST-postman-" + collection.getName() + ".xml")));
            }
        }

        private void addJson(JsonArray reporters, JsonObject reporter) {
            if (settings.getJsonReportDir() != null) {
                reporters.add("json");
                reporter.add(
                    "json",
                    object(
                        "export",
                        projectFile(settings.getJsonReportDir(), endsWithJson("TEST-postman-" + collection.getName()))));
            }
        }

        private void addHtml(JsonArray reporters, JsonObject reporter) {
            if (settings.getHtmlReportDir() != null) {
                reporters.add("html");

                JsonObject htmlOptions = object();
                reporter.add("html", htmlOptions);

                htmlOptions.addProperty(
                    "export",
                    projectFile(settings.getHtmlReportDir(), "TEST-postman-" + collection.getName() + ".html"));

                if (settings.getHtmlTemplate() != null) {
                    htmlOptions.addProperty(
                        "template",
                        projectFile(settings.getHtmlTemplate()));
                }
            }
        }

        private String projectFile(String... parts) {
            return compose(project.getProjectDir(), parts).toString();
        }

        private File compose(File file, String[] parts) {
            for (String part : parts) {
                file = new File(file, part);
            }
            return file;
        }

        private void addCli(JsonArray reporters) {
            if (settings.getCliReport()) {
                reporters.add(primitive("cli"));
            }
        }

        private void addEnvironment() {
            if (settings.getEnvironment() != null) {
                params.add("environment", new JsonPrimitive(settings.getEnvironment().toString()));
            }
        }

        private void addGlobals() {
            if (settings.getGlobals() != null) {
                params.add("globals", new JsonPrimitive(settings.getGlobals().toString()));
            }
        }

        private void addEnvVars() {
            addVars("envVar", settings.getEnvVars());
        }

        private void addGlobalVars() {
            addVars("globalVar", settings.getGlobalVars());
        }

        private void addVars(String name, Map<String, String> values) {
            if (values == null || values.isEmpty()) {
                return;
            }
            params.add(name, toVars(values));
        }

        private JsonArray toVars(Map<String, String> values) {
            JsonArray vars = array();
            values.entrySet().stream()
                    .map(e -> toVar(e.getKey(), e.getValue()))
                    .forEach(vars::add);
            return vars;
        }

        private JsonObject toVar(String key, String value) {
            JsonObject var = object();
            var.addProperty("key", key);
            var.addProperty("value", value);
            return var;
        }

        private void addBail() {
            params.add("bail", primitive(settings.getStopOnError()));
        }

        private void addNoColor() {
            params.add("noColor", primitive(settings.getNoColor()));
        }

        private void addDisableUnicode() {
            params.add("disableUnicode", primitive(settings.getDisableUnicode()));
        }

        private void addSecure() {
            params.add("insecure", primitive(!settings.getSecure()));
        }

        private void addIgnoreRedirects() {
            params.add("ignoreRedirects", primitive(settings.getIgnoreRedirects()));
        }

        private String endsWithJson(String fileName) {
            if (fileName.endsWith(".json")) {
                return fileName;
            }
            return fileName + ".json";
        }
    }


}
