package gov.raptor.gradle.plugins.buildsupport

import groovy.transform.ToString
import org.gradle.api.GradleException
import org.gradle.api.InvalidUserDataException
import org.gradle.api.Project
import org.gradle.api.artifacts.repositories.PasswordCredentials

/**
 * DSL support to allow build scripts to configure the repos to be used for fetching dependencies and
 * publishing artifacts produced by the build.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.4
 */
@SuppressWarnings("GroovyUnusedDeclaration")
@ToString(includeNames = true, includePackage = false, includeFields = true)
class RepoManagement {

    private static final String CTI_PUBLIC_NEXUS_PATTERN = 'https://nexus.ctic-inc.com/nexus/content/repositories/%s-%s'
    private static final String CTI_PUBLIC_NEXUS = 'https://nexus.ctic-inc.com/nexus/content/groups/public'
    private static final String CTI_PRIVATE_NEXUS_PATTERN = 'http://mvn.ctic-dev.com/nexus/content/repositories/%s'
    private static final String CTI_PRIVATE_NEXUS = 'http://mvn.ctic-dev.com/nexus/content/groups/public'

    private static final String DI2E_BASE_URL = 'https://nexus.di2e.net/nexus/content/'
    private static final String DI2E_CENTRAL_NEXUS = DI2E_BASE_URL + 'repositories/central'
    private static final String DI2E_PUBLISH_PATTERN = DI2E_BASE_URL + 'repositories/Private_%s_%s'
    private static final String DI2E_RAPTOR_PUBLIC_NEXUS = DI2E_BASE_URL + 'groups/RAPTORX'
    private static final String DI2E_RAPTOR_PUBLISH_PATTERN = DI2E_BASE_URL + 'repositories/Private_RAPTORX_%s'

    private static final String DI2E_NAME = 'di2e'
    private static final String DI2E_RAPTORX_NAME = 'di2eRaptorx'

    private final Project project
    private List<RepoConfig> repoConfigs = new ArrayList<>()
    private boolean noMavenLocal = false
    private boolean includeGoogle = false

    RepoManagement(Project project) {
        this.project = project

        // Support use of -PnoMavenLocal (implied by runningUnderJenkins)
        if (project.findProperty('noMavenLocal') != null || project.runningUnderJenkins) noMavenLocal = true

        // Spit out the known credential keys, if requested
        if (project.hasProperty('dumpCredentials')) {
            // This is a bit hackish (going after internals), but this is only used when debugging credential issues
            // and I'll just have to deal with it if the internal implementation of the credentials ever changes.
            project.logger.lifecycle "Known credential keys: ${project.credentials.credentials.stringPropertyNames()}"
        }
    }

    /**
     * @return {@code true} if the {@link #noMavenLocal()} configuration has been specified
     */
    boolean isNoMavenLocal() {
        return noMavenLocal
    }

    /**
     * Include {@code google( )} in the repository configuration.
     */
    void google() {
        includeGoogle = true
    }

    /**
     * @return {@code true} if google should be included in the repository configuration
     */
    boolean includeGoogle() {
        return includeGoogle
    }

    /**
     * @return All repo configurations
     */
    List<RepoConfig> getRepoConfigs() {
        return repoConfigs
    }

    /**
     * @return All repo configurations where 'publish' is set true
     */
    List<RepoConfig> getPublishingRepoConfigs() {
        return repoConfigs.findAll { it.publish }
    }

    /**
     * Disallow the use of maven local for dependency resolution.
     */
    void noMavenLocal() {
        noMavenLocal = true
    }

    /**
     * Retained for backward compatibility.  Issues a warning and then redirects to configure {@code di2eRaptorx}.
     *
     * @param props Configuration properties
     */
    void raptorx(Map props) {
        project.logger.warn("Use of the STL Nexus repo has been deprecated; Configuring di2eRaptorx instead")
        addRepoConfig(new Di2eRaptorxRepoConfig(project, props))
    }

    /**
     * Add a DI2E RaptorX configuration accepting a map of configuration properties.  Supported properties are:
     * <ul>
     *     <li>{@code publish} - boolean value, true if publishing should be configured for this repo, default {@code false}</li>
     *     <li>{@code username} - (optional) specify a username for this repo</li>
     *     <li>{@code password} - (optional) specify a password for this repo</li>
     * </ul>
     *
     * Credentials are typically extracted from the credentials store on the local machine or from an environment
     * variable.  For the DI2E RaptorX repo, the credential store keys are {@code di2eRepoUsername} and {@code di2eRepoPassword}.
     * The environment variables {@code DI2E_NEXUS_USERNAME} and {@code DI2E_NEXUS_PASSWORD} can be used to specify
     * the repo access credentials.
     *
     * @param props Configuration properties
     * @since 0.5
     */
    void di2eRaptorx(Map props) {
        addRepoConfig(new Di2eRaptorxRepoConfig(project, props))
    }

    /**
     * @return The RepoConfig for the DI2E Raptorx repo.  Null if not configured.
     * @since 0.7.2
     */
    RepoConfig getDi2eRaptorx() {
        return findRepoConfig(Di2eRaptorxRepoConfig.class)
    }

    /**
     * Add a DI2E Central configuration accepting a map of configuration properties.  Supported properties are:
     * <ul>
     *     <li>{@code username} - (optional) specify a username for this repo</li>
     *     <li>{@code password} - (optional) specify a password for this repo</li>
     * </ul>
     *
     * Credentials are typically extracted from the credentials store on the local machine or from an environment
     * variable.  For the DI2E Central repo, the credential store keys are {@code di2eRepoUsername} and {@code di2eRepoPassword}.
     * The environment variables {@code DI2E_NEXUS_USERNAME} and {@code DI2E_NEXUS_PASSWORD} can be used to specify
     * the repo access credentials.
     *
     * @param props Configuration properties
     */
    void di2eCentral(Map props) {
        addRepoConfig(new Di2eCentralRepoConfig(project, props))
    }

    /**
     * @return The RepoConfig for the DI2E Central repo.  Null if not configured.
     * @since 0.7.2
     */
    RepoConfig getDi2eCentral() {
        return findRepoConfig(Di2eCentralRepoConfig.class)
    }

    /**
     * Add a DI2E private repo configuration accepting a map of configuration properties.  Supported properties are:
     * <ul>
     *     <li>{@code name} - The name/tag of the repo. This is used to determine repo URLs and credentials</li>
     *     <li>{@code publish} - boolean value, true if publishing should be configured for this repo, default {@code false}</li>
     *     <li>{@code username} - (optional) specify a username for this repo</li>
     *     <li>{@code password} - (optional) specify a password for this repo</li>
     * </ul>
     *
     * Credentials are typically extracted from the credentials store on the local machine or from an environment
     * variable.  For a DI2E private repo, the credential store keys are {@code [name]RepoUsername} and
     * {@code [name]RepoPassword}, with a fallback to {@code di2eRepoUsername} and {@code di2eRepoPassword}.
     * The environment variables used are {@code [name]_NEXUS_USERNAME} and {@code [name]_NEXUS_PASSWORD}, with a fallback
     * to {@code DI2E_NEXUS_USERNAME} and {@code DI2E_NEXUS_PASSWORD}.
     *
     * @param props Configuration properties
     */
    void di2ePrivate(Map props) {
        addRepoConfig(new Di2ePrivateRepoConfig(project, props))
    }

    /**
     * @param name The name of the repo to find
     * @return The RepoConfig for the DI2E private repo with the given name.  Null if not configured.
     * @since 0.7.2
     */
    RepoConfig getDi2ePrivate(String name) {
        return findRepoConfig(Di2ePrivateRepoConfig.class, name)
    }

    /**
     * Add a CTI Public repo configuration accepting a map of configuration properties.  Supported properties are:
     * <ul>
     *     <li>{@code name} - The name/tag of the repo. This is used to determine repo URLs</li>
     *     <li>{@code publish} - boolean value, true if publishing should be configured for this repo, default {@code false}</li>
     *     <li>{@code username} - (optional) specify a username for this repo</li>
     *     <li>{@code password} - (optional) specify a password for this repo</li>
     * </ul>
     *
     * Credentials are typically extracted from the credentials store on the local machine or from an environment
     * variable.  For a CTI public repo, the credential store keys are {@code publicRepoUsername} and
     * {@code publicRepoPassword}.  The environment variables used are {@code CTI_PUBLIC_NEXUS_USERNAME} and
     * {@code CTI_PUBLIC_NEXUS_PASSWORD}.
     *
     * @param props Configuration properties
     */
    void ctiPublic(Map props) {
        addRepoConfig(new CTIPublicRepoConfig(project, props))
    }

    /**
     * @param name The name of the repo to find
     * @return The RepoConfig for the CTI public repo with the given name.  Null if not configured.
     * @since 0.7.2
     */
    RepoConfig getCtiPublic(String name) {
        return findRepoConfig(CTIPublicRepoConfig.class, name)
    }

    /**
     * Add a CTI Private repo configuration accepting a map of configuration properties.  Supported properties are:
     * <ul>
     *     <li>{@code name} - The name/tag of the repo. This is used to determine repo URLs</li>
     *     <li>{@code publish} - boolean value, true if publishing should be configured for this repo, default {@code false}</li>
     *     <li>{@code username} - (optional) specify a username for this repo</li>
     *     <li>{@code password} - (optional) specify a password for this repo</li>
     * </ul>
     *
     * Credentials for this repo type are expected to be managed externally, and no standard credentials discovery will
     * be performed.
     *
     * @param props Configuration properties
     */
    void ctiPrivate(Map props) {
        addRepoConfig(new CTIPPrivateRepoConfig(project, props))
    }

    /**
     * @param name The name of the repo to find
     * @return The RepoConfig for the CTI private repo with the given name.  Null if not configured.
     * @since 0.7.2
     */
    RepoConfig getCtiPrivate(String name) {
        return findRepoConfig(CTIPPrivateRepoConfig.class, name)
    }

    /**
     * Find a repo config with the given class and optional name.
     *
     * @param type The class of the config to find
     * @param name The name of the repo to find
     * @return The RepoConfig with the given type and name.  May be null if not configured.
     * @since 0.7.2
     */
    private RepoConfig findRepoConfig(Class type, String name = null) {
        repoConfigs.find { it.class == type && (!name || it.name == name) }
    }

    /**
     * Add a new repo configuration to the list.  Perform basic validation that we don't have two repos with
     * the same name on the same server (determined by the class of the repoConfig object).
     *
     * @param repoConfig Repo config to add
     */
    private void addRepoConfig(RepoConfig repoConfig) {
        boolean exists = repoConfigs.any { it.class == repoConfig.class && it.name == repoConfig.name }

        if (exists) {
            throw new GradleException("Can not configure 2 repos with the same name [${repoConfig.name}]")
        }

        project.logger.info "Add: $repoConfig"

        repoConfigs.add(repoConfig)
    }

    /**
     * Repo configuration for DI2E Central.  This has a fixed name, and  you can not publish to it.
     */
    @ToString(includeNames = true, includePackage = false, includeSuper = true, ignoreNulls = true, allProperties = false)
    private static final class Di2eCentralRepoConfig extends RepoConfig {

        // Don't allow name to be configured and publish is always false
        private static final Map OVERRIDES = [name: DI2E_NAME, publish: false]

        Di2eCentralRepoConfig(Project project, Map props) {
            super(project, DI2E_NAME, props == null ? OVERRIDES : props + OVERRIDES)

            dependencyUrlsGenerator = { -> [DI2E_CENTRAL_NEXUS] }
            publishUrlGenerator = { return null }
        }
    }

    /**
     * Repo configuration for DI2E Raptorx.  Raptor uses a Snapshots and a Releases repo for publishing.
     * The urls and name of this repo are fixed.
     *
     * @since 0.5
     */
    @ToString(includeNames = true, includePackage = false, includeSuper = true, ignoreNulls = true, allProperties = false)
    private static final class Di2eRaptorxRepoConfig extends RepoConfig {

        // Name is always di2e
        private static final Map OVERRIDES = [name: DI2E_RAPTORX_NAME]

        Di2eRaptorxRepoConfig(Project project, Map props) {
            super(project, DI2E_NAME, props == null ? OVERRIDES : props + OVERRIDES)

            dependencyUrlsGenerator = { -> [DI2E_RAPTOR_PUBLIC_NEXUS] }
            publishUrlGenerator = { boolean release -> String.format(DI2E_RAPTOR_PUBLISH_PATTERN, release ? 'Releases' : 'Snapshots') }
        }
    }

    /**
     * Repo configuration for DI2E private repositories.  Private repos are assumed to follow a standard name pattern
     * for Snapshot and Release repos.  The name pattern is:
     * {@code "https://nexus.di2e.net/nexus/content/repositories/Private_${name}_${repoType}"} where
     * repoType is either 'Snapshots' or 'Releases'.
     */
    @ToString(includeNames = true, includePackage = false, includeSuper = true, ignoreNulls = true, allProperties = false)
    private static final class Di2ePrivateRepoConfig extends RepoConfig {

        Di2ePrivateRepoConfig(Project project, Map props) {
            super(project, DI2E_NAME, props)

            dependencyUrlsGenerator = { -> [publishUrlGenerator(true), publishUrlGenerator(false)] }
            publishUrlGenerator = { boolean release -> String.format(DI2E_PUBLISH_PATTERN, name, release ? 'Releases' : 'Snapshots') }
        }
    }

    /**
     * Repo configuration for a named CTI Public repo.  CTI repos are assumed to follow a standard name pattern
     * for Snapshot and Release repos.  The name pattern is:
     * {@code "https://nexus.ctic-inc.com/nexus/content/repositories/${name}-${repoType}"} where
     * repoType is either {@code 'snapshot'} or {@code 'release'}.
     *
     */
    @ToString(includeNames = true, includePackage = false, includeSuper = true, ignoreNulls = true, allProperties = false)
    private static final class CTIPublicRepoConfig extends RepoConfig {

        CTIPublicRepoConfig(Project project, Map props) {
            super(project, props)

            dependencyUrlsGenerator = { -> [CTI_PUBLIC_NEXUS, publishUrlGenerator(true), publishUrlGenerator(false)] }
            publishUrlGenerator = { boolean release -> String.format(CTI_PUBLIC_NEXUS_PATTERN, name, release ? 'release' : 'snapshot') }

            // All public repos use the same credentials
            getEnvVarPrefix = { 'CTI_PUBLIC' }
            getCredentialsPrefix = { 'public' }
        }
    }

    /**
     * Repo configuration for a named CTI Private  repo.  CTI repos are assumed to follow a standard name pattern
     * for Snapshot and Release repos.  The name pattern is:
     * {@code "http://mvn.ctic-dev.com/nexus/content/repositories/${repoType}"} where
     * repoType is either {@code 'snapshots'} or {@code 'releases'}.
     *
     * No credentials are managed for this repo type, and the build must take place within the CTI corporate
     * network (or over a properly configured VPN).
     *
     */
    @ToString(includeNames = true, includePackage = false, includeSuper = true, ignoreNulls = true, allProperties = false)
    private static final class CTIPPrivateRepoConfig extends RepoConfig {

        // Don't allow name to be configured (as none is needed)
        private static final Map OVERRIDES = [name: 'cti']

        CTIPPrivateRepoConfig(Project project, Map props) {
            super(project, props == null ? OVERRIDES : props + OVERRIDES)

            dependencyUrlsGenerator = { -> [CTI_PRIVATE_NEXUS, publishUrlGenerator(true), publishUrlGenerator(false)] }
            publishUrlGenerator = { boolean release -> String.format(CTI_PRIVATE_NEXUS_PATTERN, release ? 'releases' : 'snapshots') }
        }

        @Override
        PasswordCredentials getCredentials() {
            // Credentials have been specified in order to support publishing. Most developers should never need to configure these.
            if (username || password) {
                return super.getCredentials()
            }

            // We need to ignore validation since the credentials are not required to read dependencies
            return null
        }
    }

    /**
     * Base repo configuration.  Tracks values pulled from configuration properties and locates credentials either
     * in the credential store or from the environment.
     */
    @ToString(includeNames = true, includePackage = false, includeFields = true, ignoreNulls = true, allProperties = false,
            excludes = ['project', 'getCredentialsPrefix', 'getEnvVarPrefix', 'publishUrlGenerator',
                    'dependencyUrlsGenerator', 'ignoreEnvironment', 'username', 'password'])
    static abstract class RepoConfig {
        protected final Project project
        protected boolean publish
        protected String name
        protected String alternateUserName
        protected String username
        protected String password
        protected Closure<String> getCredentialsPrefix = { String name -> name.toLowerCase() }
        protected Closure<String> getEnvVarPrefix = { String name -> name.toUpperCase() }
        protected boolean ignoreEnvironment = false

        Closure<String> publishUrlGenerator
        Closure<List<String>> dependencyUrlsGenerator

        RepoConfig(Project project, Map props) {
            this(project, null, props)
        }

        RepoConfig(Project project, String alternateUserName, Map props) {
            this.project = project
            this.alternateUserName = alternateUserName
            this.ignoreEnvironment = project.findProperty('ignoreEnvironment') != null

            if (props) applyProps(props)

            if (!name) throw new GradleException("Missing 'name' property on repository configuration")

            resolveCredentials()
        }

        /**
         * @return The list of URLs to add as repositories for dependency resolution
         */
        List<String> getDependencyUrls() {
            return dependencyUrlsGenerator.call()
        }

        /**
         * @return The URL to configure for publishing to this repository
         */
        String getPublishUrl() {
            return getPublishUrl(project.isRelease)
        }

        /**
         * @param release Pass {@code true} to get the release URL and {@code false} to get the snapshot URL
         *
         * @return The URL to configure for publishing to this repository
         * @since 0.5-RC3
         */
        String getPublishUrl(boolean release) {
            return publishUrlGenerator.call(release)
        }

        /**
         * @return Credentials object to use in repo configuration
         */
        PasswordCredentials getCredentials() {
            // Validate the credentials now (upon their request) instead of at initialization time in order to
            // support the initial setting of the credentials, where a consuming project configures the repos
            // on the root project, but never actually tries to use them.  Many projects have special handling
            // in their settings.gradle to skip inclusion of all sub-projects (which avoids all the dependencies
            // that require the credentials in order to be resolved) to allow setting the credentials initially.
            // Validation will be skipped when running in offline mode since gradle will only search in local caches
            // and repositories.
            if ((!username || !password) && project.getGradle().startParameter.isOffline()) {
                username = "temp_username"
                password = "temp_password"
            } else {
                if (!username) throw new InvalidUserDataException("No username configured for repo $alternateUserName/$name")
                if (!password) throw new InvalidUserDataException("No password configured for user '$username' for repo $alternateUserName/$name")
            }

            return new DefaultPasswordCredentials(username, password)
        }

        /**
         * Apply the properties from the configuration map.  Minimal validation, just ensure that the 'publish'
         * property, if specified, is a boolean
         *
         * @param props Properties to apply
         */
        protected void applyProps(Map props) {
            def publishValue = props.get('publish')
            if (publishValue && !(publishValue instanceof Boolean)) {
                throw new GradleException("The 'publish' property must be a boolean value")
            }

            publish = publishValue
            name = props.get('name')
            username = props.get('username')
            password = props.get('password')
        }

        /**
         * Try to find the username/password credentials for this repo.  Any username/password specified in the
         * configuration properties take precedence.  If not specified, we check the environment first for a
         * variable named {@code [name]_NEXUS_USERNAME} and {@code [name]_NEXUS_PASSWORD}. If those don't exist,
         * then we try the credentials plugin store using the property names {@code [name]RepoUsername} and
         * {@code [name]RepoPassword} with the 'name' converted to lower case.
         *
         * If an 'alternateUserName' has been configured and no values were found using the primary name, then we
         * repeat search using the alternate name.
         */
        protected void resolveCredentials() {
            if (!username) username = findUsername(name)
            if (!password) password = findPassword(name)

            if ((!username || !password) && alternateUserName && alternateUserName != name) {
                username = findUsername(alternateUserName)
                password = findPassword(alternateUserName)
            }

            // Spit out the resolved credentials, if requested
            if (project.hasProperty('dumpCredentials')) {
                project.logger.lifecycle "$project repo $alternateUserName/$name user='$username' pwd='$password'"
            }
        }

        /**
         * Use a given name to try to locate a user name in the environment, or falling back to the credentials store.
         *
         * @param name Base 'name' to use to locate credentials
         * @return user name if located
         */
        private String findUsername(String name) {
            def username = ignoreEnvironment ? null : System.getenv("${getEnvVarPrefix(name)}_NEXUS_USERNAME")
            return username ? username : project.credentials."${getCredentialsPrefix(name)}RepoUsername"
        }

        /**
         * Use a given name to try to locate a password in the environment, or falling back to the credentials store.
         *
         * @param name Base 'name' to use to locate credentials
         * @return password if located
         */
        private String findPassword(String name) {
            def password = ignoreEnvironment ? null : System.getenv("${getEnvVarPrefix(name)}_NEXUS_PASSWORD")
            return password ? password : project.credentials."${getCredentialsPrefix(name)}RepoPassword"
        }
    }

    /**
     * Local implementation to avoid reliance on gradle internal classes.
     *
     * @since 0.5.6
     */
    private static final class DefaultPasswordCredentials implements PasswordCredentials {
        private String username
        private String password

        DefaultPasswordCredentials(String username, String password) {
            this.username = username
            this.password = password
        }

        String getUsername() {
            return username
        }

        void setUsername(String username) {
            this.username = username
        }

        String getPassword() {
            return password
        }

        void setPassword(String password) {
            this.password = password
        }

        @Override
        String toString() {
            return String.format("Credentials [username: %s]", username)
        }
    }
}
