package gov.raptor.gradle.plugins.buildsupport

import org.ajoberstar.grgit.Commit
import org.gradle.api.*
import org.gradle.api.internal.project.ProjectInternal
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.publish.maven.tasks.AbstractPublishToMaven
import org.gradle.api.publish.maven.tasks.PublishToMavenLocal
import org.gradle.api.tasks.Delete
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.bundling.Jar
import org.gradle.api.tasks.javadoc.Javadoc
import org.gradle.api.tasks.testing.Test
import org.gradle.api.tasks.testing.TestDescriptor
import org.gradle.api.tasks.testing.TestResult
import org.gradle.util.ConfigureUtil

import java.text.SimpleDateFormat
import java.util.jar.Attributes
import java.util.jar.JarFile

/**
 * The Build Support extension.  The methods defined here are available for use by any consuming build script.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 */
@SuppressWarnings("GroovyUnusedDeclaration")
class BuildSupportExtension {

    public static final String NAME = 'buildSupport' // Name under which to register this extension

    private final ProjectInternal project

    @Internal
    private RepoManagement repoManagement

    BuildSupportExtension(ProjectInternal project) {
        this.project = project

        repoManagement = new RepoManagement(project)
    }

    /**
     * Apply the given closure to configure our repo management.
     */
    void repoManagement(Closure configureClosure) {
        ConfigureUtil.configure(configureClosure, repoManagement)
    }

    RepoManagement getRepoManagement() {
        return repoManagement
    }

    /**
     * Dump out our version and some associated build information:
     * <ul>
     *     <li>Our version</li>
     *     <li>Build date</li>
     *     <li>Build revision (from git)</li>
     *     <li>Build JDK info</li>
     * </ul>
     */
    void reportPluginVersion() {
        String version = getClass().getPackage().getImplementationVersion()

        String buildDate = 'unknown'
        String buildNumber = 'unknown'
        String buildJdk = 'unknown'

        URL jarUrl = BuildSupportExtension.getProtectionDomain().getCodeSource().getLocation()
        if (jarUrl != null && jarUrl.getProtocol() == "file") {
            File jarFile = new File(jarUrl.getFile())
            JarFile jar = new JarFile(jarFile)
            Attributes attrs = jar.manifest.mainAttributes

            buildDate = attrs.getValue('Build-Date')
            buildNumber = attrs.getValue('Build-Number')
            buildJdk = attrs.getValue('Build-Jdk')
        }

        String versionMessage = """
------------------------------------------------------------
BuildSupportPlugin $version
------------------------------------------------------------

Build date:   $buildDate
Revision:     $buildNumber
Built using:  Java $buildJdk
"""
        project.logger.lifecycle versionMessage
    }

    /**
     * Perform the "standard" configuration for a typical Raptor plugin project.  This is equivalent to calling
     * the following methods:
     * <ul>
     * <li>{@link #setupJavaProject(org.gradle.api.Project)}</li>
     * <li>{@link #configureManifest(org.gradle.api.Project)}</li>
     * <li>{@link #configureJavaTemplates(org.gradle.api.Project)}</li>
     * <li>{@link #createPluginInstallTask(org.gradle.api.Project)}</li>
     * <li><code>if(configureTestAgent) </code>{@link #setupPluginTestAgent(org.gradle.api.Project)}</li>
     * <li>{@link #configurePublishing(org.gradle.api.Project, boolean)}</li>
     * </ul>
     * A "shortcut" method reference to {@link #manifestDependencies(groovy.lang.Closure)} is added to the project's
     * ext.
     *
     * @param p The project to configure
     * @param publishToNexus Pass {@code true} to enable publishing to the STL Nexus repo
     * @param configureTestAgent Pass {@code true} to configure the AOP test agent (required for integration tests
     * against the raptor core), or {@code false} if you don't need it.  default is {@code true}.
     * @param weaveTestPackages Additional packages to weave during test execution, optional, default is empty
     */
    void configureStandardPlugin(Project p, boolean publishToNexus, boolean configureTestAgent = true, String... weaveTestPackages = []) {
        project.logger.info "Configuring $p as standard plugin Project"

        setupJavaProject(p)
        configureManifest(p)
        configureJavaTemplates(p)
        createPluginInstallTask(p)
        if (configureTestAgent) setupPluginTestAgent(p, weaveTestPackages)
        configurePublishing(p, publishToNexus)

        // A little trick to expose the manifestDependencies method of buildSupport directly to the project
        p.ext.manifestDependencies = p.buildSupport.&manifestDependencies
    }

    /**
     * Configure the standard manifest entries for the given project.  This method will configure both the 'jar'
     * and 'war' tasks on the given project.
     *
     * @param p project to configure
     * @see #configureManifestForArchiveTask(org.gradle.api.tasks.bundling.Jar)
     */
    void configureManifest(Project p) {
        Map attributes = getManifestAttributes(p)

        // We want to process both jar and war tasks, if they are present
        ['jar', 'war'].each { archiveTaskName ->
            def archiveTask = p.tasks.findByName(archiveTaskName)
            if (archiveTask) {
                archiveTask.manifest.attributes(attributes)
            }
        }
    }

    /**
     * Configure the standard manifest entries for the given archive task.
     *
     * @param task Task to configure
     * @see #getManifestAttributes(org.gradle.api.Project)
     * @since 0.5.1
     */
    void configureManifestForArchiveTask(Jar task) {
        task.manifest.attributes(getManifestAttributes(task.project))
    }

    /**
     * Create the standard manifest attributes for the given project.
     * <p>
     * The following manifest attributes are configured:
     * <table>
     * <tr>
     * <th>
     * Attribute
     * </th>
     * <th>
     * Value
     * </th>
     * </tr>
     * <tr>
     * <td><tt>Built-By</tt></td>
     * <td>The value of <tt>rootProject.name</tt></td>
     * </tr>
     * <tr>
     * <td><tt>Implementation-Version</tt></td>
     * <td>The value of the {@code buildVersion} ext variable, like '2.7.1-SNAPSHOT'</td>
     * </tr>
     * <tr>
     * <td><tt>Build-Date</tt></td>
     * <td>The date the build was performed, formatted as {@code 'yyyy-MM-dd'}</td>
     * </tr>
     * <tr>
     * <td><tt>Build-Jdk</tt></td>
     * <td>The version of the JDK used to perform the build, e.g., {@code 1.8.0_112}</td>
     * </tr>
     * <tr>
     * <td><tt>Build-Number</tt></td>
     * <td>The SHA1 of the commit from which the build was created, e.g., {@code f0d164b762e0af659dd5eb8367f72fdd8fb0b21c}</td>
     * </tr>
     * <tr>
     * <td><tt>RaptorX-Version</tt></td>
     * <td>If this is a project/plugin build, this is the raptor version against which the project was built, i.e., {@code raptorVersion}</td>
     * </tr>
     * <tr>
     * <td><tt>Build-Branch</tt></td>
     * <td>The name of the branch on which the build was performed, e.g.,
     * {@code dev} or {@code 2.7} or {@code feature/OPS-78-build-support-plugin}</td>
     * </tr>
     * </table>
     *
     * @param p Project to process
     * @return configured attributes for manifest
     * @since 0.5.1
     */
    Map getManifestAttributes(Project p) {
        return p.with {
            // Ensure a common configuration in the manifest of each of our jars
            Map buildAttrs = [
                    "Built-By"              : rootProject.name,
                    "Implementation-Version": buildVersion,
                    "Build-Date"            : new Date().format('yyyy-MM-dd'),
                    "Build-Jdk"             : System.getProperty("java.version")]

            // If this is a project/plugin build, add the Raptor version to the manifest
            if (isProjectBuild && raptorVersion) {
                buildAttrs.'RaptorX-Version' = raptorVersion
            }

            try {
                // Skip this if grgit data wasn't initialized properly
                if (grgitUsable) {
                    buildAttrs += [
                            "Build-Number": grgit.head().id,
                            "Build-Branch": grgit.branch.current.name
                    ]
                }
            } catch (Exception ignored) {
                if (!grgitFailureReported) {
                    logger.warn "It appears that grgit is not able to process this work tree; excluding build info from manifest", ignored
                    grgitFailureReported = true
                }
            }

            return buildAttrs
        }
    }

    /**
     * Configure the given project as a java project, with appropriate plugins, compiler configuration and
     * publishing after a {@code build}.
     * <p>
     * Plugins applied: {@code 'java'} and {@code 'maven-publish'}
     * <p>
     * The Java compiler is configured for source and target compatibility of 1.8, and configured to include
     * debug line numbers (but no local variable table).
     * <p>
     * A task, named {@code compile}, is created as an alias for the {@code jar} task to ease developers transition from
     * maven.
     * <p>
     * A task, named {@code releaseClean}, is created to provide a hook for any extended cleanup operations that need to
     * take place prior to performing a 'release' build.  The {@code releaseClean} task depends on {@code clean}.
     * <p>
     * The {@code build} task is configured to be finalized by {@code publish} so the produced jar file
     * is copied into the local maven repo (and Nexus, on release builds if configured) if the build succeeds.
     * <p>
     * If this is a 'release' build, then the {@code build} task is configured to depend on {@code releaseClean} and
     * and {@code updateReleaseVersion} and finalized by {@code tagRelease}.
     * <p>
     * The {@code test} task is configured to not fail the build if it is running under the Jenkins CI server, and
     * it will announce the start of each test if {@code -Dtest.announce=true} is specified on the build command line.
     */
    @SuppressWarnings("GroovyAssignabilityCheck")
    void setupJavaProject(Project p) {
        p.with {
            logger.info "Configuring $p as Java Project"

            def pluginManager = getPluginManager()
            pluginManager.apply('java')
            pluginManager.apply('maven-publish')

            setupRepositories(p)

            compileJava {
                sourceCompatibility = '1.8'
                targetCompatibility = '1.8'

                options.warnings = false
                options.debug = true

                // We don't want 'vars' debug data on release builds
                String debugData = isRelease ? 'lines,source' : 'lines,source,vars'
                options.debugOptions.debugLevel = debugData
            }

            if (JavaVersion.current().isJava8Compatible()) {
                // disable the crazy super-strict doclint tool in Java 8
                tasks.withType(Javadoc) {
                    //noinspection SpellCheckingInspection
                    options.addStringOption('Xdoclint:none', '-quiet')
                }
            }

            // Define a new task 'compile' to make conversion from maven easier for developers
            tasks.create(name: 'compile', group: 'Build',
                    description: 'Process resources, compile, and create jar (compatibility with maven)') {
                dependsOn jar
            }

            // Define a 'releaseClean' task on every project, so they can extend as needed
            tasks.create(name: 'releaseClean', type: Delete, dependsOn: clean, group: 'Build',
                    description: 'Delete ALL generated artifacts from any source, prepare for release build')

            updateParentWithReleaseClean(p)

            if (isRelease) {
                Task beforeBuildTask = null
                Task buildTask = tasks.build

                // If this is a sub-project being configured, then we need to test and ensure that the
                // root project has a 'build' task so we can properly add task dependencies to it.
                if (parent) {
                    if (!rootProject.tasks.findByName('build')) {
                        beforeBuildTask = buildTask = rootProject.tasks.create('build', DefaultTask)
                    }
                } else {
                    // On single project builds, we need to ensure we execute before the build
                    // But, build is a lifecycle task, so you can't simply add dependencies to 'build'
                    // you have to add them to the first thing that build invokes, and that is compileJava

                    beforeBuildTask = tasks.compileJava
                }

                // Configure the root-level task dependencies, note the ?. usage to avoid configuring
                // any sub-project after the first

                beforeBuildTask?.configure {
                    if (isRebuildRelease) {
                        // No updateReleaseVersion and no tag on a rebuild
                        dependsOn ':releaseClean'
                    } else {
                        dependsOn ':releaseClean', ':updateReleaseVersion'
                        buildTask.finalizedBy ':tagRelease'
                    }
                }
            }

            tasks.build.finalizedBy 'publish' // Always publish artifacts after a build

            // Configure the 'test' tasks
            configureTestTasks(p)

            // We need to ensure that we don't publish anything until all the builds are complete.
            // Otherwise, we can end up with a partially published set of artifacts when a later test fails.

            if (rootProject != project) afterEvaluate { aep ->
                if (!rootProject.tasks.findByName('build')) {
                    rootProject.tasks.create('build', DefaultTask) // Must have a build task
                }

                rootProject.tasks.build.dependsOn aep.tasks.build
            }
        }
    }

    /**
     * Ensure that the releaseClean task is available all the time at the root.
     */
    void updateParentWithReleaseClean(Project p) {
        if (p.parent) {
            p.with {
                if (!parent.tasks.findByName('releaseClean')) {
                    parent.tasks.create(name: 'releaseClean',
                            type: Delete,
                            group: 'Build',
                            description: 'Delete ALL generated artifacts from any source, prepare for release build') {
                        delete parent.buildDir // In case the build produces anything in the root (like the test repo)
                    }
                }

                parent.tasks.releaseClean.dependsOn tasks.releaseClean
            }
        }
    }

    /**
     * Apply a standard configuration to all test tasks.
     *
     * @param p Project to configure
     */
    void configureTestTasks(Project p) {
        p.with {
            tasks.withType(Test) {
                // Don't fail the build on the CI server, unless we're performing a release build
                ignoreFailures = runningUnderJenkins && !isRelease

                // If you want to have each test announced as it starts, then add -Dtest.announce=true to the command line
                if (System.getProperty('test.announce')) {
                    beforeTest { TestDescriptor test ->
                        String now = new SimpleDateFormat("HH:mm:ss.SSS").format(new Date())
                        println "$now: starting: $test.parent : $test.name"
                    }
                }

                testLogging {
                    showStandardStreams = true // Show stdout and stderr
                    exceptionFormat 'full'

                    info {
                        exceptionFormat 'full'
                        events 'started', 'passed', 'failed', 'skipped', 'standardOut', 'standardError'
                        displayGranularity 4
                    }

                    // Give a nicely formatted summary of each suites execution

                    def suiteCount = 0 // Count number of suites who's results have been reported

                    afterSuite { TestDescriptor desc, TestResult result ->
                        // In the case that there is only a single suite (as happens with default test scanning),
                        // we only want to report the test results once, meaning no totals line aggregating all the suites
                        // since there was only one.

                        def finalSummary = desc.parent == null
                        def topLevelSuite = desc.parent =~ /Gradle Test Executor/

                        if (topLevelSuite || (finalSummary && suiteCount > 1)) {
                            rootProject.testFailureCount += result.failedTestCount // Track the failure count

                            suiteCount += 1
                            def name = desc.parent ? desc.name : 'Gradle Test Run'
                            printf('Results: %-30s %s (%3d tests, %3d succeeded, %3d failed, %3d skipped) in %.2f sec%n',
                                    name, result.resultType, result.testCount, result.successfulTestCount,
                                    result.failedTestCount, result.skippedTestCount, (result.endTime - result.startTime) / 1000)
                        }
                    }
                }
            }
        }
    }

    /**
     * Configure the given project so that it has access to the Raptor bootstrap java agent.  This agent is
     * required to run container/integration tests that use the Raptor test support library.
     *
     * After calling this method, the project will have:
     * <ul>
     *     <li>A configuration named {@code testAgent}</li>
     *     <li>A dependency for {@code testAgent} on {@code gov.raptor:bootstrap:$raptorVersion}</li>
     *     <li>An {@code ext} variable {@code bootstrapJar} assigned the full path to the Raptor bootstrap jar</li>
     *     <li>The {@code jvmArgs} property of the {@code test} task will be initialized to {@code ['-ea', "-javaagent:$bootstrapJar"]}</li>
     * </ul>
     *
     * @param p The project to configure
     * @param weaveTestPackages Additional packages to weave during test execution
     */
    void setupPluginTestAgent(Project p, String[] weaveTestPackages = []) {
        p.with {
            // Add a special configuration/dependency for the java agent, located in the Raptor bootstrap jar
            configurations {
                testAgent
            }
            dependencies {
                testAgent "gov.raptor:bootstrap:$raptorVersion"
            }

            p.ext {
                // Get the path to the bootstrap jar file (we need this for the test invocation JVM args)
                bootstrapJar = configurations.testAgent.find { it.name.contains("bootstrap") }
                logger.info("$p: bootstrapJar: $bootstrapJar")
            }

            p.test.jvmArgs = ['-ea', "-javaagent:$bootstrapJar"]

            if (weaveTestPackages) p.test.systemProperty('weaveTestPackages', weaveTestPackages.join(';'))
        }
    }

    /**
     * Add in our standard repositories.  They are:
     * <ul>
     *     <li><tt>jcenter()</tt></li>
     *     <li><tt>mavenLocal()</tt>  *optional</li>
     *     <li><tt>mavenCentral()</tt></li>
     * </ul>
     *
     * <em>Note:</em> if the project property {@code noMavenLocal} is set, then <tt>mavenLocal()</tt> will not be added to the repository list
     *
     * @param p Project to configure
     */
    @SuppressWarnings("GroovyAssignabilityCheck")
    void setupRepositories(Project p) {
        p.with {
            repositories {
                jcenter()
                mavenCentral()

                // Don't include mavenLocal if excluded by the repo config
                if (!repoManagement.isNoMavenLocal()) mavenLocal()

                // Add google if configured
                if (repoManagement.includeGoogle()) google()

                // Process all the configured repos
                repoManagement.repoConfigs.forEach { config ->
                    def repoCredentials = config.getCredentials()
                    config.getDependencyUrls().forEach { repoUrl ->
                        maven {
                            name "nexus-${config.name}"
                            url repoUrl
                            if (repoCredentials) {
                                credentials {
                                    username = repoCredentials.username
                                    password = repoCredentials.password
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Validate that the current work tree is clean. Allow the <code>-Pforce</code> option to override the clean check.
     * If the tree is not clean, and <code>-Pforce</code> is not specified, then a {@link GradleException} is thrown.
     *
     * @since 0.5
     */
    void verifyTreeIsClean() {
        project.with {
            if (!grgitUsable) return // Must have usable grgit data to check this

            if (!force && !grgit.status().isClean()) {
                logger.error("Tres Status: ${grgit.status()}")
                throw new GradleException("The work tree is not clean!  You can override this check with -Pforce.")
            }
        }
    }

    //@formatter:off
    /**
     * Resolve a given tag name to a commit.  Trap any underlying exception (like an NPE which is how jgit responds)
     * and convert it into a RuntimeException with a meaningful error message.
     *
     * @param tagName The tag string to resolve to a Commit
     * @return Commit
     * @throws RuntimeException If the resolve operation fails
     */
    //@formatter:on
    Commit safeResolveCommit(String tagName) throws RuntimeException {
        try {
            return grgit.resolve.toCommit(tagName)
        } catch (Exception e) {
            throw new RuntimeException("No commit found for revision string: " + tagName, e)
        }
    }

    /**
     * Create a task, named {@code install}, that will copy the created jar/war artifact(s) into
     * {@code ../RaptorT/Raptor/bin/plugins} (by default).  If the environment variable
     * {@code ORG_GRADLE_PROJECT_pluginInstallDir} is set, it will provide a value for the {@code pluginInstallDir}
     * project property.  This may be overridden if the project defines the variable {@code pluginInstallDir}
     * (e.g., using {@code -PpluginInstallDir} on the command line, or setting it in a gradle.properties file) it
     * will override the default destination directory.
     * <p/>
     * The {@code install} task is dependent upon the {@code build} task so that if tests fail, the artifact is not copied.
     * <p/>
     * There are problems when a 'war' project is a dependency of another project. In this case when the dependent
     * project is built, the war project is built as a dependency, but it builds a jar (that would not normally be
     * created when doing a 'build' in that project directly).  That would lead to having both a jar and a war
     * being installed into the plugins directory when only a war should have been copied.  In order to handle this
     * case, a configuration parameter has been added to control this behavior.  If a plugin project's build would
     * generate both a jar and a war and both should be installed, the ext variable {@code installBothWarAndJar}
     * must be defined.
     *
     * @param p Project to configure
     */
    void createPluginInstallTask(Project p) {
        p.with {
            if (!isProjectBuild) {
                throw new GradleException("You should NEVER invoke createPluginInstallTask on a RaptorT build")
            }

            logger.info "Configuring $p to install into $raptorPluginsDir"

            tasks.create(name: 'install', group: 'Build', description: 'Copy jar/war to RaptorT/Raptor/bin/plugins (override with -PpluginInstallDir)') {
                dependsOn 'build', 'publishToMavenLocal'

                doLast {
                    File destDir = p.raptorPluginsDir

                    // Avoid creating the destination directory out of thin air

                    if (destDir.exists() && destDir.isDirectory()) {
                        // We want to copy both jar and war files, if they are present and if configured
                        // Otherwise, just install either jar or war

                        boolean isWar = pluginManager.hasPlugin('war')
                        boolean installBoth = p.hasProperty('installBothWarAndJar')

                        List<String> tasksToProcess = isWar ? ['war'] : ['jar']
                        if (isWar && installBoth) tasksToProcess = ['jar', 'war']

                        tasksToProcess.each { archiveTaskName ->

                            // Process the archive, if the appropriate task exists on the project
                            def archiveTask = tasks.findByName(archiveTaskName)
                            if (archiveTask) {
                                File archiveFile = archiveTask.archivePath

                                if (archiveFile && archiveFile.exists()) {
                                    // Delete all matching files from the installation area.  This is an
                                    // attempt to delete prior versions when this one is installed.

                                    def installedFiles = fileTree(destDir).include { f ->
                                        f.name =~ $/${archiveTask.baseName}-[0-9]+.*\.${archiveTask.name}/$
                                    }
                                    def filesToDelete = installedFiles.files

                                    if (filesToDelete.size() > 1) logger.lifecycle "Deleting: ${filesToDelete*.name}"
                                    delete installedFiles

                                    logger.lifecycle "Install $archiveFile.name into $destDir"

                                    copy {
                                        from archiveFile
                                        into destDir
                                    }
                                }
                            }
                        }
                    } else {
                        String reason = destDir.exists() ? "not a directory" : "missing"
                        logger.warn "$destDir is $reason; skipping the jar/war install!"
                    }
                }
            }
        }
    }

    /**
     * Configure a project to publish maven artifacts to the local M2 repo and, optionally, to the Nexus maven repo.
     * This supports dependency references from RaptorPlugins (or any other project) onto artifacts built here.
     * The artifacts are published to the "{@code releases}" repository using the current user's credentials.
     * <p>
     * Publishing to Nexus is only enabled when performing a "release" build (when {@code -Prelease} is
     * specified on the command line) or if {@code -PforcePublish} is specified on the command line.
     * <p>
     * Publishing is disabled if this is a snapshot build ({@code isSnapshot} is true) or
     * a rebuild ({@code -PrebuildRelease} is specified).
     * <p>
     *  If {@code -PuseTestRepo} is specified on the command line, then the artifacts will be published to a local
     *  repo within the build area, {@code $buildDir/repo} instead of to Nexus.
     * <p>
     *  Two publications are managed:
     *  <ul>
     *      <li>{@code mavenJava} - For artifacts to publish to remote maven repos</li>
     *      <li>{@code forMavenLocal} - For artifacts to publish to "maven local" repo</li>
     *  </ul>
     *
     * @param p The project to configure
     * @param toNexus Pass {@code true} to configure publishing to Nexus (if this is a release build).  Passing
     * {@code false} skips configuring Nexus publication regardless of build type.  The Default value is {@code true}.
     */
    void configurePublishing(Project p, boolean toNexus = true) {

        def component = p.pluginManager.hasPlugin('com.android.library')
                ? p.components.android
                : p.pluginManager.hasPlugin('war') ? p.components.web : p.components.java

        internalConfigurePublishing(p, toNexus) {
            from component
        }
    }

    /**
     * Configure a project to publish a specified artifact to maven repositories
     * (as configured by {@link #setupRepositories(org.gradle.api.Project)}.
     * <p>
     * This supports projects that don't generate typical components (like jars or wars), e.g., an android project
     * that needs to publish it's APK.
     * <p>
     * Publishing to Nexus is only enabled when performing a "release" build (when {@code -Prelease} is
     * specified on the command line) or if {@code -PforcePublish} is specified on the command line.
     * <p>
     * Publishing is disabled if this is a snapshot build ({@code isSnapshot} is true) or
     * a rebuild ({@code -PrebuildRelease} is specified).
     * <p>
     *  If {@code -PuseTestRepo} is specified on the command line, then the artifacts will be published to a local
     *  repo within the build area, {@code $buildDir/repo} instead of to Nexus.
     * <p>
     *  Two publications are managed:
     *  <ul>
     *      <li>{@code mavenJava} - For artifacts to publish to remote maven repos</li>
     *      <li>{@code forMavenLocal} - For artifacts to publish to "maven local" repo</li>
     *  </ul>
     *
     * @param p The project to configure
     * @param artifactToPublish The artifact to publish
     * @param toNexus Pass {@code true} to configure publishing to Nexus (if this is a release build).  Passing
     * {@code false} skips configuring Nexus publication regardless of build type.  The Default value is {@code true}.
     */
    void publishArtifact(Project p, artifactToPublish, boolean toNexus = true) {
        internalConfigurePublishing(p, toNexus) {
            artifact artifactToPublish
        }
    }

    /**
     * Common handling to configure publishing of components/artifacts to maven local and Nexus.
     *
     * @param p The project to configure
     * @param toNexus Pass {@code true} to configure publishing to Nexus (if this is a release build).  Passing
     * {@code false} skips configuring Nexus publication regardless of build type.  The Default value is {@code true}.
     * @param configClosure Closure to use to configure the publications, passed to
     * {@link #configurePublications(org.gradle.api.Project, groovy.lang.Closure)}
     */
    private void internalConfigurePublishing(Project p, boolean toNexus, configClosure) {
        p.with {
            // If the 'maven-publish' plugin hasn't been applied, then do it now
            if (!pluginManager.hasPlugin('maven-publish')) pluginManager.apply('maven-publish')

            publishing {
                // Create the empty publications
                publications {
                    forMavenLocal(MavenPublication) {}
                    mavenJava(MavenPublication) {}
                }

                configurePublications(p, configClosure)

                // We don't publish if we are creating a 'release' build of a SNAPSHOT (an unusual edge case), unless
                // the user specified -PforcePublish
                // We also don't publish if we doing a "rebuild release" from a release tag

                if (toNexus && (isForcePublish || (isRelease && !isRebuildRelease && !isSnapshot))) {
                    logger.info "Configure publishing to Nexus for $p"

                    repositories {
                        def repoType = isRelease ? 'releases' : 'snapshots'

                        if (p.hasProperty('useTestRepo')) {
                            maven {
                                logger.warn "Using test repo: $rootProject.buildDir/repo/$repoType"
                                name 'test-repo'
                                url "$rootProject.buildDir/repo/$repoType" // for testing
                            }
                        } else {
                            repoManagement.getPublishingRepoConfigs().forEach { config ->
                                def repoCredentials = config.getCredentials()

                                logger.info "Configure: nexus-${config.name} for publishing"
                                maven {
                                    //noinspection GroovyAssignabilityCheck
                                    name "nexus-${config.name}"
                                    url config.getPublishUrl()
                                    credentials {
                                        username = repoCredentials.username
                                        password = repoCredentials.password
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // Configure the publish tasks to only publish their respective publication

            tasks.withType(AbstractPublishToMaven) {
                onlyIf { task -> p.buildSupport.shouldPublish(task) }
            }

            // If noMavenLocal is configured, then disable the publishToMavenLocal task
            tasks.'publishToMavenLocal'.enabled = !repoManagement.isNoMavenLocal()

            // There is a strange behavioral difference between declaring mavenLocal() as a repository in the above
            // publishing configuration and having publish depend on publishToMavenLocal.  The former constantly
            // adds fully timestamped artifacts to the local repo, making it grow boundlessly.  The latter properly
            // updates the singular 'xxx-SNAPSHOT.jar' in the local repo, which is what SHOULD happen.
            //
            // So, for now, just force the dependency instead of declaring the repo.

            tasks.publish.dependsOn 'publishToMavenLocal'

            // And ensure that we don't try to publish things until we've successfully tagged the repo
            tasks.publish.mustRunAfter ':tagRelease'
        }
    }

    /**
     * Determine if a given publishing task should actually execute.  We manage 2 publications, and we want to ensure
     * they only execute on their appropriate targets.  Further, we don't want to interfere with any other
     * publication that the project might have configured.
     * <p>
     * In addition, publishing will be prevented when:
     * <ul>
     *     <li>any of the configured artifacts are missing</li>
     *     <li>any test failures occurred during the build</li>
     * </ul>
     * <p>
     * Hopefully, the publishing plugin will someday grow up and allow the specification of an "optional"
     * artifact.
     *
     * @param task Publishing task to test
     * @return {@code true} if the task should execute
     * @since 0.6.1
     */
    boolean shouldPublish(AbstractPublishToMaven task) {
        Project project = task.project

        // If tests failed, then don't publish
        if (project.rootProject.testFailureCount) {
            project.logger.lifecycle("There were failing tests [${project.rootProject.testFailureCount}]; publishing skipped!")
            return false
        }

        // If there are any missing artifacts, then don't publish
        def missingArtifacts = task.publication.artifacts
                .collect({ artifact -> artifact.file })
                .findAll { file -> !file.exists() }

        if (missingArtifacts) {
            project.logger.info("Artifacts missing; publishing skipped! $missingArtifacts")
            return false
        }

        boolean isMavenLocalTask = task instanceof PublishToMavenLocal
        boolean isForMavenLocalPub = task.publication == project.publishing.publications.forMavenLocal
        boolean isMavenJavaPub = task.publication == project.publishing.publications.mavenJava

        return isMavenLocalTask
                ? !repoManagement.isNoMavenLocal() && (isForMavenLocalPub || !isMavenJavaPub)
                : isMavenJavaPub || !isForMavenLocalPub
    }

    //@formatter:off
    /**
     * Apply a configuration closure to the publications managed/created by this plugin.  A typical use is to
     * add an artifact to the publications, like this:
     * <pre>
     * buildSupport.configurePublications(project) {
     *     artifact testJar
     * }
     * </pre>
     * The given closure will be applied to both the {@code mavenJava} and {@code forMavenLocal} publications.
     * @param p The project to configure
     * @param configClosure The configuration closure to apply to the publications
     * @since 0.6.1
     */
    //@formatter:on
    void configurePublications(Project p, Closure configClosure) {
        p.afterEvaluate {
            ConfigureUtil.configure(configClosure, p.publishing.publications.mavenJava)
            ConfigureUtil.configure(configClosure, p.publishing.publications.forMavenLocal)
        }
    }

    /**
     * Configure a project to create and publish a "sources" jar.  By default sources will only publish to maven
     * local.  If you really want sources published to a "real" maven repository, then pass {@code false} for
     * {@code mavenLocalOnly} (which defaults to true).
     * <p>
     * Be certain you understand the access controls of your project before you even consider publishing sources
     * anywhere other than maven local.
     * <p>
     * This method creates a {@code sourcesJar} task to produce the jar containing the sources, and creates
     * a publication named {@code 'sources'} that uses the {@code sourcesJar} output as the artifact to publish
     * with a {@code 'sources'} classifier.
     *
     * @param p The project to configure
     * @param mavenLocalOnly Pass {@code true} to limit publication to maven local only, {@code false} to publish
     *                       to all configured repositories.  Default is {@code true}.
     */
    void configureSourcePublishing(Project p, boolean mavenLocalOnly = true) {
        p.with {
            tasks.create(name: 'sourcesJar', type: Jar, group: 'Build') {
                from sourceSets.main.allJava
                archiveClassifier.set('sources')
            }

            publishing {
                publications {
                    forMavenLocal(MavenPublication) {
                        artifact sourcesJar
                    }

                    // If configured, also add the sources to the primary publication (which is published externally)
                    if (!mavenLocalOnly) {
                        mavenJava(MavenPublication) {
                            artifact sourcesJar
                        }
                    }
                }
            }
        }
    }

    /**
     * Configure a project to publish a javadoc artifact.  The project must configure the {@code javadoc} task
     * as needed.  This method will create a {@code packageJavadoc} task that creates a jar file from the output
     * of the {@code javadoc} task with a {@code 'javadoc'} classifier, and then updates the {@code 'mavenJava'}
     * publication to include that artifact.  The {@code packageJavadoc} task is only enabled if the
     * {@code javadoc} task is enabled.
     *
     * @param p The project to configure
     */
    void configureJavadocPublishing(Project p) {
        p.with {
            // Create the task to construct the javadoc jar
            tasks.create(name: 'packageJavadoc', type: Jar, dependsOn: 'javadoc', group: 'Build') {
                from javadoc
                archiveClassifier.set('javadoc')
            }

            // Delay processing until after evaluation so we truly know if the javadoc task is enabled

            project.afterEvaluate {
                boolean enabled = tasks.javadoc.enabled
                tasks.packageJavadoc.enabled = enabled // Enabled if the javadoc task is enabled

                // If javadoc is enabled, then add it to the publishing configuration
                if (enabled) {
                    publishing {
                        publications {
                            forMavenLocal(MavenPublication) {
                                artifact packageJavadoc
                            }
                            mavenJava(MavenPublication) {
                                artifact packageJavadoc
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * This method adds a "Dependencies" entry to the manifest in the jar/war artifacts.  It uses a closure to produce
     * the value of the dependencies string.  This allows for the delayed evaluation needed to resolve things like
     * project versions on nested sub-projects.
     * <p>
     * For example, this configuration:
     * <pre>    manifestDependencies { "plugin.calculators-${project(':plugin.calculators').version}" }</pre>
     * <p>
     * Would result in a manifest entry like this (ignoring the actual version of the artifact):
     * <pre>    Dependencies: plugin.calculators-1.0.0</pre>
     *
     * @param closure Must return either a single string, or an iterable (like a list) which will be joined with commas
     */
    static void manifestDependencies(Closure closure) {
        def script = closure.delegate

        script.gradle.taskGraph.whenReady { taskGraph ->
            def value = closure.call()
            if (value instanceof Iterable) {
                value = value.join(',')
            }

            // We want to copy both jar and war files, if they are present
            ['jar', 'war'].each { archiveTaskName ->
                // Process the archive, if the appropriate task exists on the project
                def archiveTask = script.tasks.findByName(archiveTaskName)
                if (archiveTask) {
                    script.logger.info "Add manifest dependencies to $archiveTask : $value"
                    archiveTask.manifest.attributes('Dependencies': value)
                }
            }
        }
    }

    /**
     * Configure a project to use "java templates."  The templates are assumed to live in
     * {@code src/main/java-templates}, which can be overridden with the {@code javaTemplatesDir} project property.
     * The generated code will be placed in {@code build/generated-sources}, which will be configured as a java source
     * dir during compilation.  The template(s) will be copied using normal gradle token expansion.
     * <p>
     * The current project will be automatically added to the token expansion map under the name 'project'
     * and additional tokens can be supplied via the project property {@code templateProperties}, which must be
     * a map of String keys to arbitrary values.
     *
     * The consuming build script can define the following properties to configure the build:
     * <ul>
     *     <li>{@code javaTemplatesDir} - the directory in which to find the template java files,
     *     defaults to {@code 'src/main/java-templates'}</li>
     *     <li>{@code templateProperties} - A map of additional properties to use during 'expand' processing</li>
     * </ul>
     * <p>
     * This script creates a new {@code copyJavaTemplates} task and configures {@code compileJava} to depend on it.
     * If the java templates directory does not exist, then the task is not created.
     */
    void configureJavaTemplates(Project p) {
        p.with {
            if (findProperty('javaTemplatesConfigured')) {
                logger.info "templates already configured"
                return
            }

            ext {
                generatedSourcesDir = "$buildDir/generated-sources"

                if (!p.hasProperty('javaTemplatesDir')) {
                    javaTemplatesDir = 'src/main/java-templates' // Apply default if not specified
                }
            }

            // If the template directory doesn't exist, then don't configure the task
            if (!file(javaTemplatesDir).exists()) {
                logger.info "templates dir [$javaTemplatesDir] is missing; skipping templates setup"
                return
            }

            ext.javaTemplatesConfigured = true

            logger.info "Configuring Java templates:\n   templatesDir = $javaTemplatesDir\n   genSrcsDir = $generatedSourcesDir"

            // Add the generated sources to our java source set
            sourceSets.main.java.srcDir generatedSourcesDir

            // Task to copy the java templates with property expansion enabled.
            //noinspection GroovyAssignabilityCheck
            tasks.create(name: 'copyJavaTemplates', group: 'Build', description: 'Copy java template files to generated-sources with token expansion') {
                inputs.dir javaTemplatesDir
                outputs.dir generatedSourcesDir

                // We encountered problems on Win10 where this task was running BEFORE clean/releaseClean,
                // so, force the proper execution order.

                mustRunAfter 'clean', 'releaseClean'
                doLast {
                    Map propertyMap = [:]

                    // Add in extra properties declared in the project
                    if (p.hasProperty('templateProperties')) {
                        //noinspection GroovyAssignabilityCheck
                        propertyMap.putAll(p.templateProperties)
                    }

                    // Add in the 'project' mapping, if not already set
                    if (!propertyMap.containsKey('project')) {
                        propertyMap.put('project', p)
                    }

                    copy {
                        from javaTemplatesDir
                        into generatedSourcesDir
                        expand(propertyMap)
                    }
                }
            }

            tasks.compileJava.dependsOn 'copyJavaTemplates'
        }
    }

    //@formatter:off
    /**
     * Configure the given project for generating RDM facade classes from Structure XML definitions.  This adds an
     * extension, of type {@link GenerateRdmFacadesExtension}, to the project.  See the documentation for that
     * extension to get details and examples of how to configure facade generation.
     *
     * @param p
     * @see GenerateRdmFacadesExtension
     * @since 0.4
     */
    //@formatter:on
    void configureRdmFacades(Project p) {
        // Create the extension, it does all the rest
        p.extensions.create(GenerateRdmFacadesExtension.EXTENSION_NAME, GenerateRdmFacadesExtension, p)
    }

    /**
     * Configure the given project for using XJC to compile XML schemas.  The invoking project will use the following
     * optional properties to configure this operation:
     * <ul>
     * <li>{@code xjcSchemasDir} - the directory in which to find the schemas to compile, default is {@code src/main/resources/schema}</li>
     * <li>{@code xjcPackage} - the package into which to generate the classes, the default is determined from the
     * package defined in {@code jxb:package} within each schema being processed</li>
     * </ul>
     * It is assumed that your binding definitions ({@code *.xjb} files), if any, are in the same directory as the schemas.
     */
    void configureXjc(Project p) {
        p.with {
            apply plugin: 'org.openrepose.gradle.plugins.jaxb'

            ext {
                xjcDestinationDir = jaxb.xjc.destinationDir
            }

            dependencies {
                // For XJC use
                jaxb 'org.glassfish.jaxb:jaxb-xjc:2.2.11'
                jaxb 'org.glassfish.jaxb:jaxb-runtime:2.2.11'
                jaxb 'javax.activation:javax.activation-api:1.2.0'
            }

            // Add the XJC generated sources to our java source set
            // This setting ensures that other projects depending on this one will see the generated sources (and IntelliJ is happy)
            sourceSets.main.java.srcDir xjcDestinationDir

            jaxb {
                // If the build script defines 'xjcSchemasDir', it will override the default of src/main/resources/schema
                String schemasDir = p.findProperty('xjcSchemasDir')
                if (schemasDir) {
                    xsdDir = bindingsDir = schemasDir
                }

                xjc {
                    header = false

                    if (p.hasProperty('xjcPackage')) {
                        generatePackage = xjcPackage
                    }
                }
            }

            logger.info "Configured XJC:\n   xsdDir = $jaxb.xsdDir\n   package = $jaxb.xjc.generatePackage\n   dest = $xjcDestinationDir"

            // Can't compile our main java code until XJC generates the schema classes
            compileJava.dependsOn 'xjc'
        }
    }

    /**
     * Execute a git command using the installed command line tools.  This should not be used for "required" operations.
     * It is here for use when the git command (like status) provides better human-readable output than the internal
     * Grgit objects.
     * <p>
     * Any output produced by the command simply goes to the console (it is not redirected).  Any exceptions are
     * trapped and reported.
     *
     * @param cmdArgs Arguments to pass to 'git'
     */
    void git(String... cmdArgs) {
        def argsStr = cmdArgs.join(' ')

        println '----------------------------------------------------------------------------'
        println "git $argsStr"
        try {
            project.exec {
                executable 'git'
                args cmdArgs
            }
        } catch (Exception e) {
            println "Failed to execute: git $argsStr"
            println("Message: $e.message")
        }
        println '----------------------------------------------------------------------------'
    }

    /**
     * Create a short description of a commit.
     */
    static String formatCommit(Commit commit) {
        return "Commit ${commit.getAbbreviatedId()} by $commit.committer.name on $commit.dateTime msg='$commit.shortMessage'"
    }
}
