package gov.raptor.gradle.plugins.buildsupport

import org.gradle.api.InvalidUserDataException

/**
 * Base class for 'release' oriented tasks.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.3
 */
class AReleaseTask extends ARepoTask {

    protected static final String[] RELEASE_OPTIONS = REPO_OPTIONS + [
            "-Pfinal \tThis is a FINAL build, so no stage is appended to the version",
            "-Prc=<TAG>\tThis is a pre-release (RC) build with the specified name (e.g., -Prc=RC1)",
    ]

    protected String makeDescription(String desc) {
        return makeDescription(desc, RELEASE_OPTIONS)
    }

    protected void validateStageProperties() {
        if (project.ext.finalBuild) {
            if (project.ext.rcBuild) throw new InvalidUserDataException("You can't specify both -Pfinal and -Prc")
        } else {
            if (!project.ext.rcBuild) throw new InvalidUserDataException("You must specify one of -Pfinal or -Prc")
        }
    }

    /**
     * Update the version file in this repo using the provided closure, which must accept a single argument of
     * type {@code RaptorVersion}.  The closure should update the version object as appropriate and then
     * that updated version will be written to the version file, committed, and pushed to the remote.
     *
     * @param updateOp The update operation to perform on the version
     * @since 0.3
     */
    void updateReleaseVersion(Closure updateOp) {

        String versionString = project.versionFile.text.trim()
        RaptorVersion theVersion = new RaptorVersion(versionString)

        logger.lifecycle "Current version: $theVersion"

        updateOp(theVersion) // Apply the update operation

        logger.lifecycle "New version: $theVersion"

        // If the version string hasn't changed, then don't bother writing/committing it
        if (theVersion.toString() == versionString) {
            logger.lifecycle "Version unchanged; skipping update/commit."
            return
        }

        logger.lifecycle "Set $project.versionFile = $theVersion"
        if (!project.noExec) project.versionFile.text = theVersion.toString()

        // Now commit the change and push it up

        logger.lifecycle "git add $project.versionFile.name"
        if (!project.noExec) project.grgit.add(patterns: [project.versionFile.name])

        def commitMessage = "Roll version to $theVersion"
        logger.lifecycle "git commit -m '$commitMessage'"
        if (!project.noExec) project.grgit.commit(message: commitMessage)

        def branchName = project.grgit.branch.current.fullName
        logger.lifecycle "git push $branchName ${project.noPush ? '(skipped)' : ''}"
        if (!project.noExec && !project.noPush) {
            project.grgit.push(refsOrSpecs: [branchName])
        }
    }
}
