package com.android.billingclient.api

import android.app.Activity
import android.app.PendingIntent
import android.content.*
import android.os.Bundle
import android.os.Handler
import android.os.IBinder
import android.os.RemoteException
import android.support.annotation.UiThread
import android.support.annotation.VisibleForTesting
import android.support.annotation.WorkerThread
import android.text.TextUtils
import com.android.billingclient.BuildConfig
import com.android.billingclient.api.Purchase.PurchasesResult
import com.android.billingclient.api.SkuDetails.SkuDetailsResult
import com.android.billingclient.util.BillingHelper
import com.android.billingclient.util.BillingHelper.INAPP_CONTINUATION_TOKEN
import com.android.billingclient.util.BillingHelper.RESPONSE_BUY_INTENT
import com.android.vending.billing.IInAppBillingService
import com.unitypay.billingmodule.BillingToClient
import org.json.JSONException
import java.util.*
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
/**
 * Implementation of [BillingClient] for communication between the in-app billing library and
 * client's application code.
 */
class BillingToto @UiThread
constructor(context: Context, listener: PurchasesUpdatedListener) : BillingToClient() {

    private var mClientState = ClientState.DISCONNECTED

    /** Main thread handler to post results from Executor.  */
    private val mUiThreadHandler = Handler()

    /** Context of the application that initialized this client.  */
    private val mApplicationContext: Context = context.applicationContext

    /**
     * Wrapper on top of PURCHASES_UPDATED broadcast receiver to return all purchases receipts to the
     * developer in one place for both app initiated and Play Store initated purhases.
     */
    private val mBroadcastManager: BillingBroadcastManager = BillingBroadcastManager(mApplicationContext, listener)


    /** Service binder  */
    private var mService: IInAppBillingService? = null

    /** Connection to the service.  */
    private var mServiceConnection: ServiceConnection? = null

    /** If subscriptions are is supported (for billing v3 and higher) or not.  */
    private var mSubscriptionsSupported: Boolean = false

    /** If subscription update is supported (for billing v5 and higher) or not.  */
    private var mSubscriptionUpdateSupported: Boolean = false

    /**
     * If purchaseHistory and buyIntentExtraParams are supported (for billing v6 and higher) or not.
     */
    private var mIABv6Supported: Boolean = false

    /**
     * Service that helps us to keep a pool of background threads suitable for current device specs.
     */
    private var mExecutorService: ExecutorService? = null

    /**
     * This receiver is triggered by local broadcast, once [ProxyBillingActivity] got the result
     * from Play Billing service in regards to current purchase process.
     */
    private val onPurchaseFinishedReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            val purchasesUpdatedListener = mBroadcastManager?.listener
            if (purchasesUpdatedListener == null) {
                BillingHelper.logWarn(
                        TAG, "PurchasesUpdatedListener is null - no way to return the response.")
                return
            }
            // Receiving the result from local broadcast and triggering a callback on listener.
            @BillingResponse
            val responseCode = intent.getIntExtra(ProxyBillingActivity.RESPONSE_CODE, BillingClient.BillingResponse.ERROR)
            val resultData = intent.getBundleExtra(ProxyBillingActivity.RESPONSE_BUNDLE)
            val purchases = BillingHelper.extractPurchases(resultData)
            purchasesUpdatedListener.onPurchasesUpdated(responseCode, purchases)
        }
    }

    /** Possible client/billing service relationship states.  */
    class ClientState {
        companion object {
            /** This client was not yet connected to billing service or was already disconnected from it.  */
            const val DISCONNECTED = 0
            /** This client is currently in process of connecting to billing service.  */
            const val CONNECTING = 1
            /** This client is currently connected to billing service.  */
            const val CONNECTED = 2
            /** This client was already closed and shouldn't be used again.  */
            const val CLOSED = 3
        }
    }

    init {
    }

    @BillingResponse
    override fun isFeatureSupported(@FeatureType feature: String): Int {
        if (!isReady) {
            return BillingClient.BillingResponse.SERVICE_DISCONNECTED
        }

        when (feature) {
            BillingClient.FeatureType.SUBSCRIPTIONS -> return if (mSubscriptionsSupported) BillingClient.BillingResponse.OK else BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED

            BillingClient.FeatureType.SUBSCRIPTIONS_UPDATE -> return if (mSubscriptionUpdateSupported)
                BillingClient.BillingResponse.OK
            else
                BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED

            BillingClient.FeatureType.IN_APP_ITEMS_ON_VR -> return isBillingSupportedOnVr(BillingClient.SkuType.INAPP)

            BillingClient.FeatureType.SUBSCRIPTIONS_ON_VR -> return isBillingSupportedOnVr(BillingClient.SkuType.SUBS)

            else -> {
                BillingHelper.logWarn(TAG, "Unsupported feature: $feature")
                return BillingClient.BillingResponse.DEVELOPER_ERROR
            }
        }
    }

    override fun isReady(): Boolean {
        return mClientState == ClientState.CONNECTED && mService != null && mServiceConnection != null
    }

    override fun startConnection(listener: BillingClientStateListener) {
        if (isReady) {
            BillingHelper.logVerbose(TAG, "Service connection is valid. No need to re-initialize.")
            listener.onBillingSetupFinished(BillingClient.BillingResponse.OK)
            return
        }

        if (mClientState == ClientState.CONNECTING) {
            BillingHelper.logWarn(
                    TAG, "Client is already in the process of connecting to billing service.")
            listener.onBillingSetupFinished(BillingClient.BillingResponse.DEVELOPER_ERROR)
            return
        }

        if (mClientState == ClientState.CLOSED) {
            BillingHelper.logWarn(
                    TAG, "Client was already closed and can't be reused. Please create another instance.")
            listener.onBillingSetupFinished(BillingClient.BillingResponse.DEVELOPER_ERROR)
            return
        }

        // Switch current state to connecting to avoid race conditions
        mClientState = ClientState.CONNECTING

        // Start listening for asynchronous purchase results via PURCHASES_UPDATED broadcasts
        mBroadcastManager.registerReceiver()

        // Subscribe to LocalBroadcastManager to get the synchronous purchase result from
        // ProxyBillingActivity
        val purchaseIntent = IntentFilter(ProxyBillingActivity.RESPONSE_INTENT_ACTION)
        LocalBroadcastManager.getInstance(mApplicationContext)
                .registerReceiver(onPurchaseFinishedReceiver, purchaseIntent)

        // Connection to billing service
        BillingHelper.logVerbose(TAG, "Starting in-app billing setup.")
        mServiceConnection = BillingServiceConnection(listener)

        val serviceIntent = Intent("com.android.vending.billing.InAppBillingService.BIND")
        serviceIntent.setPackage("com.android.vending")
        val intentServices = mApplicationContext.packageManager.queryIntentServices(serviceIntent, 0)

        if (intentServices != null && !intentServices.isEmpty()) {
            // Get component info and create ComponentName
            val resolveInfo = intentServices[0]
            if (resolveInfo.serviceInfo != null) {
                val packageName = resolveInfo.serviceInfo.packageName
                val className = resolveInfo.serviceInfo.name
                if ("com.android.vending" == packageName && className != null) {
                    val component = ComponentName(packageName, className)
                    // Specify component explicitly and don't allow stripping or replacing the package name
                    // to avoid exceptions inside 3rd party apps when Play Store was hacked:
                    // "IllegalArgumentException: Service Intent must be explicit".
                    // See: https://github.com/googlesamples/android-play-billing/issues/62 for more context.
                    val explicitServiceIntent = Intent(serviceIntent)
                    explicitServiceIntent.component = component
                    explicitServiceIntent.putExtra(LIBRARY_VERSION_KEY, LIBRARY_VERSION)
                    val connectionResult = mApplicationContext.bindService(
                            explicitServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE)
                    if (connectionResult) {
                        // Service connected successfully, listener will be called from mServiceConnection
                        BillingHelper.logVerbose(TAG, "Service was bonded successfully.")
                        return
                    } else {
                        // Service connection was blocked (e.g. this could happen in China), so we are closing
                        // the connection and notifying the listener
                        BillingHelper.logWarn(TAG, "Connection to Billing service is blocked.")
                    }
                } else {
                    // Play Store package name is not valid, ending connection
                    BillingHelper.logWarn(TAG, "The device doesn't have valid Play Store.")
                }
            }
        }
        // No service available to handle that Intent or service connection was blocked
        mClientState = ClientState.DISCONNECTED
        BillingHelper.logVerbose(TAG, "Billing service unavailable on device.")
        listener.onBillingSetupFinished(BillingClient.BillingResponse.BILLING_UNAVAILABLE)
    }

    override fun endConnection() {
        try {
            LocalBroadcastManager.getInstance(mApplicationContext)
                    .unregisterReceiver(onPurchaseFinishedReceiver)
            mBroadcastManager.destroy()
            if (mServiceConnection != null && mService != null) {
                BillingHelper.logVerbose(TAG, "Unbinding from service.")
                mApplicationContext.unbindService(mServiceConnection)
                mServiceConnection = null
            }
            mService = null
            if (mExecutorService != null) {
                mExecutorService!!.shutdownNow()
                mExecutorService = null
            }
        } catch (ex: Exception) {
            BillingHelper.logWarn(TAG, "There was an exception while ending connection: $ex")
        } finally {
            mClientState = ClientState.CLOSED
        }
    }

    override fun launchBillingFlow(activity: Activity, params: BillingFlowParams): Int {
        if (!isReady) {
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.SERVICE_DISCONNECTED)
        }

        @SkuType val skuType = params.skuType
        val newSku = params.sku

        // Checking for mandatory params fields
        if (newSku == null) {
            BillingHelper.logWarn(TAG, "Please fix the input params. SKU can't be null.")
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.DEVELOPER_ERROR)
        }

        if (skuType == null) {
            BillingHelper.logWarn(TAG, "Please fix the input params. SkuType can't be null.")
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.DEVELOPER_ERROR)
        }

        // Checking for requested features support
        if (skuType == BillingClient.SkuType.SUBS && !mSubscriptionsSupported) {
            BillingHelper.logWarn(TAG, "Current client doesn't support subscriptions.")
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED)
        }

        val isSubscriptionUpdate = params.oldSku != null

        if (isSubscriptionUpdate && !mSubscriptionUpdateSupported) {
            BillingHelper.logWarn(TAG, "Current client doesn't support subscriptions update.")
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED)
        }

        if (params.hasExtraParams() && !mIABv6Supported) {
            BillingHelper.logWarn(TAG, "Current client doesn't support extra params for buy intent.")
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED)
        }

        try {
            BillingHelper.logVerbose(
                    TAG, "Constructing buy intent for $newSku, item type: $skuType")

            val buyIntentBundle: Bundle
            // If IAB v6 is supported, we always try to use buyIntentExtraParams and report the version
            if (mIABv6Supported) {
                val extraParams = constructExtraParams(params)
                extraParams.putString(LIBRARY_VERSION_KEY, LIBRARY_VERSION)
                val apiVersion = if (params.vrPurchaseFlow) 7 else 6
                buyIntentBundle = mService!!.getBuyIntentExtraParams(
                        apiVersion,
                        mApplicationContext.packageName,
                        newSku,
                        skuType, null,
                        extraParams)
            } else if (isSubscriptionUpdate) {
                // For subscriptions update we are calling corresponding service method
                buyIntentBundle = mService!!.getBuyIntentToReplaceSkus(
                        /* apiVersion */ 5,
                        mApplicationContext.packageName,
                        Arrays.asList(params.oldSku),
                        newSku,
                        BillingClient.SkuType.SUBS, null)/* developerPayload */
            } else {
                buyIntentBundle = mService!!.getBuyIntent(
                        /* apiVersion */ 3,
                        mApplicationContext.packageName,
                        newSku,
                        skuType, null)/* developerPayload */
            }

            val responseCode = BillingHelper.getResponseCodeFromBundle(buyIntentBundle, TAG)
            if (responseCode != BillingClient.BillingResponse.OK) {
                BillingHelper.logWarn(TAG, "Unable to buy item, Error response code: $responseCode")
                return broadcastFailureAndReturnBillingResponse(responseCode)
            }
            // Launching an invisible activity that will handle the purchase result
            val intent = Intent(activity, ProxyBillingActivity::class.java)
            val pendingIntent = buyIntentBundle.getParcelable<PendingIntent>(RESPONSE_BUY_INTENT)
            intent.putExtra(RESPONSE_BUY_INTENT, pendingIntent)
            // We need an activity reference here to avoid using FLAG_ACTIVITY_NEW_TASK.
            // But we don't want to keep a reference to it inside the field to avoid memory leaks.
            // Plus all the other methods need just a Context reference, so could be used from the
            // Service or Application.
            activity.startActivity(intent)
        } catch (e: RemoteException) {
            val msg = ("RemoteException while launching launching replace subscriptions flow: "
                    + "; for sku: "
                    + newSku
                    + "; try to reconnect")
            BillingHelper.logWarn(TAG, msg)
            return broadcastFailureAndReturnBillingResponse(BillingClient.BillingResponse.SERVICE_DISCONNECTED)
        }

        return BillingClient.BillingResponse.OK
    }

    private fun broadcastFailureAndReturnBillingResponse(@BillingResponse responseCode: Int): Int {
        mBroadcastManager.listener.onPurchasesUpdated(responseCode, null)/* List<Purchase>= */
        return responseCode
    }

    override fun queryPurchases(@SkuType skuType: String): Purchase.PurchasesResult {
        if (!isReady) {
            return Purchase.PurchasesResult(BillingClient.BillingResponse.SERVICE_DISCONNECTED, null)/* purchasesList */
        }

        // Checking for the mandatory argument
        if (TextUtils.isEmpty(skuType)) {
            BillingHelper.logWarn(TAG, "Please provide a valid SKU type.")
            return Purchase.PurchasesResult(BillingClient.BillingResponse.DEVELOPER_ERROR, null)/* purchasesList */
        }

        return queryPurchasesInternal(skuType, false /* queryHistory */)
    }

    override fun querySkuDetailsAsync(
            params: SkuDetailsParams, listener: SkuDetailsResponseListener) {
        if (!isReady) {
            listener.onSkuDetailsResponse(
                    BillingClient.BillingResponse.SERVICE_DISCONNECTED, null)/* skuDetailsList */
            return
        }

        @SkuType val skuType = params.skuType
        val skusList = params.skusList

        // Checking for mandatory params fields
        if (TextUtils.isEmpty(skuType)) {
            BillingHelper.logWarn(TAG, "Please fix the input params. SKU type can't be empty.")
            listener.onSkuDetailsResponse(BillingClient.BillingResponse.DEVELOPER_ERROR, null)/* skuDetailsList */
            return
        }

        if (skusList == null) {
            BillingHelper.logWarn(TAG, "Please fix the input params. The list of SKUs can't be empty.")
            listener.onSkuDetailsResponse(BillingClient.BillingResponse.DEVELOPER_ERROR, null)/* skuDetailsList */
            return
        }

        executeAsync(
                Runnable {
                    val result = querySkuDetailsInternal(skuType, skusList)
                    // Post the result to main thread
                    postToUiThread(
                            Runnable {
                                listener.onSkuDetailsResponse(
                                        result.responseCode, result.skuDetailsList)
                            })
                })
    }

    override fun consumeAsync(purchaseToken: String, listener: ConsumeResponseListener) {
        if (!isReady) {
            listener.onConsumeResponse(BillingClient.BillingResponse.SERVICE_DISCONNECTED, null)/* purchaseToken */
            return
        }

        // Checking for the mandatory argument
        if (TextUtils.isEmpty(purchaseToken)) {
            BillingHelper.logWarn(
                    TAG, "Please provide a valid purchase token got from queryPurchases result.")
            listener.onConsumeResponse(BillingClient.BillingResponse.DEVELOPER_ERROR, purchaseToken)
            return
        }

        executeAsync(
                Runnable { consumeInternal(purchaseToken, listener) })
    }

    override fun queryPurchaseHistoryAsync(
            @SkuType skuType: String, listener: PurchaseHistoryResponseListener) {
        if (!isReady) {
            listener.onPurchaseHistoryResponse(
                    BillingClient.BillingResponse.SERVICE_DISCONNECTED, null)/* purchasesList */
            return
        }

        executeAsync(
                Runnable {
                    val result = queryPurchasesInternal(skuType, /* queryHistory */ true)

                    // Post the result to main thread
                    postToUiThread(
                            Runnable {
                                listener.onPurchaseHistoryResponse(
                                        result.responseCode, result.purchasesList)
                            })
                })
    }

    private fun constructExtraParams(params: BillingFlowParams): Bundle {
        val extraParams = Bundle()

        if (params.replaceSkusProrationMode != BillingFlowParams.ProrationMode.UNKNOWN_SUBSCRIPTION_UPGRADE_DOWNGRADE_POLICY) {
            extraParams.putInt(
                    BillingFlowParams.EXTRA_PARAM_KEY_REPLACE_SKUS_PRORATION_MODE,
                    params.replaceSkusProrationMode)
        }
        if (params.accountId != null) {
            extraParams.putString(BillingFlowParams.EXTRA_PARAM_KEY_ACCOUNT_ID, params.accountId)
        }
        if (params.vrPurchaseFlow) {
            extraParams.putBoolean(BillingFlowParams.EXTRA_PARAM_KEY_VR, true)
        }
        if (params.oldSku != null) {
            extraParams.putStringArrayList(
                    BillingFlowParams.EXTRA_PARAM_KEY_OLD_SKUS,
                    ArrayList(Arrays.asList(params.oldSku)))
        }

        return extraParams
    }

    private fun executeAsync(runnable: Runnable) {
        if (mExecutorService == null) {
            mExecutorService = Executors.newFixedThreadPool(BillingHelper.NUMBER_OF_CORES)
        }

        mExecutorService!!.submit(runnable)
    }

    /** Checks if billing on VR is supported for corresponding billing type.  */
    private fun isBillingSupportedOnVr(@SkuType skuType: String): Int {
        try {
            val supportedResult = mService!!.isBillingSupportedExtraParams(
                    7 /* apiVersion */,
                    mApplicationContext.packageName,
                    skuType,
                    generateVrBundle())
            return if (supportedResult == BillingClient.BillingResponse.OK)
                BillingClient.BillingResponse.OK
            else
                BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED
        } catch (e: RemoteException) {
            BillingHelper.logWarn(
                    TAG, "RemoteException while checking if billing is supported; " + "try to reconnect")
            return BillingClient.BillingResponse.SERVICE_DISCONNECTED
        }

    }

    /**
     * Generates a Bundle to indicate that we are request a method for VR experience within
     * extraParams
     */
    private fun generateVrBundle(): Bundle {
        val result = Bundle()
        result.putBoolean(BillingFlowParams.EXTRA_PARAM_KEY_VR, true)
        return result
    }

    @VisibleForTesting
    private fun querySkuDetailsInternal(@SkuType skuType: String, skuList: List<String>?): SkuDetails.SkuDetailsResult {
        val resultList = ArrayList<SkuDetails>()

        // Split the sku list into packs of no more than MAX_SKU_DETAILS_ITEMS_PER_REQUEST elements
        var startIndex = 0
        val listSize = skuList!!.size
        while (startIndex < listSize) {
            // Prepare a network request up to a maximum amount of supported elements
            var endIndex = startIndex + MAX_SKU_DETAILS_ITEMS_PER_REQUEST
            if (endIndex > listSize) {
                endIndex = listSize
            }
            val curSkuList = ArrayList(skuList.subList(startIndex, endIndex))
            val querySkus = Bundle()
            querySkus.putStringArrayList(GET_SKU_DETAILS_ITEM_LIST, curSkuList)
            querySkus.putString(LIBRARY_VERSION_KEY, LIBRARY_VERSION)
            val skuDetails: Bundle?
            try {
                skuDetails = mService!!.getSkuDetails(3, mApplicationContext.packageName, skuType, querySkus)
            } catch (e: RemoteException) {
                val msg = "querySkuDetailsAsync got a remote exception (try to reconnect): $e"
                BillingHelper.logWarn(TAG, msg)
                return SkuDetails.SkuDetailsResult(
                    BillingClient.BillingResponse.SERVICE_DISCONNECTED, null
                )/* skuDetailsList */
            }

            if (skuDetails == null) {
                BillingHelper.logWarn(TAG, "querySkuDetailsAsync got null sku details list")
                return SkuDetailsResult(BillingClient.BillingResponse.ITEM_UNAVAILABLE, null)/* skuDetailsList */
            }

            if (!skuDetails.containsKey(BillingHelper.RESPONSE_GET_SKU_DETAILS_LIST)) {
                @BillingResponse
                val responseCode = BillingHelper.getResponseCodeFromBundle(skuDetails, TAG)

                if (responseCode != BillingClient.BillingResponse.OK) {
                    BillingHelper.logWarn(TAG, "getSkuDetails() failed. Response code: $responseCode")
                    return SkuDetailsResult(responseCode, resultList)
                } else {
                    BillingHelper.logWarn(
                            TAG,
                            "getSkuDetails() returned a bundle with neither" + " an error nor a detail list.")
                    return SkuDetailsResult(BillingClient.BillingResponse.ERROR, resultList)
                }
            }

            val skuDetailsJsonList = skuDetails.getStringArrayList(BillingHelper.RESPONSE_GET_SKU_DETAILS_LIST)

            if (skuDetailsJsonList == null) {
                BillingHelper.logWarn(TAG, "querySkuDetailsAsync got null response list")
                return SkuDetailsResult(BillingClient.BillingResponse.ITEM_UNAVAILABLE, null)/* skuDetailsList */
            }

            for (i in skuDetailsJsonList.indices) {
                val thisResponse = skuDetailsJsonList[i]
                val currentSkuDetails: SkuDetails
                try {
                    currentSkuDetails = SkuDetails(thisResponse)
                } catch (e: JSONException) {
                    BillingHelper.logWarn(TAG, "Got a JSON exception trying to decode SkuDetails")
                    return SkuDetailsResult(BillingClient.BillingResponse.ERROR, null)/* skuDetailsList */
                }

                BillingHelper.logVerbose(TAG, "Got sku details: $currentSkuDetails")
                resultList.add(currentSkuDetails)
            }

            // Switching start index to the end of just received pack
            startIndex += MAX_SKU_DETAILS_ITEMS_PER_REQUEST
        }

        return SkuDetailsResult(BillingClient.BillingResponse.OK, resultList)
    }

    /**
     * Queries purchases or purchases history and combines all the multi-page results into one list
     */
    private fun queryPurchasesInternal(@SkuType skuType: String, queryHistory: Boolean): PurchasesResult {
        BillingHelper.logVerbose(
                TAG, "Querying owned items, item type: $skuType; history: $queryHistory")

        var continueToken: String? = null
        val resultList = ArrayList<Purchase>()

        do {
            val ownedItems: Bundle?
            try {
                if (queryHistory) {
                    // If current client doesn't support IABv6, then there is no such method yet
                    if (!mIABv6Supported) {
                        BillingHelper.logWarn(TAG, "getPurchaseHistory is not supported on current device")
                        return PurchasesResult(
                                BillingClient.BillingResponse.FEATURE_NOT_SUPPORTED, null)/* purchasesList */
                    }
                    ownedItems = mService!!.getPurchaseHistory(
                            /* apiVersion */ 6,
                            mApplicationContext.packageName,
                            skuType,
                            continueToken, null)/* extraParams */
                } else {
                    ownedItems = mService!!.getPurchases(
                            3 /* apiVersion */, mApplicationContext.packageName, skuType, continueToken)
                }
            } catch (e: RemoteException) {
                BillingHelper.logWarn(
                        TAG, "Got exception trying to get purchases: $e; try to reconnect")
                return PurchasesResult(BillingClient.BillingResponse.SERVICE_DISCONNECTED, null)/* purchasesList */
            }

            if (ownedItems == null) {
                BillingHelper.logWarn(TAG, "queryPurchases got null owned items list")
                return PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
            }

            @BillingResponse val responseCode = BillingHelper.getResponseCodeFromBundle(ownedItems, TAG)

            if (responseCode != BillingClient.BillingResponse.OK) {
                BillingHelper.logWarn(TAG, "getPurchases() failed. Response code: $responseCode")
                return PurchasesResult(responseCode, null)/* purchasesList */
            }

            if (!ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_ITEM_LIST)
                    || !ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_PURCHASE_DATA_LIST)
                    || !ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_SIGNATURE_LIST)) {
                BillingHelper.logWarn(
                        TAG, "Bundle returned from getPurchases() doesn't contain required fields.")
                return PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
            }

            val ownedSkus = ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_ITEM_LIST)
            val purchaseDataList = ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_PURCHASE_DATA_LIST)
            val signatureList = ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_SIGNATURE_LIST)

            if (ownedSkus == null) {
                BillingHelper.logWarn(TAG, "Bundle returned from getPurchases() contains null SKUs list.")
                return PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
            }

            if (purchaseDataList == null) {
                BillingHelper.logWarn(
                        TAG, "Bundle returned from getPurchases() contains null purchases list.")
                return PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
            }

            if (signatureList == null) {
                BillingHelper.logWarn(
                        TAG, "Bundle returned from getPurchases() contains null signatures list.")
                return Purchase.PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
            }

            for (i in purchaseDataList.indices) {
                val purchaseData = purchaseDataList[i]
                val signature = signatureList[i]
                val sku = ownedSkus[i]

                BillingHelper.logVerbose(TAG, "Sku is owned: $sku")
                val purchase: Purchase
                try {
                    purchase = Purchase(purchaseData, signature)
                } catch (e: JSONException) {
                    BillingHelper.logWarn(TAG, "Got an exception trying to decode the purchase: $e")
                    return PurchasesResult(BillingClient.BillingResponse.ERROR, null)/* purchasesList */
                }

                if (TextUtils.isEmpty(purchase.purchaseToken)) {
                    BillingHelper.logWarn(TAG, "BUG: empty/null token!")
                }

                resultList.add(purchase)
            }

            continueToken = ownedItems.getString(INAPP_CONTINUATION_TOKEN)
//            BillingHelper.logVerbose(TAG, "Continuation token: " + continueToken!!)
        } while (!TextUtils.isEmpty(continueToken))

        return PurchasesResult(BillingClient.BillingResponse.OK, resultList)
    }

    /** Execute the runnable on the UI/Main Thread  */
    private fun postToUiThread(runnable: Runnable) {
        mUiThreadHandler.post(runnable)
    }

    /** Consume the purchase and execute listener's callback on the Ui/Main thread  */
    @WorkerThread
    private fun consumeInternal(purchaseToken: String, listener: ConsumeResponseListener?) {
        try {
            BillingHelper.logVerbose(TAG, "Consuming purchase with token: $purchaseToken")
            @BillingResponse val responseCode = mService!!.consumePurchase(
                    3 /* apiVersion */, mApplicationContext.packageName, purchaseToken)

            if (responseCode == BillingClient.BillingResponse.OK) {
                BillingHelper.logVerbose(TAG, "Successfully consumed purchase.")
                if (listener != null) {
                    postToUiThread(
                            Runnable { listener.onConsumeResponse(responseCode, purchaseToken) })
                }
            } else {
                BillingHelper.logWarn(
                        TAG, "Error consuming purchase with token. Response code: $responseCode")

                postToUiThread(
                        Runnable {
                            BillingHelper.logWarn(TAG, "Error consuming purchase.")
                            listener!!.onConsumeResponse(responseCode, purchaseToken)
                        })
            }
        } catch (e: RemoteException) {
            postToUiThread(
                    Runnable {
                        BillingHelper.logWarn(TAG, "Error consuming purchase; ex: $e")
                        listener!!.onConsumeResponse(BillingClient.BillingResponse.SERVICE_DISCONNECTED, purchaseToken)
                    })
        }

    }

    /** Connect with Billing service and notify listener about important states.  */
    private inner class BillingServiceConnection constructor(private val mListener: BillingClientStateListener) : ServiceConnection {

        init {
            if (mListener == null) {
                throw RuntimeException("Please specify a listener to know when init is done.")
            }
        }

        override fun onServiceDisconnected(name: ComponentName) {
            BillingHelper.logWarn(TAG, "Billing service disconnected.")
            mService = null
            mClientState = ClientState.DISCONNECTED
            mListener.onBillingServiceDisconnected()
        }

        override fun onServiceConnected(name: ComponentName, service: IBinder) {
            BillingHelper.logVerbose(TAG, "Billing service connected.")

            mService = IInAppBillingService.Stub.asInterface(service)
            val packageName = mApplicationContext.packageName
            mSubscriptionsSupported = false
            mSubscriptionUpdateSupported = false
            mIABv6Supported = false
            // The code below tries to figure out, which IAB API version and features are supported by
            // current client (Play Store app on this device).
            // First, it checks for IABv6 with subscriptions to avoid extra queries for most of the
            // clients.
            // Second, it checks if IABv6 is supported for in-app only items.
            // Third, it checks for IABv5 for subscription only (since only getBuyIntentToReplaceSku was
            // added during that release).
            // And finally it checks for IABv3 for both subscriptions and inapp items.
            try {
                // Check for in-app billing API v6 support with subscriptions. This is needed for
                // getPurchaseHistory and getBuyIntentExtraParams methods.
                @BillingResponse
                var response = mService!!.isBillingSupported(/* apiVersion */6, packageName, BillingClient.SkuType.SUBS)

                if (response == BillingClient.BillingResponse.OK) {
                    BillingHelper.logVerbose(TAG, "In-app billing API version 6 with subs is supported.")
                    mIABv6Supported = true
                    mSubscriptionsSupported = true
                    mSubscriptionUpdateSupported = true
                } else {
                    // Check for in-app billing API v6 support without subcriptions. This is needed for
                    // getPurchaseHistory and getBuyIntentExtraParams methods.

                    response = mService!!.isBillingSupported(/* apiVersion */6, packageName, BillingClient.SkuType.INAPP)

                    if (response == BillingClient.BillingResponse.OK) {
                        BillingHelper.logVerbose(TAG, "In-app billing API without subs version 6 supported.")
                        mIABv6Supported = true
                    }

                    // Check for in-app billing API v5 support. This is needed for
                    // getBuyIntentToReplaceSku which allows for subscription update

                    response = mService!!.isBillingSupported(/* apiVersion */5, packageName, BillingClient.SkuType.SUBS)

                    if (response == BillingClient.BillingResponse.OK) {
                        BillingHelper.logVerbose(TAG, "In-app billing API version 5 supported.")
                        mSubscriptionUpdateSupported = true
                        mSubscriptionsSupported = true
                    } else {
                        // Check for in-app billing API v3 support with subscriptions

                        response = mService!!.isBillingSupported(/* apiVersion */3, packageName, BillingClient.SkuType.SUBS)

                        if (response == BillingClient.BillingResponse.OK) {
                            BillingHelper.logVerbose(
                                    TAG, "In-app billing API version 3 with subscriptions is supported.")
                            mSubscriptionsSupported = true
                        } else if (mIABv6Supported) {
                            // If IABv6 without subscriptions was already checked, then we return successful
                            // result even though subscriptions are not supported even for IABv3. This is a valid
                            // case for our Android Wear client, for example.
                            response = BillingClient.BillingResponse.OK
                        } else {
                            // Check for at least in-app billing API v3 support with in-app items only

                            response = mService!!.isBillingSupported(/* apiVersion */3, packageName, BillingClient.SkuType.INAPP)

                            if (response == BillingClient.BillingResponse.OK) {
                                BillingHelper.logVerbose(
                                        TAG, "In-app billing API version 3 with in-app items is supported.")
                            } else {
                                BillingHelper.logWarn(
                                        TAG, "Even billing API version 3 is not supported on this device.")
                            }
                        }
                    }
                }

                if (response == BillingClient.BillingResponse.OK) {
                    mClientState = ClientState.CONNECTED
                } else {
                    mClientState = ClientState.DISCONNECTED
                    mService = null
                }
                mListener.onBillingSetupFinished(response)
            } catch (e: RemoteException) {
                BillingHelper.logWarn(TAG, "RemoteException while setting up in-app billing$e")
                mClientState = ClientState.DISCONNECTED
                mService = null
                mListener.onBillingSetupFinished(BillingClient.BillingResponse.SERVICE_DISCONNECTED)
            }

        }
    }

    companion object {
        private val TAG = "BillingClient"

        /**
         * The maximum number of items than can be requested by a call to Billing service's
         * getSkuDetails() method
         */
        private val MAX_SKU_DETAILS_ITEMS_PER_REQUEST = 20

        /** A list of SKUs inside getSkuDetails request bundle.  */
        private val GET_SKU_DETAILS_ITEM_LIST = "ITEM_ID_LIST"

        /** Field's key to hold library version key constant.  */
        private val LIBRARY_VERSION_KEY = "libraryVersion"

        /** Version Name of the current library.  */
        private val LIBRARY_VERSION = BuildConfig.VERSION_NAME
    }
}
