(ns fastmath.optimization.lbfsgb
  (:require [fastmath.core :as m]
            [fastmath.vector :as v]))

(m/use-primitive-operators)
(set! *unchecked-math* :warn-on-boxed)

(def config {:factr 1.0e+7
           :pgtol 1.0e-5
           :m 25
           :l [1.0 -100.0 1.0 -100.0 1.0]
           :u [100.0 100 100.0 100.0 100.0]
           :x [3.0 3.0 3.0 3.0 3.0]
           :task :start
           :n 5})

(defn f [[^double x1 ^double x2 ^double x3 ^double x4 ^double x5]]
  (let [res (* 0.25 (m/sq (dec x1)))
        res (+ res (m/sq (- x2 (* x1 x1))))
        res (+ res (m/sq (- x3 (* x2 x2))))
        res (+ res (m/sq (- x4 (* x3 x3))))]
    (* 4.0 (+ res (m/sq (- x5 (* x4 x4)))))))

(defn g [[^double x1 ^double x2 ^double x3 ^double x4 ^double x5]]
  (let [t1 (- x2 (* x1 x1))
        g1 (- (* 2.0 (dec x1)) (* 16.0 x1 t1))
        t2 t1
        t1 (- x3 (* x2 x2))
        g2 (- (* 8.0 t2) (* 16.0 x2 t1))
        t2 t1
        t1 (- x4 (* x3 x3))
        g3 (- (* 8.0 t2) (* 16.0 x3 t1))
        t2 t1
        t1 (- x5 (* x4 x4))
        g4 (- (* 8.0 t2) (* 16.0 x4 t1))]
    [g1 g2 g3 g4 (* 8.0 t1)]))

(defn projgr
  ^double [x g l u]
  (->> (map (fn [^double xi ^double gi ^double li ^double ui]
              (m/abs (if (neg? gi)
                       (max (- xi ui) gi)
                       (min (- xi li) gi)))) x g l u)
       (reduce m/fast-max)))

(defn lbfgsb
  [f g {:keys [^double factr ^double pgtol ^long m ^long n l u x task]}]
  (let [isave1 (* m n)
        isave2 (* m m)
        isave3 (* 4 isave2)
        isave4 1
        isave5 (+ isave4 isave1)
        isave6 (+ isave5 isave1)
        isave7 (+ isave6 isave2)
        isave8 (+ isave7 isave2)
        isave9 (+ isave8 isave2)
        isave10 (+ isave9 isave3)
        isave11 (+ isave10 isave3)
        isave12 (+ isave11 n)
        isave13 (+ isave12 n)
        isave14 (+ isave13 n)
        isave15 (+ isave14 n)
        isave16 (+ isave15 n)
        lws  isave4
        lwy  isave5
        lsy  isave6
        lss  isave7
        lwt  isave8
        lwn  isave9
        lsnd isave10
        lz   isave11
        lr   isave12
        ld   isave13
        lt   isave14
        lxp  isave15
        lwa  isave16
        tol (* m/MACHINE-EPSILON factr)
        sbgnrm (projgr x (g x) l u)]

    sbgnrm))

(lbfgsb f g config)

(projgr (:x config) (g (:x config)) (:l config) (:u config))

#_(require '[criterium.core :as crit])

#_(crit/quick-bench (projgr (:x config) (g (:x config)) (:l config) (:u config)))


