/*
 * Copyright 2015-2016 Adrien "Litarvan" Navratil
 *
 * This file is part of the OpenLauncherLib.
 *
 * The OpenLauncherLib is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * The OpenLauncherLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the OpenLauncherLib.  If not, see <http://www.gnu.org/licenses/>.
 */
package fr.theshark34.openlauncherlib.util;

import fr.flowarg.openlauncherlib.ModifiedByFlow;
import fr.theshark34.openlauncherlib.FailException;

import java.io.BufferedWriter;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Properties;

/**
 * The Saver
 *
 * <p>
 * Save some things to a file.
 * </p>
 *
 * @author Litarvan
 * @version 3.0.2-BETA
 * @since 3.0.0-BETA
 */
@ModifiedByFlow
public class Saver
{
    /**
     * The file where to save the things
     */
    private final Path file;

    /**
     * The properties object
     */
    private final Properties properties;

    /**
     * The Saver
     *
     * @param file The file where to save the things
     */
    @Deprecated
    @ModifiedByFlow
    public Saver(File file)
    {
        this(file.toPath());
    }

    @ModifiedByFlow
    public Saver(Path file)
    {
        this.file = file;
        this.properties = new Properties();

        if (Files.exists(this.file)) this.load();
        else
        {
            try
            {
                Files.createDirectories(this.file.getParent());
                Files.createFile(this.file);
            } catch (Throwable e)
            {
                throw new FailException("Can't create properties file.");
            }
        }
    }

    /**
     * Set something
     *
     * @param key   The key
     * @param value The value
     */
    public void set(String key, String value)
    {
        this.properties.setProperty(key, value);
        this.save();
    }

    /**
     * Loads something
     *
     * @param key The key of the thing to get
     * @return The value if founded, or null
     */
    public String get(String key)
    {
        return this.properties.getProperty(key);
    }

    /**
     * Loads somethings with a default value
     *
     * @param key The key of the thing to get
     * @param def The default value
     * @return The value if founded, or the def if not
     */
    public String get(String key, String def)
    {
        final String value = this.properties.getProperty(key);
        return value == null ? def : value;
    }

    /**
     * Save the properties (automatic when you do {@link #set(String, String)})
     */
    @ModifiedByFlow
    public void save()
    {
        try
        {
            final BufferedWriter writer = Files.newBufferedWriter(this.file);
            this.properties.store(writer, "Generated by the OpenLauncherLib Saver");
            writer.close();
        } catch (Throwable t)
        {
            throw new FailException("Can't save the properties", t);
        }
    }

    /**
     * Load the properties (automatic when you do {@link #Saver(Path)}
     */
    @ModifiedByFlow
    public void load()
    {
        try
        {
            this.properties.load(Files.newInputStream(this.file));
        } catch (Throwable t)
        {
            throw new FailException("Can't load the properties", t);
        }
    }

    @ModifiedByFlow
    public void remove(String key)
    {
        this.properties.remove(key);
        this.save();
    }
}
