/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.livotov.labs.android.robotools.crypt;

import android.content.Context;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;

/**
 * Wraps {@link javax.crypto.SecretKey} instances using a public/private key pair stored in
 * the platform {@link java.security.KeyStore}. This allows us to protect symmetric keys with
 * hardware-backed crypto, if provided by the device.
 * <p>
 * See <a href="http://en.wikipedia.org/wiki/Key_Wrap">key wrapping</a> for more
 * details.
 * <p>
 * Not inherently thread safe.
 */
public class RTSecretKeyWrapper
{
    private final Cipher mCipher;
    private final KeyPair mPair;
    private final KeyStore mKeyStore;

    /**
     * Create a wrapper using the public/private key pair with the given alias.
     * If no pair with that alias exists, it will be generated.
     */
    public RTSecretKeyWrapper(Context context, String alias) throws GeneralSecurityException, IOException
    {
        mCipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");

        mKeyStore = KeyStore.getInstance("AndroidKeyStore");
        mKeyStore.load(null);

        if (!mKeyStore.containsAlias(alias))
        {
            RTCryptUtil.createRsaKey(context, alias);
        }

        // Even if we just generated the key, always read it back to ensure we
        // can read it successfully.
        final KeyStore.PrivateKeyEntry entry = (KeyStore.PrivateKeyEntry) mKeyStore.getEntry(alias, null);
        mPair = new KeyPair(entry.getCertificate().getPublicKey(), entry.getPrivateKey());
    }


    /**
     * Wrap a {@link javax.crypto.SecretKey} using the public key assigned to this wrapper.
     * Use {@link #unwrap(byte[])} to later recover the original
     * {@link javax.crypto.SecretKey}.
     *
     * @return a wrapped version of the given {@link javax.crypto.SecretKey} that can be
     * safely stored on untrusted storage.
     */
    public byte[] wrap(SecretKey key) throws GeneralSecurityException
    {
        mCipher.init(Cipher.WRAP_MODE, mPair.getPublic());
        return mCipher.wrap(key);
    }

    /**
     * Unwrap a {@link javax.crypto.SecretKey} using the private key assigned to this
     * wrapper.
     *
     * @param blob a wrapped {@link javax.crypto.SecretKey} as previously returned by
     *             {@link #wrap(javax.crypto.SecretKey)}.
     */
    public SecretKey unwrap(byte[] blob) throws GeneralSecurityException
    {
        mCipher.init(Cipher.UNWRAP_MODE, mPair.getPrivate());
        return (SecretKey) mCipher.unwrap(blob, "AES", Cipher.SECRET_KEY);
    }

    /**
     * Delete a {@link javax.crypto.SecretKey} from {@link KeyStore} for the give alia
     */
    public void removeKey(String alias)
    {
        try
        {
            if (mKeyStore.containsAlias(alias))
            {
                mKeyStore.deleteEntry(alias);
            }
        }
        catch (KeyStoreException e)
        {
            e.printStackTrace();
        }
    }
}