/**
 * DSS - Digital Signature Services
 * Copyright (C) 2015 European Commission, provided under the CEF programme
 * 
 * This file is part of the "DSS - Digital Signature Services" project.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package eu.europa.esig.dss.validation.reports;

import eu.europa.esig.dss.detailedreport.jaxb.XmlDetailedReport;
import eu.europa.esig.dss.diagnostic.jaxb.XmlDiagnosticData;
import eu.europa.esig.dss.simplereport.SimpleReport;
import eu.europa.esig.dss.simplereport.SimpleReportFacade;
import eu.europa.esig.dss.simplereport.jaxb.XmlSimpleReport;
import eu.europa.esig.validationreport.ValidationReportFacade;
import eu.europa.esig.validationreport.jaxb.ValidationReportType;
import org.xml.sax.SAXException;

import javax.xml.bind.JAXBException;
import java.io.IOException;

/**
 * This class is a container for all reports generated by the signature validation
 * process: diagnostic data, detailed report and simple report.
 */
public class Reports extends AbstractReports {

	/** Simple report */
	private final SimpleReport simpleReportWrapper;

	/** Simple report XML */
	private String xmlSimpleReport;

	/** ETSI validation report */
	private final ValidationReportType etsiValidationReport;

	/** XML validation report */
	private String xmlEtsiValidationReport;

	/**
	 * This is the default constructor to instantiate this container.
	 *
	 * @param diagnosticDataJaxb
	 *                           the JAXB {@code XmlDiagnosticData}
	 * @param detailedReport
	 *                           the JAXB {@code XmlDetailedReport}
	 * @param simpleReport
	 *                           the JAXB {@code XmlSimpleReport}
	 * @param validationReport
	 *                           the JAXB {@code ValidationReportType}
	 */
	public Reports(final XmlDiagnosticData diagnosticDataJaxb,
			final XmlDetailedReport detailedReport, final XmlSimpleReport simpleReport,
			final ValidationReportType validationReport) {
		super(diagnosticDataJaxb, detailedReport);
		this.simpleReportWrapper = new SimpleReport(simpleReport);
		this.etsiValidationReport = validationReport;
	}

	/**
	 * This method returns the wrapper to manipulate the JAXB SimpleReport
	 * 
	 * @return the wrapper {@code SimpleReport}
	 */
	public SimpleReport getSimpleReport() {
		return simpleReportWrapper;
	}

	/**
	 * This method returns the JAXB SimpleReport
	 * 
	 * @return the JAXB {@code XmlSimpleReport}
	 */
	public XmlSimpleReport getSimpleReportJaxb() {
		return simpleReportWrapper.getJaxbModel();
	}

	/**
	 * This method returns the JAXB EtsiValidationReport
	 * 
	 * @return the JAXB {@code ValidationReportType}
	 */
	public ValidationReportType getEtsiValidationReportJaxb() {
		return etsiValidationReport;
	}
	
	/**
	 * This method returns the XML representation of the JAXB SimpleReport String
	 * 
	 * @return a String with the XML content of the JAXB {@code SimpleReport}
	 * @throws DSSReportException - in case of marshalling error
	 */
	@Override
	public String getXmlSimpleReport() {
		try {
			if (xmlSimpleReport == null) {
				xmlSimpleReport = SimpleReportFacade.newFacade().marshall(getSimpleReportJaxb(), validateXml);
			}
			return xmlSimpleReport;
		} catch (JAXBException | IOException | SAXException e) {
			throw new DSSReportException("An error occurred during marshalling of JAXB Simple Report", e);
		}
	}

	/**
	 * This method returns the XML representation of the JAXB ETSI Validation Report
	 * String
	 * 
	 * @return a String with the XML content of the JAXB
	 *         {@code ValidationReportType}
	 * @throws DSSReportException - in case of marshalling error
	 */
	public String getXmlValidationReport() {
		try {
			if (xmlEtsiValidationReport == null) {
				xmlEtsiValidationReport = ValidationReportFacade.newFacade().marshall(getEtsiValidationReportJaxb(), validateXml);
			}
			return xmlEtsiValidationReport;
		} catch (JAXBException | IOException | SAXException e) {
			throw new DSSReportException("An error occurred during marshalling of JAXB Etsi Validation Report", e);
		}
	}

	@Override
	public void print() {
		System.out.println("----------------Diagnostic data-----------------");
		System.out.println(getXmlDiagnosticData());
		System.out.println("----------------Validation report---------------");
		System.out.println(getXmlDetailedReport());
		System.out.println("----------------Simple report-------------------");
		System.out.println(getXmlSimpleReport());
		System.out.println("----------------ETSI validation report-------------------");
		if (etsiValidationReport != null) {
			System.out.println(getXmlValidationReport());
		} else {
			System.out.println("---------- ETSI validation report is disabled -----------");
		}
		System.out.println("---------------------------------------------------------");
	}

}
