/**
 * DSS - Digital Signature Services
 * Copyright (C) 2015 European Commission, provided under the CEF programme
 * 
 * This file is part of the "DSS - Digital Signature Services" project.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package eu.europa.esig.dss.validation.reports;

import eu.europa.esig.dss.detailedreport.jaxb.XmlDetailedReport;
import eu.europa.esig.dss.diagnostic.jaxb.XmlDiagnosticData;
import eu.europa.esig.dss.simplecertificatereport.SimpleCertificateReport;
import eu.europa.esig.dss.simplecertificatereport.SimpleCertificateReportFacade;
import eu.europa.esig.dss.simplecertificatereport.jaxb.XmlSimpleCertificateReport;
import org.xml.sax.SAXException;

import javax.xml.bind.JAXBException;
import java.io.IOException;

/**
 * This class is a container for all reports generated by the certificate
 * validation process: diagnostic data, detailed report and simple report.
 */
public class CertificateReports extends AbstractReports {

	/** The simple certificate report */
	private final SimpleCertificateReport simpleReportWrapper;

	/** Simple certificate report XML */
	private String xmlSimpleReport;

	/**
	 * This is the default constructor to instantiate this container.
	 *
	 * @param diagnosticDataJaxb
	 *                           the JAXB {@code XmlDiagnosticData}
	 * @param detailedReport
	 *                           the JAXB {@code XmlDetailedReport}
	 * @param simpleReport
	 *                           the JAXB {@code XmlSimpleCertificateReport}
	 */
	public CertificateReports(final XmlDiagnosticData diagnosticDataJaxb, final XmlDetailedReport detailedReport,
			final XmlSimpleCertificateReport simpleReport) {
		super(diagnosticDataJaxb, detailedReport);
		this.simpleReportWrapper = new SimpleCertificateReport(simpleReport);
	}

	/**
	 * This method returns the wrapper to manipulate the JAXB
	 * SimpleCertificateReport
	 *
	 * @return the wrapper {@code SimpleCertificateReport}
	 */
	public SimpleCertificateReport getSimpleReport() {
		return simpleReportWrapper;
	}

	/**
	 * This method returns the JAXB XmlSimpleCertificateReport
	 * 
	 * @return the JAXB {@code XmlSimpleCertificateReport}
	 */
	public XmlSimpleCertificateReport getSimpleReportJaxb() {
		return simpleReportWrapper.getJaxbModel();
	}

	/**
	 * This method returns the XML representation of the JAXB
	 * XmlSimpleCertificateReport String
	 * 
	 * @return a String with the XML content of the JAXB
	 *         {@code XmlSimpleCertificateReport}
	 * @throws DSSReportException - in case of marshalling error
	 */
	@Override
	public String getXmlSimpleReport() throws DSSReportException {
		try {
			if (xmlSimpleReport == null) {
				xmlSimpleReport = SimpleCertificateReportFacade.newFacade().marshall(getSimpleReportJaxb(), validateXml);
			}
			return xmlSimpleReport;
		} catch (JAXBException | IOException | SAXException e) {
			throw new DSSReportException("An error occurred during marshalling of JAXB Simple Report", e);
		}
	}

}
