/*
 * $Id: PdfReaderInstance.java 3527 2008-07-06 15:34:38Z blowagie $
 *
 * Copyright 2001, 2002 Paulo Soares
 *
 * The contents of this file are subject to the Mozilla Public License Version 1.1
 * (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the License.
 *
 * The Original Code is 'iText, a free JAVA-PDF library'.
 *
 * The Initial Developer of the Original Code is Bruno Lowagie. Portions created by
 * the Initial Developer are Copyright (C) 1999, 2000, 2001, 2002 by Bruno Lowagie.
 * All Rights Reserved.
 * Co-Developer of the code is Paulo Soares. Portions created by the Co-Developer
 * are Copyright (C) 2000, 2001, 2002 by Paulo Soares. All Rights Reserved.
 *
 * Contributor(s): all the names of the contributors are added in the source code
 * where applicable.
 *
 * Alternatively, the contents of this file may be used under the terms of the
 * LGPL license (the "GNU LIBRARY GENERAL PUBLIC LICENSE"), in which case the
 * provisions of LGPL are applicable instead of those above.  If you wish to
 * allow use of your version of this file only under the terms of the LGPL
 * License and not to allow others to use your version of this file under
 * the MPL, indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by the LGPL.
 * If you do not delete the provisions above, a recipient may use your version
 * of this file under either the MPL or the GNU LIBRARY GENERAL PUBLIC LICENSE.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the MPL as stated above or under the terms of the GNU
 * Library General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Library general Public License for more
 * details.
 *
 * If you didn't download this code from the following link, you should check if
 * you aren't using an obsolete version:
 * http://www.lowagie.com/iText/
 */

package com.aowagie.text.pdf;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
/**
 * Instance of PdfReader in each output document.
 *
 * @author Paulo Soares (psoares@consiste.pt)
 */
class PdfReaderInstance {
    private static final PdfLiteral IDENTITYMATRIX = new PdfLiteral("[1 0 0 1 0 0]");
    private static final PdfNumber ONE = new PdfNumber(1);
    private final int myXref[];
    private final PdfReader reader;
    private final RandomAccessFileOrArray file;
    private final HashMap importedPages = new LinkedHashMap();
    private final PdfWriter writer;
    private final HashMap visited = new LinkedHashMap();
    private ArrayList nextRound = new ArrayList();

    PdfReaderInstance(final PdfReader reader, final PdfWriter writer) {
        this.reader = reader;
        this.writer = writer;
        this.file = reader.getSafeFile();
        this.myXref = new int[reader.getXrefSize()];
    }

    PdfReader getReader() {
        return this.reader;
    }

    PdfImportedPage getImportedPage(final int pageNumber) {
        if (!this.reader.isOpenedWithFullPermissions()) {
			throw new IllegalArgumentException("PdfReader not opened with owner password");
		}
        if (pageNumber < 1 || pageNumber > this.reader.getNumberOfPages()) {
			throw new IllegalArgumentException("Invalid page number: " + pageNumber);
		}
        final Integer i = Integer.valueOf(pageNumber);
        PdfImportedPage pageT = (PdfImportedPage)this.importedPages.get(i);
        if (pageT == null) {
            pageT = new PdfImportedPage(this, this.writer, pageNumber);
            this.importedPages.put(i, pageT);
        }
        return pageT;
    }

    int getNewObjectNumber(final int number, final int generation) {
        if (this.myXref[number] == 0) {
            this.myXref[number] = this.writer.getIndirectReferenceNumber();
            this.nextRound.add(Integer.valueOf(number));
        }
        return this.myXref[number];
    }

    RandomAccessFileOrArray getReaderFile() {
        return this.file;
    }

    PdfObject getResources(final int pageNumber) {
        final PdfObject obj = PdfReader.getPdfObjectRelease(this.reader.getPageNRelease(pageNumber).get(PdfName.RESOURCES));
        return obj;
    }

    /**
     * Gets the content stream of a page as a PdfStream object.
     * @param	pageNumber			the page of which you want the stream
     * @param	compressionLevel	the compression level you want to apply to the stream
     * @return	a PdfStream object
     * @since	2.1.3 (the method already existed without param compressionLevel)
     */
    PdfStream getFormXObject(final int pageNumber, final int compressionLevel) throws IOException {
        final PdfDictionary page = this.reader.getPageNRelease(pageNumber);
        final PdfObject contents = PdfReader.getPdfObjectRelease(page.get(PdfName.CONTENTS));
        final PdfDictionary dic = new PdfDictionary();
        byte bout[] = null;
        if (contents != null) {
            if (contents.isStream()) {
				dic.putAll((PRStream)contents);
			} else {
				bout = this.reader.getPageContent(pageNumber, this.file);
			}
        } else {
			bout = new byte[0];
		}
        dic.put(PdfName.RESOURCES, PdfReader.getPdfObjectRelease(page.get(PdfName.RESOURCES)));
        dic.put(PdfName.TYPE, PdfName.XOBJECT);
        dic.put(PdfName.SUBTYPE, PdfName.FORM);
        final PdfImportedPage impPage = (PdfImportedPage)this.importedPages.get(Integer.valueOf(pageNumber));
        dic.put(PdfName.BBOX, new PdfRectangle(impPage.getBoundingBox()));
        final PdfArray matrix = impPage.getMatrix();
        if (matrix == null) {
			dic.put(PdfName.MATRIX, IDENTITYMATRIX);
		} else {
			dic.put(PdfName.MATRIX, matrix);
		}
        dic.put(PdfName.FORMTYPE, ONE);
        PRStream stream;
        if (bout == null) {
            stream = new PRStream((PRStream)contents, dic);
        }
        else {
            stream = new PRStream(this.reader, bout, compressionLevel);
            stream.putAll(dic);
        }
        return stream;
    }

    private void writeAllVisited() throws IOException {
        while (!this.nextRound.isEmpty()) {
            final ArrayList vec = this.nextRound;
            this.nextRound = new ArrayList();
            for (int k = 0; k < vec.size(); ++k) {
                final Integer i = (Integer)vec.get(k);
                if (!this.visited.containsKey(i)) {
                    this.visited.put(i, null);
                    final int n = i.intValue();
                    this.writer.addToBody(this.reader.getPdfObjectRelease(n), this.myXref[n]);
                }
            }
        }
    }

    void writeAllPages() throws IOException {
        try {
            this.file.reOpen();
            for (final Iterator it = this.importedPages.values().iterator(); it.hasNext();) {
                final PdfImportedPage ip = (PdfImportedPage)it.next();
                this.writer.addToBody(ip.getFormXObject(this.writer.getCompressionLevel()), ip.getIndirectReference());
            }
            writeAllVisited();
        }
        finally {
            try {
                this.reader.close();
                this.file.close();
            }
            catch (final Exception e) {
                //Empty on purpose
            }
        }
    }
}
