package com.aowagie.text.pdf;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;

/** The BoundedBufferedReader class.
 * A BufferedReader that prevents DoS attacks by providing bounds for line length and number of lines.
 * Copyright (c) 2011 - Sean Malone.
 * The BoundedBufferedReader is published by Sean Malone under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * @author Sean Malone (sean@seantmalone.com)
 * @version 1.1 */
public final class BoundedBufferedReader extends BufferedReader {

	private static final int DEFAULT_MAX_LINES = 4096;			//Max lines per file
	private static final int DEFAULT_MAX_LINE_LENGTH = 4096;	//Max bytes per line

	private final int readerMaxLines;
	private final int readerMaxLineLen;
	private int currentLine = 1;

	/** Crea un <i>BufferedReader</i> con l&iacute;mites de lectura.
	 * @param reader Lector subyacente.
	 * @param maxLines N&uacute;mero m&aacute;ximo de l&iacute;neas que permitimos leer.
	 * @param maxLineLen N&uacute;mero m&aacute;ximo de octetos por l&iacute;nea que permitimos leer. */
	public BoundedBufferedReader(final Reader reader, final int maxLines, final int maxLineLen)	{
		super(reader);
		if (maxLines<=0 || maxLineLen<=0) {
			throw new IllegalArgumentException("BoundedBufferedReader - maxLines and maxLineLen must be greater than 0"); //$NON-NLS-1$
		}

		this.readerMaxLines = maxLines;
		this.readerMaxLineLen = maxLineLen;
	}

	/** Crea un <i>BufferedReader</i> con l&iacute;mites de lectura por defecto.
	 * @param reader Lector subyacente. */
	public BoundedBufferedReader(final Reader reader) {
		super(reader);
		this.readerMaxLines = DEFAULT_MAX_LINES;
		this.readerMaxLineLen = DEFAULT_MAX_LINE_LENGTH;
	}

	@Override
	public String readLine() throws IOException {
		//Check readerMaxLines limit
		if (this.currentLine > this.readerMaxLines) {
			throw new IOException("BoundedBufferedReader - Line read limit has been reached."); //$NON-NLS-1$
		}
		this.currentLine++;

		int currentPos=0;
		final char[] data=new char[this.readerMaxLineLen];
		final int CR = 13;
		final int LF = 10;
		int currentCharVal=super.read();

		//Read characters and add them to the data buffer until we hit the end of a line or the end of the file.
		while( currentCharVal!=CR && currentCharVal!=LF && currentCharVal>=0) {
			data[currentPos++]=(char) currentCharVal;
			//Check readerMaxLineLen limit
			if (currentPos<this.readerMaxLineLen) {
				currentCharVal = super.read();
			}
			else {
				break;
			}
		}

		if (currentCharVal<0) {
			//End of file
			if (currentPos>0) {
				//Return last line
				return new String(data,0,currentPos);
			}
			return null;
		}
		//Remove newline characters from the buffer
		if(currentCharVal==CR) {
			//Check for LF and remove from buffer
			super.mark(1);
			if (super.read() != LF) {
				super.reset();
			}
		}
		else if(currentCharVal!=LF) {
			//readerMaxLineLen has been hit, but we still need to remove newline characters.
			super.mark(1);
			final int nextCharVal = super.read();
			if (nextCharVal==CR) {
				super.mark(1);
				if (super.read() != LF) {
					super.reset();
				}
			}
			else if (nextCharVal!=LF) {
				super.reset();
			}
		}
		return new String(data,0,currentPos);
	}
}